/*
                Interactive  Shock Wave  Program

     Program to perform two dimensional analysis of supersonic flow
        past a wedge including oblique and normal shock conditions

                     Version 1.1.0a   - 29 June 00

                         Written by Tom Benson
                       NASA Glenn Research Center

>                              NOTICE
>This software is in the Public Domain.  It may be freely copied and used in
>non-commercial products, assuming proper credit to the author is given.  IT
>MAY NOT BE RESOLD.  If you want to use the software for commercial
>products, contact the author.
>No copyright is claimed in the United States under Title 17, U. S. Code.
>This software is provided "as is" without any warranty of any kind, either
>express, implied, or statutory, including, but not limited to, any warranty
>that the software will conform to specifications, any implied warranties of
>merchantability, fitness for a particular purpose, and freedom from
>infringement, and any warranty that the documentation will conform to the
>program, or any warranty that the software will be error free.
>In no event shall NASA be liable for any damages, including, but not
>limited to direct, indirect, special or consequential damages, arising out
>of, resulting from, or in any way connected with this software, whether or
>not based on warranty, contract, tort or otherwise, whether or not injury
>was sustained by persons or property or otherwise, and whether or not loss
>was sustained from, or arose out of the results of, or use of, the software
>or services provided hereunder.

      New Test : 
                 * change output box colors
      Old Test :
                                                     TJB 29 June 00
*/

import java.awt.*;
import java.lang.Math ;

public class Shock extends java.applet.Applet {

   double gama, mach0, ang1, ang2, sepval ;
   final double convdr = 3.14515926/180.;

   static double angr, delmax, dist, gamma ;
   int prob, nramps, nshocks, nslip ;
              //  flow parameters
   static double[] turning = new double[15] ;
   static double[] mach1   = new double[15] ;
   static double[] mach2   = new double[15] ;
   static double[] prat    = new double[15] ;
   static double[] trat    = new double[15] ;
   static double[] ptrat   = new double[15] ;
   static double[] rhorat  = new double[15] ;
   static double[] defl    = new double[15] ;
   static double[] shkang  = new double[15] ;
   static double[] pspo    = new double[15] ;
   static double[] tsto    = new double[15] ;
   static double[] ptpto   = new double[15] ;
   static double[] rsro    = new double[15] ;
   static boolean[] detach = new boolean[15] ;
              //  wedge geometry
   static double[] ang = new double[3] ;
   static int[] wfamily = new int[3] ;
   static double[] winter = new double[3] ;
   static double[] wxbgn = new double[3] ;
   static double[] wxnd = new double[3] ;
   static double[] wybgn = new double[3] ;
   static double[] wynd = new double[3] ;
   static double[] wslope = new double[3] ;
              // shock geometry
   static double[] sang = new double[15] ;
   static int[] sfamily = new int[15] ;
   static double[] sinter = new double[15] ;
   static double[] sxbgn = new double[15] ;
   static double[] sxnd = new double[15] ;
   static double[] sybgn = new double[15] ;
   static double[] synd = new double[15] ;
   static double[] sslope = new double[15] ;
              // expansion geometry
   static double[] expang = new double[15] ;
   static int[] efamily = new int[15] ;
   static double[] einter = new double[15] ;
   static double[] exbgn = new double[15] ;
   static double[] exnd = new double[15] ;
   static double[] eybgn = new double[15] ;
   static double[] eynd = new double[15] ;
   static double[] eslope = new double[15] ;
              // slip line geometry
   static double[] slinter = new double[7] ;
   static double[] slxbgn = new double[7] ;
   static double[] slxnd = new double[7] ;
   static double[] slybgn = new double[7] ;
   static double[] slynd = new double[7] ;
   static double[] slslope = new double[7] ;

   Viewer view ;
   Num num ;
   Image offscreenImg ;
   Graphics offsGg ;

   public void init() {
     boolean which = false ;
     Shock a = new Shock() ;
 
     offscreenImg = createImage(this.size().width,
                      this.size().height) ;
     offsGg = offscreenImg.getGraphics() ;
 
     setLayout(new GridLayout(1,2,0,0)) ;

     setDefaults () ;

     a.mach0 = 2.0 ;
     a.ang1  = 10.0 ;
     a.gama  = 1.4 ;
     prob = 0 ;
 
     view = new Viewer(this) ;

     num = new Num(this, (float) a.mach0, (float) a.ang1) ;

     add(view) ;
     add(num) ;

     comPute(a.mach0, a.ang1) ;
  }
 
  public Insets insets() {
     return new Insets(10,10,10,10) ;
  }

  public void handleText(Event evt) {
     Double V1,V2,V3,V4,V5 ;
     double v1,v2,v3,v4,v5 ;
     double ang1mn, ang1rng ;
     float fl1 ;
     int i1,i3,i4,i5 ;
 
 // Mach number - range from 1.0 to 5.0
     V1 = Double.valueOf(num.inp.inleft.f1.getText()) ;
     v1 = V1.doubleValue() ;
     if(v1 < 1.0) {
        v1 = 1.02 ;
        fl1 = (float) v1 ;
        num.inp.inleft.f1.setText(String.valueOf(fl1)) ;
     }
     if(v1 > 5.0) {
        v1 = 5.0 ;
        fl1 = (float) v1 ;
        num.inp.inleft.f1.setText(String.valueOf(fl1)) ;
     }
 // ramp angle # 1  range from 0 to +30 for prob = 0
     V3 = Double.valueOf(num.inp.inleft.f3.getText()) ;
     v3 = V3.doubleValue() ;
     ang1mn = 5.0 ;
     ang1rng = 25.0 ;
     if(v3 < ang1mn) {
       v3 = ang1mn ;
       fl1 = (float) v3 ;
       num.inp.inleft.f3.setText(String.valueOf(fl1)) ;
     }
     if(v3 > 30.0) {
        v3 =  30.0 ;
        fl1 = (float) v3 ;
        num.inp.inleft.f3.setText(String.valueOf(fl1)) ;
     }

     i1 = (int) (((v1 - 1.0)/4.0)*1000.) ;
     i3 = (int) (((v3 - ang1mn)/ang1rng)*1000.) ;

     num.inp.inright.s1.setValue(i1) ;
     num.inp.inright.s3.setValue(i3) ;

     comPute(v1, v3) ;
  }

  public void handleBar(Event evt) {
     int i1, i3, i4, i5 ;
     Double V2 ;
     double v1, v2, v3, v4, v5 ;
     double ang1mn, ang1rng ;
     float fl1, fl3, fl4, fl5 ;

     ang1mn = 5.0 ;
     ang1rng = 25.0 ;

     i1 = num.inp.inright.s1.getValue() ;
     i3 = num.inp.inright.s3.getValue() ;

     v1 = i1 * 4.0 / 1000. + 1.0 ;
     v3 = i3 * ang1rng / 1000. + ang1mn ;
 
     fl1 = (float) v1 ;
     fl3 = (float) v3 ;

     num.inp.inleft.f1.setText(String.valueOf(fl1)) ;
     num.inp.inleft.f3.setText(String.valueOf(fl3)) ;

     comPute(v1, v3) ;    
  }

   public void setDefaults() {
     nslip = 0 ;
     nramps = 1 ;
     nshocks = 1 ;

     dist = 100. ; 
     ang[0]  = 0.0 ;
     prat[0] = 1.0 ;
     trat[0] = 1.0 ;
     ptrat[0] = 1.0 ;
     rhorat[0] = 1.0 ;
     pspo[0] = 1.0 ;
     tsto[0] = 1.0 ;
     rsro[0] = 1.0 ;
     ptpto[0]= 1.0 ;
     defl[0] = 0.0 ;
     turning[0] = 0.0 ;
     shkang[0] = 0.0 ;
   }

   public void comPute(double machin, double del1) {

       mach2[0] = mach1[0] = machin;
       gamma  = 1.4 ;
       ang[1] = del1 ;

       getGeom () ;
 
       loadZero() ;
 
       anlSing() ;
 
       loadOut() ;
 
       view.repaint();
   }

   public void loadZero() {
       int i ;

       nslip = 0 ; 
       for(i=1; i<=14; ++i) {
            pspo[i] = 0.0 ;
            tsto[i] = 0.0 ;
            rsro[i] = 0.0 ;
            ptpto[i] = 0.0 ;
            prat[i] = 0.0 ;
            trat[i] = 0.0 ;
            rhorat[i] = 0.0 ;
            ptrat[i] = 0.0 ;
            turning[i] = 0.0 ;
            defl[i] = 0.0 ;
            shkang[i] = 0.0 ;
            mach2[i] = 0.0 ;
            detach[i] = false ;
            getGeomOblq(300.0,300.0,1,i) ;
       }
   }

   public void anlSing() {     //  Analysis for Single  Wedge.

       mach1[1] = mach2[0] ;
       nshocks = 1 ;
       nramps = 1 ;
       angr  = ang[1] * convdr ;
       detach[1] = false ;

       getAnglim(mach1[1],gamma) ;

       if (ang[1] > delmax) {
         getNorm(mach1[1],gamma,0,1) ;
         getGeomNorm(wxbgn[1],wybgn[1],290.,wfamily[1],1) ;
         turning[1] = ang[1] ;
       }
       else {
         shkang[1]  = getShkang(mach1[1],angr,gamma) ;
         getOblq(mach1[1],shkang[1],gamma,0,1) ;
         turning[1] = turning[0] + defl[1] ;
         sang[1] = turning[0] + shkang[1] ;
         getGeomOblq(wxbgn[1],wybgn[1],wfamily[1],1) ;
       }
       return ;
   }

   public void getGeom () {
          // wedge geometry
       int i ;
         wfamily[1] = 1 ;
         wslope[1]  = Math.tan(convdr * ang[1]) ;
         wxbgn[1]   = 50. ;
         wybgn[1]   = 0.0 ;
         winter[1]  = wybgn[1] - wslope[1] * wxbgn[1] ;

         wfamily[2] = 1 ;
         wslope[2]  = Math.tan(convdr * (ang[1] + ang[2])) ;
         wxbgn[2]   = wxbgn[1] + dist ;
         wybgn[2]   = wxbgn[2] * wslope[1] + winter[1];
         winter[2]  = wybgn[2] - wslope[2] * wxbgn[2] ;
         wxnd[2]    = 290. ;
         wynd[2]    = wxnd[2] * wslope[2] + winter[2] ;

         wxnd[1]    = wxbgn[2] ;
         wynd[1]    = wybgn[2] ;
 
       return;
   }

   public void getGeomOblq (double xi, double yi, int fam, int index) {
          // oblique shock geometry
       sfamily[index] = fam ;
       sslope[index]  = fam*Math.tan(convdr * sang[index]) ;
       sxbgn[index]   = xi ;
       sybgn[index]   = yi ;
       sinter[index]  = yi - sslope[index] * xi ;
       sxnd[index]    = 290. ;
       synd[index]    = sxnd[index]*sslope[index] + sinter[index] ;
       return;
   }

   public void getGeomNorm (double xi, double yi, double ye,
                             int fam, int index) {
          // normal shock geometry
       sfamily[index] = fam ;
       sslope[index]  = 0.0 ;
       sxbgn[index]   = xi ;
       sybgn[index]   = yi ;
       sinter[index]  = yi ;
       sxnd[index]    = xi ;
       synd[index]    = ye ;
       return;
   }

   public void loadOut() {
      int outzn ;

      outzn = 1;

      num.out.o1.setText(String.valueOf(filter3(mach2[outzn]))) ;
      num.out.o3.setText(String.valueOf(filter3(shkang[outzn])));
      num.out.o2.setText(String.valueOf(filter3(prat[outzn]))) ;
      num.out.o4.setText(String.valueOf(filter3(ptrat[outzn]))) ;
      num.out.o6.setText(String.valueOf(filter3(trat[outzn]))) ;
      num.out.o7.setText(String.valueOf(filter3(rhorat[outzn])));
  }

   public float filter3(double inumbr) {
     //  output only to .001
       float number ;
       int intermed ;

       intermed = (int) (inumbr * 1000.) ;
       number = (float) (intermed / 1000. );
       return number ;
  }

   public void getAnglim (double machin, double gam) {
       double a1, ab1, ac1, sints, msq, mfor, thmx, cotd ;

       msq = machin * machin ;
       mfor = msq * msq ;

       a1 = 2.0 * gam * mfor ;
       ab1 = 4.0 * msq - (gam + 1.0) * mfor ;
       ac1 = -(2.0 + (gam + 1.0) * msq) ;
       sints = (-ab1 + Math.sqrt(Math.pow(ab1,2.0)-4.0*a1*ac1))/(2.0*a1) ;
       thmx = Math.asin(Math.sqrt(sints)) ;

       cotd = Math.tan(thmx)*(((gam+1.0)*msq)/    
                (2.0*(msq * sints - 1.0))-1.0);
       delmax = (Math.atan(1.0/cotd))/convdr ;
 
       return;
   }

   public double getShkang (double machin, double delr, double gam) {
        // Iterate to get Shock Angle given Wedge Angle.
      double cotd,mst,gp1,number ;    
      double theto,thetn,delo,deln,deriv ;

      gp1 = gam + 1.0 ;
      theto = Math.asin(1.0/machin) ; 
      delo = 0.0 ;
      thetn = theto + 3.0 * convdr ;
      while (Math.abs(delr - delo) > .0001) {
         mst = machin * Math.sin(thetn) ;
         cotd = Math.tan(thetn)*((gp1*machin*machin)/    
                (2.0*(mst * mst - 1.0))-1.0);
         deln = Math.atan(1.0/cotd) ;
         deriv = (deln-delo)/(thetn-theto) ;
         delo = deln ;
         theto = thetn ;
         thetn = theto + (delr-delo)/deriv ;
      }

      number = theto / convdr ;
      return number;
   }

   public void getOblq (double machin, double shang, double gam,
                        int upstrm, int index) {
          // NACA 1135 - oblique shock relations.
       double mst, gm1, gp1, msq, m2sq, cotd ;

       mst = machin * Math.sin(shang*convdr) ;
       msq = machin * machin ;
       gm1 = gam - 1.0 ;
       gp1 = gam + 1.0 ;

       prat[index] = (2.0*gam*mst*mst - gm1)/gp1 ;
       rhorat[index] = (gp1*mst*mst)/(gm1*mst*mst + 2.0) ;
       trat[index] = (2.0*gam*mst*mst - gm1) * (gm1*mst*mst + 2.0)
                  /(mst*mst*Math.pow(gp1,2.0)) ; 
       ptrat[index] = (Math.pow(((gp1*mst*mst)/(gm1*mst*mst+2.0)),(gam/gm1)))
               * Math.pow((gp1/(2.0*gam*mst*mst - gm1)),(1.0/gm1)) ;
       m2sq = ((msq * mst * mst * Math.pow(gp1,2.0)) +
              (-4.0 * (mst*mst  - 1.0) * (gam*mst*mst + 1.0))) /
              ((2.0*gam*mst*mst - gm1) * (gm1*mst*mst + 2.0)) ;
       mach2[index] = Math.sqrt(m2sq) ;
       cotd = Math.tan(shang*convdr)*((gp1*msq)/    
                (2.0*(mst * mst - 1.0))-1.0);
       defl[index] = (Math.atan(1.0/cotd))/convdr ;
       mach1[index] = machin ;

       pspo[index] = pspo[upstrm]*prat[index] ;
       tsto[index] = tsto[upstrm]*trat[index] ;
       rsro[index] = rsro[upstrm]*rhorat[index] ;
       ptpto[index] = ptpto[upstrm]*ptrat[index] ;
  
       return;
   }

   public void getNorm (double machin, double gam, 
                         int upstrm, int index) {
          // NACA 1135 - normal shock relations.
       double gm1, gp1, msq, m2sq ;

       msq = machin * machin ;
       gm1 = gam - 1.0 ;
       gp1 = gam + 1.0 ;

       prat[index] = (2.0*gam*msq - gm1)/gp1 ;
       rhorat[index] = (gp1*msq)/(gm1*msq + 2.0) ;
       trat[index] = prat[index] / rhorat[index] ;
       ptrat[index] = (Math.pow(rhorat[index],(gam/gm1)))
               * (Math.pow((1.0/prat[index]),(1.0/gm1))) ;
       m2sq = msq / (prat[index] * rhorat[index]) ;
       mach2[index] = Math.sqrt(m2sq) ;
       defl[index] = 0.0 ;
       mach1[index] = machin ;
       shkang[index] = 90.0 ;
       sang[index] = 90.0 ;
       detach[index] = true ;

       pspo[index] = pspo[upstrm]*prat[index] ;
       tsto[index] = tsto[upstrm]*trat[index] ;
       rsro[index] = rsro[upstrm]*rhorat[index] ;
       ptpto[index] = ptpto[upstrm]*ptrat[index] ;
 
       return;
   }

   class Num extends Panel {
     Shock outerparent ;
     Inp inp ;
     Out out ;

     Num (Shock target, float v1, float v3) {                           
          outerparent = target ;
          setLayout(new GridLayout(2,1,10,10)) ;
 
          inp = new Inp(outerparent, v1, v3) ;  
          out = new Out(outerparent) ;
 
          add(inp) ;
          add(out) ;
     }

     class Inp extends Panel {
       Shock outerparent ;
       Inright inright ;
       Inleft inleft ;

       Inp (Shock target, float v1, float v3) {
                             
          outerparent = target ;
          setLayout(new GridLayout(1,2,10,10)) ;

          inleft = new Inleft(outerparent, v1, v3) ; 
          inright = new Inright(outerparent, v1, v3) ;
 
          add(inleft) ;
          add(inright) ;
       }
 
       class Inright extends Panel {
        Shock outerparent ;
        Scrollbar s1, s3 ;
        Label l1 ;

        Inright (Shock target,float v1, float v3) {
            int i1, i3 ;
 
            outerparent = target ;
            setLayout(new GridLayout(5,1,10,10)) ;

            i1 = (int) ((((double)v1 - 1.0)/4.0)*1000.) ;
            i3 = (int) ((((double)v3 - 5.0)/30.0)*1000.) ;
 
            s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
            s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);

            l1 = new Label("Input", Label.LEFT) ;
            l1.setForeground(Color.blue) ;

            add(l1) ; 
            add(s1) ;
            add(s3) ;
            add(new Label(" ", Label.CENTER)) ;
            add(new Label(" ", Label.CENTER)) ;
          }

          public boolean handleEvent(Event evt) {
            if(evt.id == Event.SCROLL_ABSOLUTE) {
               this.outerparent.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_DOWN) {
               this.outerparent.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_UP) {
               this.outerparent.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_DOWN) {
               this.outerparent.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_UP) {
               this.outerparent.handleBar(evt) ;
               return true ;
            }
            else return false ;
          }
        }

        class Inleft extends Panel {
          Shock outerparent ;
          TextField f1, f3;
          Label l1,l2 ;

          Inleft (Shock target, float v1, float v3 ) {
            
            outerparent = target ;
            setLayout(new GridLayout(5,2,2,10)) ;
  
            f1 = new TextField(String.valueOf(v1),5) ;
            f3 = new TextField(String.valueOf(v3),5) ;

            l1 = new Label("Up", Label.RIGHT) ;
            l1.setForeground(Color.blue) ;
            l2 = new Label("stream", Label.LEFT) ;
            l2.setForeground(Color.blue) ;

            add(l1) ;
            add(l2) ;
            add(new Label("Mach", Label.CENTER)) ;  
            add(f1) ;
            add(new Label(" Angle", Label.CENTER)) ;
            add(f3) ;
            add(new Label(" ", Label.CENTER)) ;
            add(new Label(" ", Label.CENTER)) ;
            add(new Label(" ", Label.CENTER)) ;
            add(new Label(" ", Label.CENTER)) ;

          }
  
          public boolean handleEvent(Event evt) {
            if(evt.id == Event.ACTION_EVENT) {
               this.outerparent.handleText(evt) ;
               return true ;
            }
            else return false ;
          }
        }
     }

     class Out extends Panel {
       Shock outerparent ;
       TextField o1, o2, o3, o4, o6, o7 ;
       Label lo1,lo2,lo3,lo4 ;

       Out (Shock target) {

          outerparent = target ;
          setLayout(new GridLayout(5,4,2,10)) ;
 
          o1 = new TextField() ;
          o1.setBackground(Color.black) ;
          o1.setForeground(Color.green) ;
          o2 = new TextField() ;
          o2.setBackground(Color.black) ;
          o2.setForeground(Color.green) ;
          o3 = new TextField() ;
          o3.setBackground(Color.black) ;
          o3.setForeground(Color.green) ;
          o4 = new TextField() ;
          o4.setBackground(Color.black) ;
          o4.setForeground(Color.green) ;
          o6 = new TextField() ;
          o6.setBackground(Color.black) ;
          o6.setForeground(Color.green) ;
          o7 = new TextField() ;
          o7.setBackground(Color.black) ;
          o7.setForeground(Color.green) ;
 
          lo1 = new Label("Down", Label.RIGHT) ;
          lo1.setForeground(Color.red) ;
          lo2 = new Label("stream", Label.LEFT) ;
          lo2.setForeground(Color.red) ;
          lo3 = new Label("Output ", Label.CENTER) ;
          lo3.setForeground(Color.red) ;
          lo4 = new Label("Zone 1", Label.CENTER) ;
          lo4.setForeground(Color.red) ;

          add(lo1) ;
          add(lo2) ;
          add(lo3) ;
          add(lo4) ;

          add(new Label("Mach", Label.CENTER)) ;  
          add(o1) ;  
          add(new Label("p ratio", Label.CENTER)) ;  
          add(o2) ;

          add(new Label("Shock Angle", Label.CENTER)) ; 
          add(o3) ; 
          add(new Label("pt ratio", Label.CENTER)) ;  
          add(o4) ; 

          add(new Label(" ", Label.CENTER)) ;
          add(new Label(" ", Label.CENTER)) ;
          add(new Label("T ratio", Label.CENTER)) ; 
          add(o6) ;

          add(new Label(" ", Label.CENTER)) ;
          add(new Label(" ", Label.CENTER)) ;
          add(new Label("r ratio", Label.CENTER)) ;  
          add(o7) ; 
       }
 
     } 
  }

  class Viewer extends Canvas {
     Shock outerparent ;

     Viewer (Shock target) {
         setBackground(Color.white) ;
     }
 
     public void update(Graphics g) {
         view.paint(g) ;
     }

     public void paint(Graphics g) {
       int i,k ;
       int exes[] = new int[3] ;
       int whys[] = new int[3] ;
       int yorgn = 150 ;
       int xorgn = 50 ;
       int xlong = 290 ;

       offsGg.setColor(Color.white) ;
       offsGg.fillRect(0,0,300,300) ;
       offsGg.setColor(Color.blue) ;
       offsGg.drawString("Upstream", 0, 20) ;
       offsGg.drawString("Flow", xorgn-40, 35) ;
       offsGg.setColor(Color.black) ;
       offsGg.fillRect(xorgn-40,42,30,9) ;
       exes[0] = xorgn ;
       whys[0] = 46;
       exes[1] = xorgn - 10;
       whys[1] = 56;
       exes[2] = xorgn - 10;
       whys[2] = 36;
       Polygon poly1 = new Polygon(exes,whys,3) ;
       offsGg.fillPolygon(poly1) ;
          // draw geometry
       offsGg.setColor(Color.red) ;
          // draw ramps
       for (i = 1; i <= nramps; ++i) {
         exes[0] = (int) wxbgn[i] ;
         whys[0] = yorgn - (int) (wxbgn[i]*wslope[i] + winter[i]) ;
         exes[1] = xlong ;
         whys[1] = yorgn - (int) (290.*wslope[i] + winter[i]) ;
         exes[2] = xlong ;
         whys[2] = yorgn + (int) (290.*wslope[i] + winter[i]) ;
         offsGg.setColor(Color.red) ;
         Polygon poly = new Polygon(exes,whys,3) ;
         offsGg.fillPolygon(poly) ;
         offsGg.drawString("1", exes[1]-20, whys[1]-10) ;
         offsGg.drawString("1", exes[1]-20, whys[2]+10) ;
         exes[0] = (int) wxbgn[i] ;
         whys[0] = yorgn - (int) (wxbgn[i]*wslope[i] + winter[i]) ;
         exes[1] = xlong ;
         whys[1] = whys[0] ;
         offsGg.setColor(Color.white) ;
         offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
         offsGg.drawString("Angle", exes[1]-40, whys[0]-5) ;
       }
          // draw streamlines
       for (i = 1; i <= 4; ++i) {
         exes[0] = 0 ;
         whys[0] = yorgn - (i-1)*20 ;
         whys[1] = yorgn - (i-1)*20 ;
         if (detach[1]) {
            exes[1] = xorgn ;
            exes[2] = xlong ;
            whys[2] = yorgn - (int) (290.*wslope[1] + winter[1]) -(i-1)*20 ;
         }
         else {
            exes[1] = (int) (((i-1)*20 - sinter[1])/sslope[1]) ;
            exes[2] = xlong ;
            whys[2] = yorgn - (int) (wslope[1]*(exes[2]-exes[1])) -(i-1)*20 ;
         }
         offsGg.setColor(Color.black) ; 
         offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
         offsGg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
         exes[0] = 10+i*5 ;
         exes[1] = exes[0]-5;
         whys[1] = whys[0]-5;
         exes[2] = exes[1];
         whys[2] = whys[0]+5;
         offsGg.fillPolygon(exes,whys,3) ;
       }
       for (i = 1; i <= 4; ++i) {
         exes[0] = 0 ;
         whys[0] = yorgn + (i-1)*20 ;
         whys[1] = yorgn + (i-1)*20 ;
         if (detach[1]) {
            exes[1] = xorgn ;
            exes[2] = xlong ;
            whys[2] = yorgn + (int) (290.*wslope[1] + winter[1]) +(i-1)*20 ;
         }
         else {
            exes[1] = (int) (((i-1)*20 - sinter[1])/sslope[1]) ;
            exes[2] = xlong ;
            whys[2] = yorgn + (int) (wslope[1]*(exes[2]-exes[1])) +(i-1)*20 ;
         }
         offsGg.setColor(Color.black) ; 
         offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
         offsGg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
         exes[0] = 10+i*5 ;
         exes[1] = exes[0]-5;
         whys[1] = whys[0]-5;
         exes[2] = exes[1];
         whys[2] = whys[0]+5;
         offsGg.fillPolygon(exes,whys,3) ;
       }
          // draw shock waves or expansion
       for(i=1; i <= nshocks; ++i) {
         exes[0] = (int) sxbgn[i] ;
         whys[0] = yorgn - (int) sybgn[i] ;
         exes[1] = (int) sxnd[i] ;
         whys[1] = yorgn - (int) synd[i] ;
         if (detach[i]) {
           offsGg.setColor(Color.magenta) ; 
           offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
         }
         else {
           offsGg.setColor(Color.blue) ; 
           offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;  
         }
         whys[1] = yorgn + (int) synd[i] ;
         if (detach[i]) {
           offsGg.setColor(Color.magenta) ; 
           offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
         }
         else {
           offsGg.setColor(Color.blue) ; 
           offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;  
         }
       }
       g.drawImage(offscreenImg,0,0,this) ;
    }
  }
}
