/*
                        TunnelSim - Open
                 Interactive Wind Tunnel Program

     Program performs one dimensional design and ideal analysis of
                      open return wind tunnel 
             Derived from nozzle java program and vuduct

                     Version 1.0g   - 15 July 09

                             Written by 
                              Tom Benson
                       NASA Glenn Research Center
                                 and
                              Anthony Vila
                        Vanderbilt  University

>                              NOTICE
>This software is in the Public Domain.  It may be freely copied and used in
>non-commercial products, assuming proper credit to the author is given.  IT
>MAY NOT BE RESOLD.  If you want to use the software for commercial
>products, contact the author.
>No copyright is claimed in the United States under Title 17, U. S. Code.
>This software is provided "as is" without any warranty of any kind, either
>express, implied, or statutory, including, but not limited to, any warranty
>that the software will conform to specifications, any implied warranties of
>merchantability, fitness for a particular purpose, and freedom from
>infringement, and any warranty that the documentation will conform to the
>program, or any warranty that the software will be error free.
>In no event shall NASA be liable for any damages, including, but not
>limited to direct, indirect, special or consequential damages, arising out
>of, resulting from, or in any way connected with this software, whether or
>not based on warranty, contract, tort or otherwise, whether or not injury
>was sustained by persons or property or otherwise, and whether or not loss
>was sustained from, or arose out of the results of, or use of, the software
>or services provided hereunder.

      New Test : 
                 * Begin layout
                 * Enable sliders/io boxes
                 * modify input panels
                 * set up plots and graphics
                 * bring in metric units
                 * add action listeners
                   add blinker logic from Anthony
      Old Test

                                                     TJB 15 July 09
*/

import java.awt.*;
import java.awt.event.* ;
import java.lang.Math ;

public class Tuno extends java.applet.Applet {

   final double pi = 3.14515926;
   final double convdr = 3.14515926/180.;
   final double runiv = 1545. ;

   static double bwmn, bwmx, bhmn, bhmx, blmn, blmx ;
   static double bpmn, bpmx, bvmn, bvmx, bamn, bamx ;
   static double bsmn, bsmx ;
   static double twmn, twmx, thmn, thmx, tlmn, tlmx ;
   static double tpmn, tpmx, tvmn, tvmx, tamn, tamx ;
   static double dwmn, dwmx, dhmn, dhmx, dlmn, dlmx ;
   static double dpmn, dpmx, dvmn, dvmx, damn, damx ;
   static double wdth[] = new double[10] ;
   static double hite[] = new double[10] ;
   static double length[] = new double[10] ;
   static double slopeh[] = new double[10] ;
   static double slopew[] = new double[10] ;
   static double area[] = new double[10] ;
   static double vel[] = new double[10] ;
   static double pres[] = new double[10] ;
   static double mach[] = new double[10] ;
   static double air[] = new double[10] ;
   static int xsec[]  = new int[10] ;
   static int shapew[] = new int[10] ;
   static int shapeh[] = new int[10] ;
   static int nsecs,nzone,ivflag,gflag ;

   static double xwu[]  = new double[61] ;
   static double ywu[]  = new double[61] ;
   static double xwl[]  = new double[61] ;
   static double ywl[]  = new double[61] ;
   static double xhu[]  = new double[61] ;
   static double yhu[]  = new double[61] ;
   static double xhl[]  = new double[61] ;
   static double yhl[]  = new double[61] ;

   static int antim,ancol,lunits ; 
   static int antg,ancg,blsep,compr,ovflag ; 
   static double lconv,pconv,mconv ;
   static double rho,mu,pspri,rey,airout,difa,cona,x2,x3,yw2,yw3,yh2,yh3 ;

   int xtv,ytv,sldloc ;
   static double factp,factv, yplo ;
   static double ypl[]  = new double[10] ;
   static double oarea[] = new double[10] ;
   static double ovel[] = new double[10] ;
   static double opres[] = new double[10] ;
   static double omach[] = new double[10] ;

   Image offscreenImg ;
   Image offscreen2Img ;
   Graphics offsGg ;
   Graphics offs2Gg ;
   Pan pan ;
   Pic pic ;
   CardLayout laying, layinf ;

   Blinker1 blink1;
   Blinker2 blink2;

   public void init() {
     Tuno a = new Tuno() ;

     offscreenImg = createImage(this.size().width,
                      this.size().height) ;
     offsGg = offscreenImg.getGraphics() ;
     offscreen2Img = createImage(this.size().width,
                      this.size().height) ;
     offs2Gg = offscreen2Img.getGraphics() ;

     setLayout(new GridLayout(1,2,5,5)) ;

     setDefaults () ;

     blink1 = new Blinker1(this);    //sets up blinker thread
     blink2 = new Blinker2(this);

     pic = new Pic(this) ;
     pan = new Pan(this) ;

     add(pic) ;
     add(pan) ;

     comPute() ;
     pic.view.start() ;
     pic.grph.start() ;
     blink1.start();
     blink2.start();
  }

  public Insets insets() {
     return new Insets(10,10,10,10) ;
  }

   public void setDefaults() {

       lunits = 0 ;
       lconv = 1.;                         /*  feet    */
       pconv = 1.0  ;                     /* lb/sq in */
       mconv = 1.0 ;                   /* airflow rate lbs/sec*/
       rho = .0023769 ;
       mu = .0001567 ;

       nsecs = 3;
       nzone = 3;
       length[0] = 1.0 ;
       wdth[0] = 1.0 ;
       hite[0] = 1.0 ;
       gflag = 1;
       ovflag = 1 ;
                          // bellmouth geometry
       bwmn = .10 ;
       bwmx = 10.0 ;
       bhmn = .10 ;
       bhmx = 10.0 ;
       bsmn = -5.0 ;
       bsmx = 0.0 ;
       blmn = .10 ;
       blmx = 20.0 ;
       wdth[1]  = 2.0 ;
       hite[1] = 1.0 ;
       length[1] = 2.0 ;
       slopew[1] = -1.0 ;
       slopeh[1] = 0.0 ;
       xsec[1] = 1 ;
       shapew[1] = 2 ;
       shapeh[1] = 1 ;
                          // bellmouth flow
       bpmn = 13.0 ;
       bpmx = 14.7 ;
       bvmn = 0.0 ;
       bvmx = 350.0 ;
       bamn = 0.0 ;
       bamx = 5000.0 ;
       vel[1] = 42.0 ;
       pres[1] = 14.686 ;
       air[1] = 84. ;
                          // test section geometry
       twmn = .10 ;
       twmx = 10.0 ;
       thmn = .10 ;
       thmx = 10.0 ;
       tlmn = .10 ;
       tlmx = 20.0 ;
       wdth[2]  = 1.0 ;
       hite[2] = 1.0 ;
       length[2] = 1.0 ;
       slopew[2] = 0.0 ;
       slopeh[2] = 0.0 ;
       xsec[2] = 2 ;
       shapew[2] = 1 ;
       shapeh[2] = 1 ;
                          // test section flow
       tpmn = 13.0 ;
       tpmx = 14.7 ;
       tvmn = 0.0 ;
       tvmx = 350.0 ;
       tamn = 0.0 ;
       tamx = 5000.0 ;
       vel[2] = 83.0 ;
       pres[2] = 14.643 ;
       air[2] = 84. ;
                          // diffuser geometry
       dwmn = .10 ;
       dwmx = 10.0 ;
       dhmn = .10 ;
       dhmx = 10.0 ;
       dlmn = .10 ;
       dlmx = 20.0 ;
       wdth[3]  = 2.0 ;
       hite[3] = 2.0 ;
       length[3] = 5.0 ;
       slopew[3] = 0.0 ;
       slopeh[3] = 0.0 ;
       xsec[3] = 3 ;
       shapew[3] = 1 ;
       shapeh[3] = 1 ;
                          // diffuser flow
       dpmn = 13.0 ;
       dpmx = 14.7 ;
       dvmn = 0.0 ;
       dvmx = 350.0 ;
       damn = 0.0 ;
       damx = 5000.0 ;
       vel[3] = 27.0 ;
       pres[3] = 14.694 ;
       air[3] = 84. ;
   // graphics
       factv = 30.0 ;
       xtv = 55 ;
       ytv = 80 ;
       sldloc = 50 ;
   }

   public void comPute() {

       blsep = 0 ;
       compr = 0 ;
 
       getGeom () ;
       getFstream () ;

       if (cona >= 7.0) blsep = 1 ;
       anlys ();                      // perform  analysis 
       if (mach[2] > .3) compr = 1 ;

       getPlot ();  //load up the plot variable

       loadOut() ;
 
       pic.view.repaint();
       pic.grph.repaint();
   }

   public void getFstream () {        /* get the freestream conditions */
       int index ;

       switch (ivflag) {
          case 1:  {            /*   pressure specified */
              vel[gflag] = getVel(pres[gflag]) ;
              vel[1] = vel[gflag] * area[gflag] / area[1] ;
              break ;
          }
          case 2:  {          /* velocity specified */
              vel[1] = vel[gflag] * area[gflag] / area[1] ;
              break ;
          }
          case 3:  {            /*  airflow specified */
              vel[1] = air[gflag] / area[1] ;
              break ;
          }
       }
       return ;
   }

   public void getGeom ()  {       //  define the geometry  
       double aw,bw,cw,dw,del,lng ;
       double ah,bh,ch,dh,wint,hint ;
       int i ;

       aw = 0.0 ;
       bw = 0.0 ;
       cw = 0.0 ;
       ah = 0.0 ;
       bh = 0.0 ;
       ch = 0.0 ;
                                    // calculate zonal areas 
       for (i=1; i<=nsecs; ++i) {
         switch (xsec[i]) {
           case 1: area[i] = wdth[i] * hite[i] ;
                   break;
           case 2: area[i] = wdth[i] * wdth[i] ;
                   break;
           case 3: area[i] = pi * wdth[i] * wdth[i] / 4.0 ; 
                   break;
         }
       }

       difa = (area[3] - area[2])/(length[3]) ;
       cona = (wdth[3] - wdth[2])/(2.0*length[3]) / convdr ;
                        /* define geometric constants for bellmouth */
       dw = wdth[1]/2.0 ;
       switch (shapew[1]) {   // width curve fit
           case 1: {               /* linear */
               cw = (wdth[2] - wdth[1]) / (2.0 * length[1]) ;
               bw = 0.0 ;
               aw = 0.0 ;
               break ;
           }
           case 2:  {              /* quadratic */
               cw = (wdth[2] - wdth[1]) / length[1] ;
               bw = - cw / (2.0 * length[1]) ;
               aw = 0.0 ;
               break ;
           }
           case 3:  {             /* cubic */
               cw = slopew[1] ;
               bw = 3.0*(wdth[2] - wdth[1])/
                  (2.0*length[1]*length[1])-2.0*slopew[1]/length[1] ;
               aw = - (2.0 * bw * length[1] + slopew[1]) /
                  (3.0*length[1]*length[1]) ;
              break ;
           }
       }

       dh = hite[1]/2.0 ;
       switch (shapeh[1]) {    // height curve fit
           case 1: {               /* linear */
               ch = (hite[2] - hite[1]) / (2.0 * length[1]) ;
               bh = 0.0 ;
               ah = 0.0 ;
               break ;
           }
           case 2:  {              /* quadratic */
               ch = (hite[2] - hite[1]) / length[1] ;
               bh = - ch / (2.0 * length[1]) ;
               ah = 0.0 ;
               break ;
           }
           case 3:  {             /* cubic */
               ch = slopeh[1] ;
               bh = 3.0*(hite[2] - hite[1])/
                  (2.0*length[1]*length[1])-2.0*slopeh[1]/length[1] ;
               ah = - (2.0 * bh * length[1] + slopeh[1]) /
                  (3.0*length[1]*length[1]) ;
              break ;
           }
       }
                        /* calculate 45 points along the bellmouth */
       del = length[1] / 45.0 ;
       for (i=1; i<=45; ++i) {   // width
         xwu[i] = (i-1)*del ;
         ywu[i] =  aw * xwu[i] * xwu[i] * xwu[i]
                +  bw * xwu[i] * xwu[i]
                +  cw * xwu[i] + dw ;
       }
                                            /* load up test section */
       xwu[46] = length[1] ;
       ywu[46] = wdth[2] / 2.0 ;
       xwu[47] = length[1] + length[2] ;
       ywu[47] = wdth[2] / 2.0 ;
                                            /* load up diffuser */
       xwu[48] = length[1] + length[2] + length[3]/3.0 ;
       ywu[48] = (wdth[2] + (wdth[3] - wdth[2])/3.0)/2.0 ;
       xwu[49] = length[1] + length[2] + length[3]*2.0/3.0 ;
       ywu[49] = (wdth[2] + 2.0 * (wdth[3] - wdth[2])/3.0)/2.0 ;
       xwu[50] = length[1] + length[2] + length[3] ;
       ywu[50] = wdth[3] / 2.0 ;
                                    /* calculate lower surface */
       for (i=1; i<= 50; ++i) {
         xwl[i] = xwu[i] ;
         ywl[i] = - ywu[i] ;
       }

       for (i=1; i<=45; ++i) {   // height
         xhu[i] = (i-1)*del ;
         yhu[i] =  ah * xhu[i] * xhu[i] * xhu[i]
                +  bh * xhu[i] * xhu[i]
                +  ch * xhu[i] + dh ;
       }
                                            /* load up test section */
       xhu[46] = length[1] ;
       yhu[46] = hite[2] / 2.0 ;
       xhu[47] = length[1] + length[2] ;
       yhu[47] = hite[2] / 2.0 ;
                                            /* load up diffuser */
       xhu[48] = length[1] + length[2] + length[3]/3.0 ;
       yhu[48] = (hite[2] + (hite[3] - hite[2])/3.0)/2.0 ;
       xhu[49] = length[1] + length[2] + length[3]*2.0/3.0 ;
       yhu[49] = (hite[2] + 2.0 * (hite[3] - hite[2])/3.0)/2.0 ;
       xhu[50] = length[1] + length[2] + length[3] ;
       yhu[50] = hite[3] / 2.0 ;
                                    /* calculate lower surface */
       for (i=1; i<= 50; ++i) {
         xhl[i] = xhu[i] ;
         yhl[i] = - yhu[i] ;
       }

       oarea[1] = area[1] ;
       x2 = length[1] / 3.0 ;
       wint = aw * x2 * x2 * x2
           +  bw * x2 * x2
           +  cw * x2 + dw ;
       yw2 = -wint ;
       hint = ah * x2 * x2 * x2
           +  bh * x2 * x2
           +  ch * x2 + dh ;
       yh2 = hint ;
       oarea[2] = 4.0 * wint * hint ;
       x3 = length[1] * 2.0 / 3.0 ;
       wint = aw * x3 * x3 * x3
           +  bw * x3 * x3
           +  cw * x3 + dw ;
       yw3 = -wint ;
       hint = ah * x3 * x3 * x3
           +  bh * x3 * x3
           +  ch * x3 + dh ;
       yh3 = hint ;
       oarea[3] = 4.0 * wint * hint ;
       oarea[4] = area[2] ;
       oarea[5] = area[2] ;
       oarea[6] = area[2] +difa*(length[3])/3.0 ;
       oarea[7] = area[2] +difa*(length[3])*2.0/3.0 ;
       oarea[8] = area[3] ;

       return ;
   }

    public void anlys () {
  // calculate velocity, pressure, and mach given free stream velocity 
       double vprtc,v1,v2 ;
       float fl1, fl2 ;
       int i,i1,i2 ;

       for (i=1; i<=nsecs; ++i) {
           vel[i]  = vel[1] * area[1] / area[i] ;
           pres[i] = getPres(vel[i]) ;
           mach[i] = vel[i] / 1116. ;
           air[i]  = vel[i] * area[i] ;
       }

       for (i=1; i<=8; ++i) {
           vprtc = vel[1] * area[1] / oarea[i] ;
           if (vprtc > 350.) vprtc = 350. ;
           ovel[i]  = vprtc ;
           opres[i] = getPres(ovel[i]) ;
           omach[i] = ovel[i] / 1116. ;
       }

       rey = vel[2] * wdth[2] / mu ;
       airout = vel[2] *area[2] ;

  // update input variable for selected flow variable
       switch (gflag) {
          case 1:  {            /* bell mouth  */
              if (ivflag == 1) {  // change in pressure
                 v1 = vel[1] * lconv ;
                 v2 = air[1] * mconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.blm.lf.f2.setText(String.valueOf(fl1)) ;
                 pan.inf.blm.lf.f3.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - bvmn)/(bvmx-bvmn))*1000.) ;
                 i2 = (int) (((v2 - bamn)/(bamx-bamn))*1000.) ;
                 pan.inf.blm.rt.s2.setValue(i1) ;
                 pan.inf.blm.rt.s3.setValue(i2) ;
              }
              if (ivflag == 2) {  // change in velocity
                 v1 = pres[1] * pconv ;
                 v2 = air[1] * mconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.blm.lf.f1.setText(String.valueOf(fl1)) ;
                 pan.inf.blm.lf.f3.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - bpmn)/(bpmx-bpmn))*1000.) ;
                 i2 = (int) (((v2 - bamn)/(bamx-bamn))*1000.) ;
                 pan.inf.blm.rt.s1.setValue(i1) ;
                 pan.inf.blm.rt.s3.setValue(i2) ;
              }
              if (ivflag == 3) {  // change in airflow
                 v1 = pres[1] * pconv ;
                 v2 = vel[1] * lconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.blm.lf.f1.setText(String.valueOf(fl1)) ;
                 pan.inf.blm.lf.f2.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - bpmn)/(bpmx-bpmn))*1000.) ;
                 i2 = (int) (((v2 - bvmn)/(bvmx-bvmn))*1000.) ;
                 pan.inf.blm.rt.s1.setValue(i1) ;
                 pan.inf.blm.rt.s2.setValue(i2) ;
              }
              break ;
          }
          case 2:  {          /* test section */
              if (ivflag == 1) {  // change in pressure
                 v1 = vel[2] * lconv ;
                 v2 = air[2] * mconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.tst.lf.f2.setText(String.valueOf(fl1)) ;
                 pan.inf.tst.lf.f3.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - tvmn)/(tvmx-tvmn))*1000.) ;
                 i2 = (int) (((v2 - tamn)/(tamx-tamn))*1000.) ;
                 pan.inf.tst.rt.s2.setValue(i1) ;
                 pan.inf.tst.rt.s3.setValue(i2) ;
              }
              if (ivflag == 2) {  // change in velocity
                 v1 = pres[2] * pconv ;
                 v2 = air[2] * mconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.tst.lf.f1.setText(String.valueOf(fl1)) ;
                 pan.inf.tst.lf.f3.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - tpmn)/(tpmx-tpmn))*1000.) ;
                 i2 = (int) (((v2 - tamn)/(tamx-tamn))*1000.) ;
                 pan.inf.tst.rt.s1.setValue(i1) ;
                 pan.inf.tst.rt.s3.setValue(i2) ;
              }
              if (ivflag == 3) {  // change in airflow
                 v1 = pres[2] * pconv ;
                 v2 = vel[2] * lconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.tst.lf.f1.setText(String.valueOf(fl1)) ;
                 pan.inf.tst.lf.f2.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - tpmn)/(tpmx-tpmn))*1000.) ;
                 i2 = (int) (((v2 - tvmn)/(tvmx-tvmn))*1000.) ;
                 pan.inf.tst.rt.s1.setValue(i1) ;
                 pan.inf.tst.rt.s2.setValue(i2) ;
              }
              break ;
          }
          case 3:  {            /*  diffuser */
              if (ivflag == 1) {  // change in pressure
                 v1 = vel[3] * lconv ;
                 v2 = air[3] * mconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.dif.lf.f2.setText(String.valueOf(fl1)) ;
                 pan.inf.dif.lf.f3.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - dvmn)/(dvmx-dvmn))*1000.) ;
                 i2 = (int) (((v2 - damn)/(damx-damn))*1000.) ;
                 pan.inf.dif.rt.s2.setValue(i1) ;
                 pan.inf.dif.rt.s3.setValue(i2) ;
              }
              if (ivflag == 2) {  // change in velocity
                 v1 = pres[3] * pconv ;
                 v2 = air[3] * mconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.dif.lf.f1.setText(String.valueOf(fl1)) ;
                 pan.inf.dif.lf.f3.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - dpmn)/(dpmx-dpmn))*1000.) ;
                 i2 = (int) (((v2 - damn)/(damx-damn))*1000.) ;
                 pan.inf.dif.rt.s1.setValue(i1) ;
                 pan.inf.dif.rt.s3.setValue(i2) ;
              }
              if (ivflag == 3) {  // change in airflow
                 v1 = pres[3] * pconv ;
                 v2 = vel[3] * lconv ;
                 fl1 = filter3(v1) ;
                 fl2 = filter3(v2) ;
                 pan.inf.dif.lf.f1.setText(String.valueOf(fl1)) ;
                 pan.inf.dif.lf.f2.setText(String.valueOf(fl2)) ;
                 i1 = (int) (((v1 - dpmn)/(dpmx-dpmn))*1000.) ;
                 i2 = (int) (((v2 - dvmn)/(dvmx-dvmn))*1000.) ;
                 pan.inf.dif.rt.s1.setValue(i1) ;
                 pan.inf.dif.rt.s2.setValue(i2) ;
              }
              break ;
          }
       }
       return ;
   }

    public void getPlot () {
  //  load the plot variables
       int i ;
       double ypmax;

       for (i=1; i<=8; ++i) {
         switch (ovflag) {
           case 1: {               // velocity
                ypl[i] = ovel[i] * lconv ;         
                yplo = 0.0 ;
                factp = 180. / tvmx ;
                break ;
           }
           case 2: {               // pressure
                ypl[i] = opres[i] * pconv ;         
                yplo = 13.7 * pconv ;
                factp = 180. / (1.0 * pconv) ;
                break ;
           }
           case 3: {               // mach
                ypl[i] = omach[i] ;         
                yplo = 0.0 ;
                factp = 180. / .3 ;
                break ;
           }
           case 4: {               // area
                ypl[i] = oarea[i] * lconv * lconv ;         
                yplo = 0.0 ;
                ypmax = 1.0 ;
                if (ypl[1] >= 1.0) ypmax = 10.0 ;
                if (ypl[1] >= 10.0) ypmax = 25.0 ;
                if (ypl[1] >= 25.0) ypmax = bwmx * blmx ;
                factp = 180. / ypmax ;
                break ;
           }
         }
       }
       return ;
   }

   class Blinker1 implements Runnable	//Class sets up blinker for the compressible flow warning label
       {
       Tuno outerparent;
       Thread blinks;

       Blinker1(Tuno target)
            {
            outerparent = target;
            blinks = null;
            }

       public void delay(int milliseconds) 
           {
           // Pause for the specified number of milliseconds,
           // where 1000 milliseconds equal one second.
           try 
               {
               Thread.sleep(milliseconds);
               }
           catch (InterruptedException e) 
               {
               }
           }

       public void start()
           {
           if (blinks == null)
                {
                blinks = new Thread(this);
                blinks.start();
                }
           }

       public void stop()
          {
          if (blinks != null)
	        {
              blinks.stop();
      	  pan.out.ler1.setForeground(Color.white) ;
      	  pan.out.ler2.setBackground(Color.white) ;
              }
          }

       public void run()
            {
            while (true)
			{
            	pan.out.ler2.setBackground(Color.white) ;
            	delay(250);
            	pan.out.ler2.setBackground(Color.red) ;
            	delay(250);
			pan.out.ler2.setBackground(Color.white);
                  delay(250);
			}
		}
       }

  class Blinker2 implements Runnable         //Class sets up blinker for the flow separation warning label
       {
       Tuno outerparent;
       Thread blinks;

       Blinker2(Tuno target)
            {
            outerparent = target;
            blinks = null;
            }

       public void delay(int milliseconds) 
           {
           // Pause for the specified number of milliseconds,
           // where 1000 milliseconds equal one second.
           try 
               {
               Thread.sleep(milliseconds);
               }
           catch (InterruptedException e) 
               {
               }
           }

       public void start()
           {
           if (blinks == null)
                {
                blinks = new Thread(this);
                blinks.start();
                }
           }

       public void stop()
          {
          if (blinks != null)
	        {
              blinks.stop();
      	  pan.out.ler1.setForeground(Color.white) ;
      	  pan.out.ler3.setBackground(Color.white) ;
              }
          }

       public void run()
            {
            while (true)
			{
            	pan.out.ler3.setBackground(Color.white) ;
            	delay(250);
            	pan.out.ler3.setBackground(Color.red) ;
            	delay(250);
			pan.out.ler3.setBackground(Color.white);
                  delay(250);
			}
		}
       }

   public void loadOut() {
      String outair,outvel,outprs,outarea,outdeg ;

      outair = " cu ft/s" ;
      if (lunits == 1) outair = " cu m/s" ;
      outvel = " ft/s" ;
      if (lunits == 1) outvel = " m/s" ;
      outprs = " psi" ;
      if (lunits == 1) outprs = " kPa" ;
      outarea = " sq ft" ;
      if (lunits == 1) outarea = " sq m" ;
      outdeg = " deg" ;

      pan.out.o1.setText(String.valueOf(filter3(lconv * vel[2])) + outvel) ;
      pan.out.o2.setText(String.valueOf(filter3(mach[2]))) ;
      pan.out.o3.setText(String.valueOf(filter3(pconv * pres[2])) + outprs) ;
      pan.out.o4.setText(String.valueOf(filter3(mconv * air[2])) + outair) ;
      pan.out.o5.setText(String.valueOf(filter0(rey))) ;
      pan.out.o6.setText(String.valueOf(filter3(cona)) + outdeg) ;

      pan.out.ler1.setForeground(Color.white) ;
      pan.out.ler2.setBackground(Color.white) ;
      pan.out.ler3.setBackground(Color.white) ;

       if (compr == 1) {
          pan.out.ler2.setVisible(true);   //makes the compressible flow label visible if there is compressible flow
          pan.out.ler1.setForeground(Color.red) ;
//          pan.out.ler2.setBackground(Color.red) ;
      }
	else
		{
		pan.out.ler2.setVisible(false);  //sets compressible flow label invisible during normal non-compressible flow
            }
     if (blsep == 1) {
	    pan.out.ler3.setVisible(true);    //sets flow separation label visible for flow separation
          pan.out.ler1.setForeground(Color.red) ;
//          pan.out.ler3.setBackground(Color.red) ;
      }
	else
		{
		pan.out.ler3.setVisible(false);    //turns the flow separation label invisible during normal conditions
		}
      
   }

   public void loadInput() {
      int i1, i2, i3, i4, i5, i6, i7, i8 ;
      Double V1, V2, V3, V4, V5, V6, V7, V8 ;
      double v1, v2, v3, v4,  v5, v6, v7, v8 ;
      float fl1, fl2, fl3, fl4, fl5, fl6, fl7, fl8 ;
                        //  labels
       if (lunits == 0) {
          pan.ing.blmr.lf.l1.setText("Height-ft") ;
          pan.ing.blmr.lf.l2.setText("Width-ft") ;
          pan.ing.blms.lf.l1.setText("Side-ft") ;
          if (xsec[1] == 3) pan.ing.blms.lf.l1.setText("Diameter-ft") ;
          pan.inf.blm.lf.l4.setText("Length-ft") ;
          pan.inf.blm.lf.l1.setText("Pressure-psi") ;
          pan.inf.blm.lf.l2.setText("Velocity-ft/s") ;
          pan.inf.blm.lf.l3.setText("Airflow-cu ft/s") ;
          pan.ing.tstr.lf.l1.setText("Height-ft") ;
          pan.ing.tstr.lf.l2.setText("Width-ft") ;
          pan.ing.tsts.lf.l1.setText("Side-ft") ;
          if (xsec[2] == 3) pan.ing.tsts.lf.l1.setText("Diameter-ft") ;
          pan.inf.tst.lf.l4.setText("Length-ft") ;
          pan.inf.tst.lf.l1.setText("Pressure-psi") ;
          pan.inf.tst.lf.l2.setText("Velocity-ft/s") ;
          pan.inf.tst.lf.l3.setText("Airflow-cu ft/s") ;
          pan.ing.difr.lf.l1.setText("Height-ft") ;
          pan.ing.difr.lf.l2.setText("Width-ft") ;
          pan.ing.difs.lf.l1.setText("Side-ft") ;
          if (xsec[3] == 3) pan.ing.difs.lf.l1.setText("Diameter-ft") ;
          pan.inf.dif.lf.l4.setText("Length-ft") ;
          pan.inf.dif.lf.l1.setText("Pressure-psi") ;
          pan.inf.dif.lf.l2.setText("Velocity-ft/s") ;
          pan.inf.dif.lf.l3.setText("Airflow-cu ft/s") ;
       }
       if (lunits == 1) {
          pan.ing.blmr.lf.l1.setText("Height-m") ;
          pan.ing.blmr.lf.l2.setText("Width-m") ;
          pan.ing.blms.lf.l1.setText("Side-m") ;
          if (xsec[1] == 3) pan.ing.blms.lf.l1.setText("Diameter-m") ;
          pan.inf.blm.lf.l4.setText("Length-m") ;
          pan.inf.blm.lf.l1.setText("Pressure-kPa") ;
          pan.inf.blm.lf.l2.setText("Velocity-m/s") ;
          pan.inf.blm.lf.l3.setText("Airflow-cu m/s") ;
          pan.ing.tstr.lf.l1.setText("Height-m") ;
          pan.ing.tstr.lf.l2.setText("Width-m") ;
          pan.ing.tsts.lf.l1.setText("Side-m") ;
          if (xsec[2] == 3) pan.ing.tsts.lf.l1.setText("Diameter-m") ;
          pan.inf.tst.lf.l4.setText("Length-m") ;
          pan.inf.tst.lf.l1.setText("Pressure-kPa") ;
          pan.inf.tst.lf.l2.setText("Velocity-m/s") ;
          pan.inf.tst.lf.l3.setText("Airflow-cu m/s") ;
          pan.ing.difr.lf.l1.setText("Height-m") ;
          pan.ing.difr.lf.l2.setText("Width-m") ;
          pan.ing.difs.lf.l1.setText("Side-m") ;
          if(xsec[3] == 3) pan.ing.difs.lf.l1.setText("Diameter-m") ;
          pan.inf.dif.lf.l4.setText("Length-m") ;
          pan.inf.dif.lf.l1.setText("Pressure-kPa") ;
          pan.inf.dif.lf.l2.setText("Velocity-m/s") ;
          pan.inf.dif.lf.l3.setText("Airflow-cu m/s") ;
       }

  // bellmouth
       v1 = hite[1] * lconv ;
       v2 = wdth[1] * lconv ;
       v3 = slopew[1] ;
       v8 = slopeh[1] ;
       v4 = length[1] * lconv ;
       v5 = pres[1] * pconv ;
       v6 = vel[1] * lconv ;
       v7 = air[1] * mconv ;

       fl1 = filter3(v1) ;
       fl2 = filter3(v2) ;
       fl3 = filter3(v3) ;
       fl8 = filter3(v8) ;
       fl4 = filter3(v4) ;
       fl5 = filter3(v5) ;
       fl6 = filter3(v6) ;
       fl7 = filter3(v7) ;

       pan.ing.blmr.lf.f1.setText(String.valueOf(fl1)) ;
       pan.ing.blms.lf.f1.setText(String.valueOf(fl2)) ;
       pan.ing.blmr.lf.f2.setText(String.valueOf(fl2)) ;
       pan.ing.blmr.lf.f3.setText(String.valueOf(fl3)) ;
       pan.ing.blms.lf.f3.setText(String.valueOf(fl3)) ;
       pan.ing.blmr.lf.f4.setText(String.valueOf(fl8)) ;
       pan.inf.blm.lf.f4.setText(String.valueOf(fl4)) ;
       pan.inf.blm.lf.f1.setText(String.valueOf(fl5)) ;
       pan.inf.blm.lf.f2.setText(String.valueOf(fl6)) ;
       pan.inf.blm.lf.f3.setText(String.valueOf(fl7)) ;

       i1 = (int) (((v1 - bhmn)/(bhmx-bhmn))*1000.) ;
       i2 = (int) (((v2 - bwmn)/(bwmx-bwmn))*1000.) ;
       i3 = (int) (((v3 - bsmn)/(bsmx-bsmn))*1000.) ;
       i8 = (int) (((v8 - bsmn)/(bsmx-bsmn))*1000.) ;
       i4 = (int) (((v4 - blmn)/(blmx-blmn))*1000.) ;
       i5 = (int) (((v5 - bpmn)/(bpmx-bpmn))*1000.) ;
       i6 = (int) (((v6 - bvmn)/(bvmx-bvmn))*1000.) ;
       i7 = (int) (((v7 - bamn)/(bamx-bamn))*1000.) ;

       pan.ing.blmr.rt.s1.setValue(i1) ;
       pan.ing.blms.rt.s1.setValue(i2) ;
       pan.ing.blmr.rt.s2.setValue(i2) ;
       pan.ing.blmr.rt.s3.setValue(i3) ;
       pan.ing.blms.rt.s3.setValue(i3) ;
       pan.ing.blmr.rt.s4.setValue(i8) ;
       pan.inf.blm.rt.s4.setValue(i4) ;
       pan.inf.blm.rt.s1.setValue(i5) ;
       pan.inf.blm.rt.s2.setValue(i6) ;
       pan.inf.blm.rt.s3.setValue(i7) ;

       i1 = xsec[1] -1 ;
       pan.inf.blm.lf.shapch.select(i1) ;
       i2 = shapew[1] -1 ;
       pan.ing.blmr.lf.bmshpw.select(i2) ;
       i3 = shapeh[1] -1 ;
       pan.ing.blmr.lf.bmshph.select(i3) ;
       pan.ing.blms.lf.bmshpc.select(i2) ;

  // test section
       v1 = hite[2] * lconv ;
       v2 = wdth[2] * lconv ;
       v4 = length[2] * lconv ;
       v5 = pres[2] * pconv ;
       v6 = vel[2] * lconv ;
       v7 = air[2] * mconv ;

       fl1 = filter3(v1) ;
       fl2 = filter3(v2) ;
       fl4 = filter3(v4) ;
       fl5 = filter3(v5) ;
       fl6 = filter3(v6) ;
       fl7 = filter3(v7) ;

       pan.ing.tstr.lf.f1.setText(String.valueOf(fl1)) ;
       pan.ing.tsts.lf.f1.setText(String.valueOf(fl2)) ;
       pan.ing.tstr.lf.f2.setText(String.valueOf(fl2)) ;
       pan.inf.tst.lf.f4.setText(String.valueOf(fl4)) ;
       pan.inf.tst.lf.f1.setText(String.valueOf(fl5)) ;
       pan.inf.tst.lf.f2.setText(String.valueOf(fl6)) ;
       pan.inf.tst.lf.f3.setText(String.valueOf(fl7)) ;

       i1 = (int) (((v1 - thmn)/(thmx-thmn))*1000.) ;
       i2 = (int) (((v2 - twmn)/(twmx-twmn))*1000.) ;
       i4 = (int) (((v4 - tlmn)/(tlmx-tlmn))*1000.) ;
       i5 = (int) (((v5 - tpmn)/(tpmx-tpmn))*1000.) ;
       i6 = (int) (((v6 - tvmn)/(tvmx-tvmn))*1000.) ;
       i7 = (int) (((v7 - tamn)/(tamx-tamn))*1000.) ;

       pan.ing.tstr.rt.s1.setValue(i1) ;
       pan.ing.tsts.rt.s1.setValue(i2) ;
       pan.ing.tstr.rt.s2.setValue(i2) ;
       pan.inf.tst.rt.s4.setValue(i4) ;
       pan.inf.tst.rt.s1.setValue(i5) ;
       pan.inf.tst.rt.s2.setValue(i6) ;
       pan.inf.tst.rt.s3.setValue(i7) ;

       i1 = xsec[2] -1 ;
       pan.inf.tst.lf.shapch.select(i1) ;

  // diffuser
       v1 = hite[3] * lconv ;
       v2 = wdth[3] * lconv ;
       v4 = length[3] * lconv ;
       v5 = pres[3] * pconv ;
       v6 = vel[3] * lconv ;
       v7 = air[3] * mconv ;

       fl1 = filter3(v1) ;
       fl2 = filter3(v2) ;
       fl4 = filter3(v4) ;
       fl5 = filter3(v5) ;
       fl6 = filter3(v6) ;
       fl7 = filter3(v7) ;

       pan.ing.difr.lf.f1.setText(String.valueOf(fl1)) ;
       pan.ing.difs.lf.f1.setText(String.valueOf(fl2)) ;
       pan.ing.difr.lf.f2.setText(String.valueOf(fl2)) ;
       pan.inf.dif.lf.f4.setText(String.valueOf(fl4)) ;
       pan.inf.dif.lf.f1.setText(String.valueOf(fl5)) ;
       pan.inf.dif.lf.f2.setText(String.valueOf(fl6)) ;
       pan.inf.dif.lf.f3.setText(String.valueOf(fl7)) ;

       i1 = (int) (((v1 - dhmn)/(dhmx-dhmn))*1000.) ;
       i2 = (int) (((v2 - dwmn)/(dwmx-dwmn))*1000.) ;
       i4 = (int) (((v4 - dlmn)/(dlmx-dlmn))*1000.) ;
       i5 = (int) (((v5 - dpmn)/(dpmx-dpmn))*1000.) ;
       i6 = (int) (((v6 - dvmn)/(dvmx-dvmn))*1000.) ;
       i7 = (int) (((v7 - damn)/(damx-damn))*1000.) ;

       pan.ing.difr.rt.s1.setValue(i1) ;
       pan.ing.difs.rt.s1.setValue(i2) ;
       pan.ing.difr.rt.s2.setValue(i2) ;
       pan.inf.dif.rt.s4.setValue(i4) ;
       pan.inf.dif.rt.s1.setValue(i5) ;
       pan.inf.dif.rt.s2.setValue(i6) ;
       pan.inf.dif.rt.s3.setValue(i7) ;

       i1 = xsec[3] -1 ;
       pan.inf.dif.lf.shapch.select(i1) ;
   }

   public int filter0(double inumbr) {
     //  output only to .
       int number ;

       number = (int) (inumbr) ;
       return number ;
   }
 
   public float filter2(double inumbr) {
     //  output only to .01
       float number ;
       int intermed ;

       intermed = (int) (inumbr * 100.) ;
       number = (float) (intermed / 100. );
       return number ;
   }

   public float filter3(double inumbr) {
     //  output only to .001
       float number ;
       int intermed ;

       intermed = (int) (inumbr * 1000.) ;
       number = (float) (intermed / 1000. );
       return number ;
   }

   public double getVel (double pressure)  {
      // utility to get velocity in ft/s given the pressure in psi
       double number ;

       number = Math.sqrt( 2.0 * (14.7 - pressure) * 144. / rho );
       return number ;
   }

   public double getPres (double velocity)  {
      // utility to get pressure in psi  given the velocity in ft/s
       double number ;

       number = 14.7 - rho * velocity * velocity * .5 / 144. ;
       return number ;
   }

   public void setUnits() {   // Switching Units
      double bwmns, bwmxs, bhmns, bhmxs, blmns, blmxs ;
      double bpmns, bpmxs, bvmns, bvmxs, bamns, bamxs ;
      double twmns, twmxs, thmns, thmxs, tlmns, tlmxs ;
      double tpmns, tpmxs, tvmns, tvmxs, tamns, tamxs ;
      double dwmns, dwmxs, dhmns, dhmxs, dlmns, dlmxs ;
      double dpmns, dpmxs, dvmns, dvmxs, damns, damxs ;

      bwmns = bwmn / lconv ;
      bwmxs = bwmx / lconv ;
      bhmns = bhmn / lconv ;
      bhmxs = bhmx / lconv ;
      blmns = blmn / lconv ;
      blmxs = blmx / lconv ;
      bpmns = bpmn / pconv ;
      bpmxs = bpmx / pconv ;
      bvmns = bvmn / lconv ;
      bvmxs = bvmx / lconv ;
      bamns = bamn / mconv ;
      bamxs = bamx / mconv ;
      twmns = twmn / lconv ;
      twmxs = twmx / lconv ;
      thmns = thmn / lconv ;
      thmxs = thmx / lconv ;
      tlmns = tlmn / lconv ;
      tlmxs = tlmx / lconv ;
      tpmns = tpmn / pconv ;
      tpmxs = tpmx / pconv ;
      tvmns = tvmn / lconv ;
      tvmxs = tvmx / lconv ;
      tamns = tamn / mconv ;
      tamxs = tamx / mconv ;
      dwmns = dwmn / lconv ;
      dwmxs = dwmx / lconv ;
      dhmns = dhmn / lconv ;
      dhmxs = dhmx / lconv ;
      dlmns = dlmn / lconv ;
      dlmxs = dlmx / lconv ;
      dpmns = dpmn / pconv ;
      dpmxs = dpmx / pconv ;
      dvmns = dvmn / lconv ;
      dvmxs = dvmx / lconv ;
      damns = damn / mconv ;
      damxs = damx / mconv ;

      switch (lunits) {
          case 0: {                             /* British Imperial */
            lconv = 1.;                         /*  feet    */
            pconv = 1.0  ;                     /* lb/sq in */
            mconv = 1.0 ;                   /* airflow rate cu ft/sec*/
            break;
          }
          case 1: {                             /* Metric */
            lconv = .3048;                      /* meters */
            pconv = 6.895 ;                   /* kilo-pascals */
            mconv = .02832 ;                    /* cu m/sec */
            break ;
          }
       }

       bwmn = bwmns * lconv ;
       bwmx = bwmxs * lconv ;
       bhmn = bhmns * lconv ;
       bhmx = bhmxs * lconv ;
       blmn = blmns * lconv ;
       blmx = blmxs * lconv ;
       bpmn = bpmns * pconv ;
       bpmx = bpmxs * pconv ;
       bvmn = bvmns * lconv ;
       bvmx = bvmxs * lconv ;
       bamn = bamns * mconv ;
       bamx = bamxs * mconv ;
       twmn = bwmns * lconv ;
       twmx = bwmxs * lconv ;
       thmn = thmns * lconv ;
       thmx = thmxs * lconv ;
       tlmn = tlmns * lconv ;
       tlmx = tlmxs * lconv ;
       tpmn = tpmns * pconv ;
       tpmx = tpmxs * pconv ;
       tvmn = tvmns * lconv ;
       tvmx = tvmxs * lconv ;
       tamn = tamns * mconv ;
       tamx = tamxs * mconv ;
       dwmn = dwmns * lconv ;
       dwmx = dwmxs * lconv ;
       dhmn = dhmns * lconv ;
       dhmx = dhmxs * lconv ;
       dlmn = dlmns * lconv ;
       dlmx = dlmxs * lconv ;
       dpmn = dpmns * pconv ;
       dpmx = dpmxs * pconv ;
       dvmn = dvmns * lconv ;
       dvmx = dvmxs * lconv ;
       damn = damns * mconv ;
       damx = damxs * mconv ;
 
       loadInput () ;
       return ;
    }

   class Pan extends Panel {
        Tuno outerparent ;
        Con con ;
        Ing ing ;
        Inf inf ;
        Out out ;

        Pan (Tuno target) {

           outerparent = target ;
           setLayout(new GridLayout(4,1,5,5)) ;

           con = new Con(outerparent) ;
           ing = new Ing(outerparent) ;
           inf = new Inf(outerparent) ;
           out = new Out(outerparent) ;

           add(con) ;
           add(inf) ;
           add(ing) ;
           add(out) ;
        }

        class Con extends Panel {
           Tuno outerparent ;
           Label l1,l2,l3,l4,l5,l6 ;
           Choice untch ;
           Button b3,bbm,bts,bdf ;
           TextField o1,o2,o3;
        
           Con (Tuno target) {

               outerparent = target ;
               setLayout(new GridLayout(4,3,5,5)) ;

               l1 = new Label("TunnelSim ", Label.RIGHT) ;
               l1.setForeground(Color.red) ;

               l5 = new Label("Version 1.0g", Label.CENTER) ;
               l5.setForeground(Color.red) ;
 
               l2 = new Label("- Open Return", Label.LEFT) ;
               l2.setForeground(Color.red) ;

               l3 = new Label("1-D Simulation", Label.CENTER) ;
               l3.setForeground(Color.red) ;

               l4 = new Label("Ideal Flow", Label.CENTER) ;
               l4.setForeground(Color.red) ;
 
               b3 = new Button("Reset") ;
               b3.setBackground(Color.red) ;
               b3.setForeground(Color.white) ;

               bbm = new Button("Bellmouth") ;
               bbm.setBackground(Color.yellow) ;
               bbm.setForeground(Color.black) ;

               bts = new Button("Test Section") ;
               bts.setBackground(Color.white) ;
               bts.setForeground(Color.black) ;

               bdf = new Button("Diffuser") ;
               bdf.setBackground(Color.white) ;
               bdf.setForeground(Color.black) ;

               untch = new Choice() ;
               untch.addItem("Imperial Units") ;
               untch.addItem("Metric Units");
               untch.select(0) ;
     
               o1 = new TextField("2.0",5) ;
               o2 = new TextField("2.0",5) ;
               o3 = new TextField("2.0",5) ;

               add(l1) ;
               add(l2) ;
               add(b3) ;

               add(new Label(" ", Label.RIGHT)) ;
               add(l5) ;
               add(untch) ;

               add(l3) ;
               add(new Label("Sections", Label.CENTER)) ;
               add(l4) ;

  //             add(o1) ;
 //              add(o2) ;
  //             add(o3) ;

               add(bbm) ;
               add(bts) ;
               add(bdf) ;
            }
    
            public boolean action(Event evt, Object arg) {
                String label = (String)arg ;
                int  whichin ;
     
                if (evt.target instanceof Choice) {
                    lunits = untch.getSelectedIndex() ;

                    setUnits() ;

                    comPute() ;
                    return true ;
                }
    
                if(evt.target instanceof Button) {
                   if(label.equals("Reset")) {
                      setDefaults() ;
                      bbm.setBackground(Color.yellow) ;
                      bts.setBackground(Color.white) ;
                      bdf.setBackground(Color.white) ;
                      layinf.show(inf, "first")  ;
                      if (xsec[1] <= 1) laying.show(ing, "first")  ;
                      if (xsec[1] > 1) laying.show(ing, "second")  ;

                      setUnits() ;
                      loadInput() ;
                      untch.select(0) ;
                      comPute() ;
                   } 
                   if(label.equals("Bellmouth")) {
                      gflag = 1;
                      bbm.setBackground(Color.yellow) ;
                      bts.setBackground(Color.white) ;
                      bdf.setBackground(Color.white) ;
                      layinf.show(inf, "first")  ;
                      if (xsec[1] <= 1) laying.show(ing, "first")  ;
                      if (xsec[1] > 1) laying.show(ing, "second")  ;
                      loadInput() ;
                   } 
                   if(label.equals("Test Section")) {
                      gflag = 2;
                      bbm.setBackground(Color.white) ;
                      bts.setBackground(Color.yellow) ;
                      bdf.setBackground(Color.white) ;
                      layinf.show(inf, "second")  ;
                      if (xsec[2] <= 1) laying.show(ing, "third")  ;
                      if (xsec[2] > 1) laying.show(ing, "fourth")  ;
                      loadInput() ;
                   } 
                   if(label.equals("Diffuser")) {
                      gflag = 3;
                      bbm.setBackground(Color.white) ;
                      bts.setBackground(Color.white) ;
                      bdf.setBackground(Color.yellow) ;
                      layinf.show(inf, "third")  ;
                      if (xsec[3] <= 1) laying.show(ing, "fifth")  ;
                      if (xsec[3] > 1) laying.show(ing, "sixth")  ;
                      loadInput() ;
                   } 
                   return true ;
                }
                else return false ;
            }
 
        }  // end Con

        class Ing extends Panel {
             Tuno outerparent ;
             Blmr blmr ;
             Blms blms ;
             Tstr tstr ;
             Tsts tsts ;
             Difr difr ;
             Difs difs ;
     
             Ing (Tuno target) {

                outerparent = target ;
                laying = new CardLayout() ;
                setLayout(laying) ;

                blmr = new Blmr(outerparent) ;
                blms = new Blms(outerparent) ;
                tstr = new Tstr(outerparent) ;
                tsts = new Tsts(outerparent) ;
                difr = new Difr(outerparent) ;
                difs = new Difs(outerparent) ;

                add ("first", blmr) ;
                add ("second", blms) ;
                add ("third", tstr) ;
                add ("fourth", tsts) ;
                add ("fifth", difr) ;
                add ("sixth", difs) ;
     
             }
 
   // Bellmouth
             class Blmr extends Panel {  // rectangular geometery
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Blmr (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2,f3,f4;
                      Label l1,l2 ;
                      Choice bmshpw,bmshph ;
       
                      Lf (Tuno target) {
       
                        outerparent = target ;
                        setLayout(new GridLayout(4,2,2,5)) ;

                        l1 = new Label("Height-ft", Label.CENTER) ;
                        f1 = new TextField("1.0",5) ;

                        l2 = new Label("Width-ft", Label.CENTER) ;
                        f2 = new TextField("2.0",5) ;

                        f3 = new TextField("-1.0",5) ;
                        f4 = new TextField("0.0",5) ;

                        bmshpw = new Choice() ;
                        bmshpw.addItem("Linear");
                        bmshpw.addItem("Quadratic");
                        bmshpw.addItem("Cubic") ;
                        bmshpw.setBackground(Color.white) ;
                        bmshpw.setForeground(Color.blue) ;
                        bmshpw.select(1) ;

                        bmshph = new Choice() ;
                        bmshph.addItem("Linear");
                        bmshph.addItem("Quadratic");
                        bmshph.addItem("Cubic") ;
                        bmshph.setBackground(Color.white) ;
                        bmshph.setForeground(Color.blue) ;
                        bmshph.select(0) ;


                        add(l2) ;
                        add(f2) ;
                        add(bmshpw) ;
                        add(f3) ;

                        add(l1) ;
                        add(f1) ;
                        add(bmshph) ;
                        add(f4) ;
                    }
 
                    public boolean handleEvent(Event evt) {
                        Double V1, V2, V3, V4 ;
                        double v1, v2, v3, v4 ;
                        float fl1 ;
                        int i1, i2, i3, i4 ;

                        if(evt.id == Event.ACTION_EVENT) {

                           V1 = Double.valueOf(f1.getText()) ;
                           v1 = V1.doubleValue() ;
                           V2 = Double.valueOf(f2.getText()) ;
                           v2 = V2.doubleValue() ;
                           V3 = Double.valueOf(f3.getText()) ;
                           v3 = V3.doubleValue() ;
                           V4 = Double.valueOf(f4.getText()) ;
                           v4 = V4.doubleValue() ;
 
                           shapew[1] = bmshpw.getSelectedIndex() + 1 ;
                           shapeh[1] = bmshph.getSelectedIndex() + 1 ;
                       // slope - width
                           slopew[1] = v3 ;
                           if(v3 < bsmn) {
                              slopew[1] = v3 = bsmn ;
                              fl1 = (float) v3 ;
                              f3.setText(String.valueOf(fl1)) ;
                           }
                           if(v3 > bsmx) {
                              slopew[1] = v3 = bsmx ;
                              fl1 = (float) v3 ;
                              f3.setText(String.valueOf(fl1)) ;
                           }
                       // slope - height
                           slopeh[1] = v4 ;
                           if(v4 < bsmn) {
                              slopeh[1] = v4 = bsmn ;
                              fl1 = (float) v4 ;
                              f4.setText(String.valueOf(fl1)) ;
                           }
                           if(v4 > bsmx) {
                              slopeh[1] = v4 = bsmx ;
                              fl1 = (float) v4 ;
                              f4.setText(String.valueOf(fl1)) ;
                           }

                       // height
                           if(v1 < bhmn) {
                              v1 = bhmn ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           if(v1 > bhmx) {
                              v1 = bhmx ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           hite[1] = v1 / lconv ;

                       // width
                           if(v2 < bwmn) {
                              v2 = bwmn ;
                              fl1 = (float) v2 ;
                              f2.setText(String.valueOf(fl1)) ;
                           }
                           if(v2 > bwmx) {
                              v2 = bwmx ;
                              fl1 = (float) v2 ;
                              f2.setText(String.valueOf(fl1)) ;
                           }
                           wdth[1] = v2 / lconv ;

                           i1 = (int) (((v1 - bhmn)/(bhmx-bhmn))*1000.) ;
                           i2 = (int) (((v2 - bwmn)/(bwmx-bwmn))*1000.) ;
                           i3 = (int) (((v3 - bsmn)/(bsmx-bsmn))*1000.) ;
                           i4 = (int) (((v4 - bsmn)/(bsmx-bsmn))*1000.) ;

                           rt.s1.setValue(i1) ;
                           rt.s2.setValue(i2) ;
                           rt.s3.setValue(i3) ;
                           rt.s4.setValue(i4) ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1, s2, s3, s4 ;

                     Rt (Tuno target) {
                        int i1,i2,i3,i4 ;

                        outerparent = target ;
                        setLayout(new GridLayout(4,1,5,5)) ;

                        i1 = (int) (((1.0 - bhmn)/(bhmx-bhmn))*1000.) ;
                        i2 = (int) (((2.0 - bwmn)/(bwmx-bwmn))*1000.) ;
                        i3 = (int) (((-1.0 - bsmn)/(bsmx-bsmn))*1000.) ;
                        i4 = (int) (((0.0 - bsmn)/(bsmx-bsmn))*1000.) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                        s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);
                        s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
                        s4 = new Scrollbar(Scrollbar.HORIZONTAL,i4,10,0,1000);

                        add(s2) ;
                        add(s3) ;
                        add(s1) ;
                        add(s4) ;
                    }

                    public boolean handleEvent(Event evt) {
                         if(evt.id == Event.ACTION_EVENT) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_ABSOLUTE) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         else return false ;
                     }

                     public void handleBar(Event evt) {
                        int i1, i2, i3, i4 ;
                        Double V1, V2, V3, V4 ;
                        double v1, v2, v3, v4 ;
                        float fl1, fl2, fl3, fl4 ;

                        i1 = s1.getValue() ;
                        i2 = s2.getValue() ;
                        i3 = s3.getValue() ;
                        i4 = s4.getValue() ;

                        v1 = i1 * (bhmx - bhmn)/ 1000. + bhmn ;
                        v2 = i2 * (bwmx - bwmn)/ 1000. + bwmn ;
                        slopew[1]  = v3 = i3 * (bsmx - bsmn)/ 1000. + bsmn ;
                        slopeh[1]  = v4 = i4 * (bsmx - bsmn)/ 1000. + bsmn ;
                        hite[1] = v1 / lconv ;
                        wdth[1] = v2 / lconv ;

                        fl1 = (float) v1 ;
                        fl2 = (float) v2 ;
                        fl3 = (float) v3 ;
                        fl4 = (float) v4 ;

                        lf.f1.setText(String.valueOf(filter3(fl1))) ;
                        lf.f2.setText(String.valueOf(filter3(fl2))) ;
                        lf.f3.setText(String.valueOf(filter3(fl3))) ;
                        lf.f4.setText(String.valueOf(filter3(fl4))) ;

                        comPute() ;
                     }
                 } // end right
             } // end blmr

             class Blms extends Panel {  // square or circular geometry
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Blms (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2,f3,f4;
                      Label l1,l2 ;
                      Choice bmshpc ;
       
                      Lf (Tuno target) {
       
                        outerparent = target ;
                        setLayout(new GridLayout(4,2,2,5)) ;

                        l1 = new Label("Side-ft", Label.CENTER) ;
                        f1 = new TextField("1.0",5) ;

                        f3 = new TextField("-1.0",5) ;

                        bmshpc = new Choice() ;
                        bmshpc.addItem("Linear");
                        bmshpc.addItem("Quadratic");
                        bmshpc.addItem("Cubic") ;
                        bmshpc.setBackground(Color.white) ;
                        bmshpc.setForeground(Color.blue) ;
                        bmshpc.select(1) ;

                        add(l1) ;
                        add(f1) ;
                        add(bmshpc) ;
                        add(f3) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }
 
                    public boolean handleEvent(Event evt) {
                        Double V1, V2, V3, V4 ;
                        double v1, v2, v3, v4 ;
                        float fl1 ;
                        int i1, i2, i3, i4 ;

                        if(evt.id == Event.ACTION_EVENT) {

                           V1 = Double.valueOf(f1.getText()) ;
                           v1 = V1.doubleValue() ;
                           V3 = Double.valueOf(f3.getText()) ;
                           v3 = V3.doubleValue() ;
 
                           shapew[1] = bmshpc.getSelectedIndex() + 1 ;
                       // slope - width
                           slopew[1] = v3 ;
                           if(v3 < bsmn) {
                              slopew[1] = v3 = bsmn ;
                              fl1 = (float) v3 ;
                              f3.setText(String.valueOf(fl1)) ;
                           }
                           if(v3 > bsmx) {
                              slopew[1] = v3 = bsmx ;
                              fl1 = (float) v3 ;
                              f3.setText(String.valueOf(fl1)) ;
                           }
                       // slope - height
                           shapeh[1] = shapew[1] ;
                           slopeh[1] = slopew[1] ;

                       // height
                           if(v1 < bhmn) {
                              v1 = bhmn ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           if(v1 > bhmx) {
                              v1 = bhmx ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           hite[1] = v1 / lconv ;

                       // width
                           wdth[1] = hite[1] ;

                           i1 = (int) (((v1 - bhmn)/(bhmx-bhmn))*1000.) ;
                           i3 = (int) (((v3 - bsmn)/(bsmx-bsmn))*1000.) ;

                           rt.s1.setValue(i1) ;
                           rt.s3.setValue(i3) ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1, s2, s3, s4 ;

                     Rt (Tuno target) {
                        int i1,i2,i3,i4 ;

                        outerparent = target ;
                        setLayout(new GridLayout(4,1,5,5)) ;

                        i1 = (int) (((1.0 - bhmn)/(bhmx-bhmn))*1000.) ;
                        i3 = (int) (((-1.0 - bsmn)/(bsmx-bsmn))*1000.) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                        s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);

                        add(s1) ;
                        add(s3) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }

                    public boolean handleEvent(Event evt) {
                         if(evt.id == Event.ACTION_EVENT) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_ABSOLUTE) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         else return false ;
                     }

                     public void handleBar(Event evt) {
                        int i1, i2, i3, i4 ;
                        Double V1, V2, V3, V4 ;
                        double v1, v2, v3, v4 ;
                        float fl1, fl2, fl3, fl4 ;

                        i1 = s1.getValue() ;
                        i3 = s3.getValue() ;

                        v1 = i1 * (bhmx - bhmn)/ 1000. + bhmn ;
                        slopew[1]  = v3 = i3 * (bsmx - bsmn)/ 1000. + bsmn ;
                        slopeh[1]  = slopew[1] ;
                        hite[1] = v1 / lconv ;
                        wdth[1] = hite[1] ;

                        fl1 = (float) v1 ;
                        fl3 = (float) v3 ;

                        lf.f1.setText(String.valueOf(filter3(fl1))) ;
                        lf.f3.setText(String.valueOf(filter3(fl3))) ;

                        comPute() ;
                     }
                 } // end right
             } // end blms

 // Test Section
             class Tstr extends Panel {  // rectangular test section
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Tstr (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2;
                      Label l1,l2 ;
       
                      Lf (Tuno target) {
       
                        outerparent = target ;
                        setLayout(new GridLayout(4,2,2,5)) ;

                        l1 = new Label("Height-ft", Label.CENTER) ;
                        f1 = new TextField("1.0",5) ;

                        l2 = new Label("Width-ft", Label.CENTER) ;
                        f2 = new TextField("1.0",5) ;

                        add(l2) ;
                        add(f2) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;

                        add(l1) ;
                        add(f1) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }
 
                    public boolean handleEvent(Event evt) {
                        Double V1, V2 ;
                        double v1, v2 ;
                        float fl1 ;
                        int i1, i2 ;

                        if(evt.id == Event.ACTION_EVENT) {

                           V1 = Double.valueOf(f1.getText()) ;
                           v1 = V1.doubleValue() ;
                           V2 = Double.valueOf(f2.getText()) ;
                           v2 = V2.doubleValue() ;
 
                       // height
                           if(v1 < thmn) {
                              v1 = thmn ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           if(v1 > thmx) {
                              v1 = thmx ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           hite[2] = v1 / lconv ;

                       // width
                           if(v2 < twmn) {
                              v2 = twmn ;
                              fl1 = (float) v2 ;
                              f2.setText(String.valueOf(fl1)) ;
                           }
                           if(v2 > twmx) {
                              v2 = twmx ;
                              fl1 = (float) v2 ;
                              f2.setText(String.valueOf(fl1)) ;
                           }
                           wdth[2] = v2 / lconv ;

                           i1 = (int) (((v1 - thmn)/(thmx-thmn))*1000.) ;
                           i2 = (int) (((v2 - twmn)/(twmx-twmn))*1000.) ;

                           rt.s1.setValue(i1) ;
                           rt.s2.setValue(i2) ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1, s2 ;

                     Rt (Tuno target) {
                        int i1,i2,i3 ;

                        outerparent = target ;
                        setLayout(new GridLayout(4,1,5,5)) ;

                        i1 = (int) (((1.0 - thmn)/(thmx-thmn))*1000.) ;
                        i2 = (int) (((1.0 - twmn)/(twmx-twmn))*1000.) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                        s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);

                        add(s2) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(s1) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }

                    public boolean handleEvent(Event evt) {
                         if(evt.id == Event.ACTION_EVENT) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_ABSOLUTE) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         else return false ;
                     }

                     public void handleBar(Event evt) {
                        int i1, i2 ;
                        Double V1, V2 ;
                        double v1, v2 ;
                        float fl1, fl2 ;

                        i1 = s1.getValue() ;
                        i2 = s2.getValue() ;

                        v1 = i1 * (thmx - thmn)/ 1000. + thmn ;
                        v2 = i2 * (twmx - twmn)/ 1000. + twmn ;
                        hite[2] = v1 / lconv ;
                        wdth[2] = v2 / lconv ;

                        fl1 = (float) v1 ;
                        fl2 = (float) v2 ;

                        lf.f1.setText(String.valueOf(filter3(fl1))) ;
                        lf.f2.setText(String.valueOf(filter3(fl2))) ;

                        comPute() ;
                     }
                 } // end right
             } // end tstr

             class Tsts extends Panel {  // square or circular test section
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Tsts (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2;
                      Label l1,l2 ;
       
                      Lf (Tuno target) {
       
                        outerparent = target ;
                        setLayout(new GridLayout(4,2,2,5)) ;

                        l1 = new Label("Side-ft", Label.CENTER) ;
                        f1 = new TextField("1.0",5) ;

                        add(l1) ;
                        add(f1) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }
 
                    public boolean handleEvent(Event evt) {
                        Double V1, V2 ;
                        double v1, v2 ;
                        float fl1 ;
                        int i1, i2 ;

                        if(evt.id == Event.ACTION_EVENT) {

                           V1 = Double.valueOf(f1.getText()) ;
                           v1 = V1.doubleValue() ;
 
                       // height
                           if(v1 < thmn) {
                              v1 = thmn ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           if(v1 > thmx) {
                              v1 = thmx ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           hite[2] = v1 / lconv ;

                       // width
                           wdth[2] = hite[2] ;

                           i1 = (int) (((v1 - thmn)/(thmx-thmn))*1000.) ;

                           rt.s1.setValue(i1) ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1, s2 ;

                     Rt (Tuno target) {
                        int i1,i2,i3 ;

                        outerparent = target ;
                        setLayout(new GridLayout(4,1,5,5)) ;

                        i1 = (int) (((1.0 - thmn)/(thmx-thmn))*1000.) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);

                        add(s1) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }

                    public boolean handleEvent(Event evt) {
                         if(evt.id == Event.ACTION_EVENT) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_ABSOLUTE) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         else return false ;
                     }

                     public void handleBar(Event evt) {
                        int i1, i2 ;
                        Double V1, V2 ;
                        double v1, v2 ;
                        float fl1, fl2 ;

                        i1 = s1.getValue() ;

                        v1 = i1 * (thmx - thmn)/ 1000. + thmn ;
                        hite[2] = v1 / lconv ;
                        wdth[2] = hite[2] ;

                        fl1 = (float) v1 ;

                        lf.f1.setText(String.valueOf(filter3(fl1))) ;

                        comPute() ;
                     }
                 } // end right
             } // end tsts

  // Diffuser
             class Difr extends Panel { // rectangular diffuser
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Difr (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2;
                      Label l1,l2 ;
       
                      Lf (Tuno target) {
       
                        outerparent = target ;
                        setLayout(new GridLayout(4,2,2,5)) ;

                        l1 = new Label("Height-ft", Label.CENTER) ;
                        f1 = new TextField("2.0",5) ;

                        l2 = new Label("Width-ft", Label.CENTER) ;
                        f2 = new TextField("2.0",5) ;

                        add(l2) ;
                        add(f2) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;

                        add(l1) ;
                        add(f1) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }
 
                    public boolean handleEvent(Event evt) {
                        Double V1, V2 ;
                        double v1, v2 ;
                        float fl1 ;
                        int i1, i2 ;

                        if(evt.id == Event.ACTION_EVENT) {

                           V1 = Double.valueOf(f1.getText()) ;
                           v1 = V1.doubleValue() ;
                           V2 = Double.valueOf(f2.getText()) ;
                           v2 = V2.doubleValue() ;
 
                       // height
                           if(v1 < dhmn) {
                              v1 = dhmn ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           if(v1 > dhmx) {
                              v1 = dhmx ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           hite[3] = v1 / lconv ;

                       // width
                           if(v2 < dwmn) {
                              v2 = dwmn ;
                              fl1 = (float) v2 ;
                              f2.setText(String.valueOf(fl1)) ;
                           }
                           if(v2 > dwmx) {
                              v2 = dwmx ;
                              fl1 = (float) v2 ;
                              f2.setText(String.valueOf(fl1)) ;
                           }
                           wdth[3] = v2 / lconv ;

                           i1 = (int) (((v1 - dhmn)/(dhmx-dhmn))*1000.) ;
                           i2 = (int) (((v2 - dwmn)/(dwmx-dwmn))*1000.) ;

                           rt.s1.setValue(i1) ;
                           rt.s2.setValue(i2) ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1, s2 ;

                     Rt (Tuno target) {
                        int i1,i2,i3 ;

                        outerparent = target ;
                        setLayout(new GridLayout(4,1,5,5)) ;

                        i1 = (int) (((2.0 - dhmn)/(dhmx-dhmn))*1000.) ;
                        i2 = (int) (((2.0 - dwmn)/(dwmx-dwmn))*1000.) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                        s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);

                        add(s2) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(s1) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }

                    public boolean handleEvent(Event evt) {
                         if(evt.id == Event.ACTION_EVENT) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_ABSOLUTE) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         else return false ;
                     }

                     public void handleBar(Event evt) {
                        int i1, i2 ;
                        Double V1, V2 ;
                        double v1, v2 ;
                        float fl1, fl2 ;

                        i1 = s1.getValue() ;
                        i2 = s2.getValue() ;

                        v1 = i1 * (dhmx - dhmn)/ 1000. + dhmn ;
                        v2 = i2 * (dwmx - dwmn)/ 1000. + dwmn ;
                        hite[3] = v1 / lconv ;
                        wdth[3] = v2 / lconv ;

                        fl1 = (float) v1 ;
                        fl2 = (float) v2 ;

                        lf.f1.setText(String.valueOf(filter3(fl1))) ;
                        lf.f2.setText(String.valueOf(filter3(fl2))) ;

                        comPute() ;
                     }
                 } // end right
             } // end difr

             class Difs extends Panel { // square or circular diffuser
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Difs (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2;
                      Label l1,l2 ;
       
                      Lf (Tuno target) {
       
                        outerparent = target ;
                        setLayout(new GridLayout(4,2,2,5)) ;

                        l1 = new Label("Side-ft", Label.CENTER) ;
                        f1 = new TextField("2.0",5) ;

                        add(l1) ;
                        add(f1) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;

                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }
 
                    public boolean handleEvent(Event evt) {
                        Double V1, V2 ;
                        double v1, v2 ;
                        float fl1 ;
                        int i1, i2 ;

                        if(evt.id == Event.ACTION_EVENT) {

                           V1 = Double.valueOf(f1.getText()) ;
                           v1 = V1.doubleValue() ;
 
                       // height
                           if(v1 < dhmn) {
                              v1 = dhmn ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           if(v1 > dhmx) {
                              v1 = dhmx ;
                              fl1 = (float) v1 ;
                              f1.setText(String.valueOf(fl1)) ;
                           }
                           hite[3] = v1 / lconv ;

                       // width
                           wdth[3] = hite[3] ;

                           i1 = (int) (((v1 - dhmn)/(dhmx-dhmn))*1000.) ;

                           rt.s1.setValue(i1) ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1, s2 ;

                     Rt (Tuno target) {
                        int i1,i2,i3 ;

                        outerparent = target ;
                        setLayout(new GridLayout(4,1,5,5)) ;

                        i1 = (int) (((2.0 - dhmn)/(dhmx-dhmn))*1000.) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);

                        add(s1) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }

                    public boolean handleEvent(Event evt) {
                         if(evt.id == Event.ACTION_EVENT) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_ABSOLUTE) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_LINE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_DOWN) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         if(evt.id == Event.SCROLL_PAGE_UP) {
                            this.handleBar(evt) ;
                            return true ;
                         }
                         else return false ;
                     }

                     public void handleBar(Event evt) {
                        int i1, i2 ;
                        Double V1, V2 ;
                        double v1, v2 ;
                        float fl1, fl2 ;

                        i1 = s1.getValue() ;

                        v1 = i1 * (dhmx - dhmn)/ 1000. + dhmn ;
                        hite[3] = v1 / lconv ;
                        wdth[3] = hite[3] ;

                        fl1 = (float) v1 ;

                        lf.f1.setText(String.valueOf(filter3(fl1))) ;

                        comPute() ;
                     }
                 } // end right
             } // end difs
        }  // end in geom

        class Inf extends Panel {
             Tuno outerparent ;
             Blm blm ;
             Tst tst ;
             Dif dif ;
     
             Inf (Tuno target) {

                outerparent = target ;
                layinf = new CardLayout() ;
                setLayout(layinf) ;

                blm = new Blm(outerparent) ;
                tst = new Tst(outerparent) ;
                dif = new Dif(outerparent) ;

                add ("first", blm) ;
                add ("second", tst) ;
                add ("third", dif) ;
     
             }
 
             class Blm extends Panel {
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Blm (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2,f3,f4;
                      Label l1,l2,l3,l4 ;
                      Choice shapch ;
       
                      Lf (Tuno target) {
       
                        outerparent = target ;
                        setLayout(new GridLayout(5,2,2,5)) ;

                        ActionListener uppres = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handp() ;
                              }
                        } ;
                        ActionListener upvel = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handv() ;
                              }
                        } ;
                        ActionListener upair = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handa() ;
                              }
                        } ;

                        l4 = new Label("Length-ft", Label.CENTER) ;
                        f4 = new TextField("2.0",5) ;

                        l1 = new Label("Pressure-psi", Label.CENTER) ;
                        f1 = new TextField("14.686",5) ;
                        f1.addActionListener(uppres) ;

                        l2 = new Label("Velocity-ft/s", Label.CENTER) ;
                        f2 = new TextField("42",5) ;
                        f2.addActionListener(upvel) ;

                        l3 = new Label("Airflow-cu ft/s", Label.CENTER) ;
                        f3 = new TextField("84",5) ;
                        f3.addActionListener(upair) ;

                        shapch = new Choice() ;
                        shapch.addItem("Rectangle") ;
                        shapch.addItem("Square");
                        shapch.addItem("Circle");
                        shapch.setBackground(Color.white) ;
                        shapch.setForeground(Color.blue) ;
                        shapch.select(0) ;

                        add(l1) ;
                        add(f1) ;

                        add(l2) ;
                        add(f2) ;

                        add(l3) ;
                        add(f3) ;

                        add(l4) ;
                        add(f4) ;

                        add(new Label("X-Section: ", Label.RIGHT)) ;
                        add(shapch) ;
                    } 
                    public void handp () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
                    // pressure
                        V1 = Double.valueOf(f1.getText()) ;
                        v1 = V1.doubleValue() ;
   
                        if(v1 < bpmn) {
                            v1 = bpmn ;
                            fl1 = (float) v1 ;
                            f1.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > bpmx) {
                            v1 = bpmx ;
                            fl1 = (float) v1 ;
                            f1.setText(String.valueOf(fl1)) ;
                         }
                         pres[1] = v1 / pconv ;
                         ivflag = 1 ;
                         i1 = (int) (((v1 - bpmn)/(bpmx-bpmn))*1000.) ;
                         rt.s1.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }

                    public void handv () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
             // velocity
                        V1 = Double.valueOf(f2.getText()) ;
                        v1 = V1.doubleValue() ;
                        if(v1 < bvmn) {
                            v1 = bvmn ;
                            fl1 = (float) v1 ;
                            f2.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > bvmx) {
                            v1 = bvmx ;
                            fl1 = (float) v1 ;
                            f2.setText(String.valueOf(fl1)) ;
                         }
                         vel[1] = v1 / lconv ;
                         ivflag = 2 ;
                         i1 = (int) (((v1 - bvmn)/(bvmx-bvmn))*1000.) ;
                         rt.s2.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }

                    public void handa () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
       // mass flow
                        V1 = Double.valueOf(f3.getText()) ;
                        v1 = V1.doubleValue() ;
 
                        if(v1 < bamn) {
                            v1 = bamn ;
                            fl1 = (float) v1 ;
                            f3.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > bamx) {
                            v1 = bamx ;
                            fl1 = (float) v1 ;
                            f3.setText(String.valueOf(fl1)) ;
                         }
                         air[1] = v1 / mconv ;
                         ivflag = 3 ;
                         i1 = (int) (((v1 - bamn)/(bamx-bamn))*1000.) ;
                         rt.s3.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }

                    public boolean handleEvent(Event evt) {
                        Double V4 ;
                        double v4 ;
                        float fl1 ;
                        int i4  ;

                        if(evt.id == Event.ACTION_EVENT) {

                           xsec[1] = shapch.getSelectedIndex() + 1 ;

                           switch (xsec[1]) {
                               case 1:  {    //  rectangular cross-section
                                   laying.show(ing, "first")  ;
                                   break ;
                               }
                               case 2:  {    //  square cross-section
                                   if (lunits == 0) ing.blms.lf.l1.setText("Side-ft") ;
                                   if (lunits == 1) ing.blms.lf.l1.setText("Side-m") ;
                                   shapeh[1] = shapew[1] ;
                                   slopeh[1] = slopew[1] ;
                                   hite[1] = wdth[1] ;
                                   laying.show(ing, "second")  ;
                                   break ;
                               }
                               case 3:  {    //  circular cross-section
                                   if (lunits == 0) ing.blms.lf.l1.setText("Diameter-ft") ;
                                   if (lunits == 1) ing.blms.lf.l1.setText("Diameter-m") ;
                                   shapeh[1] = shapew[1] ;
                                   slopeh[1] = slopew[1] ;
                                   hite[1] = wdth[1] ;
                                   laying.show(ing, "second")  ;
                                   break ;
                               }
                           }
                          V4 = Double.valueOf(f4.getText()) ;
                          v4 = V4.doubleValue() ;
                // length
                           if(v4 < blmn) {
                              v4 = blmn ;
                              fl1 = (float) v4 ;
                              f4.setText(String.valueOf(fl1)) ;
                           }
                           if(v4 > blmx) {
                              v4 = blmx ;
                              fl1 = (float) v4 ;
                              f4.setText(String.valueOf(fl1)) ;
                           }
                           length[1] = v4 / lconv ;
                        
                           loadInput() ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1, s2, s3, s4 ;

                     Rt (Tuno target) {
                        int i1,i2,i3,i4 ;

                        outerparent = target ;
                        setLayout(new GridLayout(5,1,5,5)) ;
                        AdjustmentListener upres = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handp() ;
                              }
                        } ;
                        AdjustmentListener uvel = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handv() ;
                              }
                        } ;
                        AdjustmentListener uair = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handa() ;
                              }
                        } ;
                        AdjustmentListener ulong = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handl() ;
                              }
                        } ;

                        i4 = (int) (((2.0 - blmn)/(blmx-blmn))*1000.) ;
                        i1 = (int) (((14.686 - bpmn)/(bpmx-bpmn))*1000.) ;
                        i2 = (int) (((42 - bvmn)/(bvmx-bvmn))*1000.) ;
                        i3 = (int) (((84 - bamn)/(bamx-bamn))*1000.) ;

                        s4 = new Scrollbar(Scrollbar.HORIZONTAL,i4,10,0,1000);
                        s4.addAdjustmentListener(ulong) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                        s1.addAdjustmentListener(upres) ;

                        s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);
                        s2.addAdjustmentListener(uvel) ;

                        s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
                        s3.addAdjustmentListener(uair) ;

                        add(s1) ;
                        add(s2) ;
                        add(s3) ;
                        add(s4) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }

                    public void handp () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s1.getValue() ;
                        v1 = i1 * (bpmx - bpmn)/ 1000. + bpmn ;
                        pres[1]   = v1 / pconv ;
                        fl1 = (float) v1 ;
                        lf.f1.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 1 ;
                        comPute() ;
                    }
                    public void handv () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s2.getValue() ;
                        v1 = i1 * (bvmx - bvmn)/ 1000. + bvmn ;
                        vel[1]   = v1 / lconv ;
                        fl1 = (float) v1 ;
                        lf.f2.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 2 ;
                        comPute() ;
                    }
                    public void handa () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s3.getValue() ;
                        v1 = i1 * (bamx - bamn)/ 1000. + bamn ;
                        air[1]   = v1 / mconv ;
                        fl1 = (float) v1 ;
                        lf.f3.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 3 ;
                        comPute() ;
                    }

                     public void handl() {
                        int i4 ;
                        Double V4 ;
                        double v4 ;
                        float fl4 ;
                        i4 = s4.getValue() ;

                        v4 = i4 * (blmx - blmn)/ 1000. + blmn ;
                        length[1] = v4 / lconv ;

                        fl4 = (float) v4 ;

                        lf.f4.setText(String.valueOf(filter3(fl4))) ;
                        comPute() ;
                     }
                 } // end right
             } // end blm

             class Tst extends Panel {
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Tst (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2,f3, f4;
                      Label l1,l2,l3,l4 ;
                      Choice shapch ;
       
                      Lf (Tuno target) {
       
                        outerparent = target ;
                        setLayout(new GridLayout(5,2,2,5)) ;
                        ActionListener uppres = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handp() ;
                              }
                        } ;
                        ActionListener upvel = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handv() ;
                              }
                        } ;
                        ActionListener upair = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handa() ;
                              }
                        } ;

                        l4 = new Label("Length-ft", Label.CENTER) ;
                        f4 = new TextField("1.0",5) ;

                        l1 = new Label("Pressure-psi", Label.CENTER) ;
                        f1 = new TextField("14.643",5) ;
                        f1.addActionListener(uppres) ;

                        l2 = new Label("Velocity-ft/s", Label.CENTER) ;
                        f2 = new TextField("83.0",5) ;
                        f2.addActionListener(upvel) ;

                        l3 = new Label("Airflow-cu ft/s", Label.CENTER) ;
                        f3 = new TextField("84.",5) ;
                        f3.addActionListener(upair) ;

                        shapch = new Choice() ;
                        shapch.addItem("Rectangle") ;
                        shapch.addItem("Square");
                        shapch.addItem("Circle");
                        shapch.setBackground(Color.white) ;
                        shapch.setForeground(Color.blue) ;
                        shapch.select(1) ;

                        add(l1) ;
                        add(f1) ;

                        add(l2) ;
                        add(f2) ;

                        add(l3) ;
                        add(f3) ;

                        add(l4) ;
                        add(f4) ;

                        add(new Label("X-Section ", Label.RIGHT)) ;
                        add(shapch) ;
                    }

                    public void handp () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
                    // pressure
                        V1 = Double.valueOf(f1.getText()) ;
                        v1 = V1.doubleValue() ;
   
                        if(v1 < tpmn) {
                            v1 = tpmn ;
                            fl1 = (float) v1 ;
                            f1.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > tpmx) {
                            v1 = tpmx ;
                            fl1 = (float) v1 ;
                            f1.setText(String.valueOf(fl1)) ;
                         }
                         pres[2] = v1 / pconv ;
                         ivflag = 1 ;
                         i1 = (int) (((v1 - tpmn)/(tpmx-tpmn))*1000.) ;
                         rt.s1.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }

                    public void handv () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
             // velocity
                        V1 = Double.valueOf(f2.getText()) ;
                        v1 = V1.doubleValue() ;
                        if(v1 < tvmn) {
                            v1 = tvmn ;
                            fl1 = (float) v1 ;
                            f2.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > tvmx) {
                            v1 = tvmx ;
                            fl1 = (float) v1 ;
                            f2.setText(String.valueOf(fl1)) ;
                         }
                         vel[2] = v1 / lconv ;
                         ivflag = 2 ;
                         i1 = (int) (((v1 - tvmn)/(tvmx-tvmn))*1000.) ;
                         rt.s2.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }

                    public void handa () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
       // mass flow
                        V1 = Double.valueOf(f3.getText()) ;
                        v1 = V1.doubleValue() ;
 
                        if(v1 < tamn) {
                            v1 = tamn ;
                            fl1 = (float) v1 ;
                            f3.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > tamx) {
                            v1 = tamx ;
                            fl1 = (float) v1 ;
                            f3.setText(String.valueOf(fl1)) ;
                         }
                         air[2] = v1 / mconv ;
                         ivflag = 3 ;
                         i1 = (int) (((v1 - tamn)/(tamx-tamn))*1000.) ;
                         rt.s3.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }
                    public boolean handleEvent(Event evt) {
                        Double V4 ;
                        double v4 ;
                        float fl1 ;
                        int i4 ;

                        if(evt.id == Event.ACTION_EVENT) {

                           xsec[2] = shapch.getSelectedIndex() + 1 ;

                           switch (xsec[2]) {
                               case 1:  {    //  rectangular cross-section
                                   laying.show(ing, "third")  ;
                                   break ;
                               }
                               case 2:  {    //  square cross-section
                                   if (lunits == 0) ing.tsts.lf.l1.setText("Side-ft") ;
                                   if (lunits == 1) ing.tsts.lf.l1.setText("Side-m") ;
                                   laying.show(ing, "fourth")  ;
                                   break ;
                               }
                               case 3:  {    //  circular cross-section
                                   if (lunits == 0) ing.tsts.lf.l1.setText("Diameter-ft") ;
                                   if (lunits == 1) ing.tsts.lf.l1.setText("Diameter-m") ;
                                   laying.show(ing, "fourth")  ;
                                   break ;
                               }
                           }
                           V4 = Double.valueOf(f4.getText()) ;
                           v4 = V4.doubleValue() ;
 
                       // length
                           if(v4 < tlmn) {
                              v4 = tlmn ;
                              fl1 = (float) v4 ;
                              f4.setText(String.valueOf(fl1)) ;
                           }
                           if(v4 > tlmx) {
                              v4 = tlmx ;
                              fl1 = (float) v4 ;
                              f4.setText(String.valueOf(fl1)) ;
                           }
                           length[2] = v4 / lconv ;

                           i4 = (int) (((v4 - tlmn)/(tlmx-tlmn))*1000.) ;
                           rt.s4.setValue(i4) ;

                           loadInput() ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1, s2, s3, s4 ;

                     Rt (Tuno target) {
                        int i1,i2,i3,i4 ;

                        outerparent = target ;
                        setLayout(new GridLayout(5,1,5,5)) ;
                        AdjustmentListener upres = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handp() ;
                              }
                        } ;
                        AdjustmentListener uvel = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handv() ;
                              }
                        } ;
                        AdjustmentListener uair = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handa() ;
                              }
                        } ;
                        AdjustmentListener ulong = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handl() ;
                              }
                        } ;
                        i1 = (int) (((14.643 - tpmn)/(tpmx-tpmn))*1000.) ;
                        i2 = (int) (((83 - tvmn)/(tvmx-tvmn))*1000.) ;
                        i3 = (int) (((84 - tamn)/(tamx-tamn))*1000.) ;
                        i4 = (int) (((1.0 - tlmn)/(tlmx-tlmn))*1000.) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                        s1.addAdjustmentListener(upres) ;

                        s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);
                        s2.addAdjustmentListener(uvel) ;

                        s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
                        s3.addAdjustmentListener(uair) ;

                        s4 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
                        s4.addAdjustmentListener(ulong) ;


                        add(s1) ;
                        add(s2) ;
                        add(s3) ;
                        add(s4) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }

                     public void handp () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s1.getValue() ;
                        v1 = i1 * (tpmx - tpmn)/ 1000. + tpmn ;
                        pres[2]   = v1 / pconv ;
                        fl1 = (float) v1 ;
                        lf.f1.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 1 ;
                        comPute() ;
                    }
                    public void handv () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s2.getValue() ;
                        v1 = i1 * (tvmx - tvmn)/ 1000. + tvmn ;
                        vel[2]   = v1 / lconv ;
                        fl1 = (float) v1 ;
                        lf.f2.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 2 ;
                        comPute() ;
                    }
                    public void handa () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s3.getValue() ;
                        v1 = i1 * (tamx - tamn)/ 1000. + tamn ;
                        air[2]   = v1 / mconv ;
                        fl1 = (float) v1 ;
                        lf.f3.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 3 ;
                        comPute() ;
                    }
                     public void handl() {
                        int i4 ;
                        Double V4 ;
                        double v4 ;
                        float fl4 ;
                        i4 = s4.getValue() ;

                        v4 = i4 * (tlmx - tlmn)/ 1000. + tlmn ;
                        length[2] = v4 / lconv ;

                        fl4 = (float) v4 ;

                        lf.f4.setText(String.valueOf(filter3(fl4))) ;
                        comPute() ;
                     }
                 } // end right
             } // end tst

             class Dif extends Panel {
                Tuno outerparent ;
                Lf lf ;
                Rt rt ;
        
                Dif (Tuno target) {

                   outerparent = target ;
                   setLayout(new GridLayout(1,2,5,5)) ;
 
                   lf = new Lf(outerparent) ;
                   rt = new Rt(outerparent) ;

                   add(lf) ;
                   add(rt) ;
                }
 
                class Lf extends Panel {
                      Tuno outerparent ;
                      TextField f1,f2,f3,f4;
                      Label l1,l2,l3,l4 ;
                      Choice shapch ;
      
                      Lf (Tuno target) {
     
                        outerparent = target ;
                        setLayout(new GridLayout(5,2,2,5)) ;
                        ActionListener uppres = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handp() ;
                              }
                        } ;
                        ActionListener upvel = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handv() ;
                              }
                        } ;
                        ActionListener upair = new ActionListener () {
                              public void actionPerformed(ActionEvent e) {
                                  handa() ;
                              }
                        } ;

                        l4 = new Label("Length-ft", Label.CENTER) ;
                        f4 = new TextField("5.0",5) ;

                        l1 = new Label("Pressure-psi", Label.CENTER) ;
                        f1 = new TextField("14.694",5) ;
                        f1.addActionListener(uppres) ;

                        l2 = new Label("Velocity-ft/s", Label.CENTER) ;
                        f2 = new TextField("27",5) ;
                        f2.addActionListener(upvel) ;

                        l3 = new Label("Airflow-cu ft/s", Label.CENTER) ;
                        f3 = new TextField("84.",5) ;
                        f3.addActionListener(upair) ;

                        shapch = new Choice() ;
                        shapch.addItem("Rectangle") ;
                        shapch.addItem("Square");
                        shapch.addItem("Circle");
                        shapch.setBackground(Color.white) ;
                        shapch.setForeground(Color.blue) ;
                        shapch.select(2) ;

                        add(l1) ;
                        add(f1) ;

                        add(l2) ;
                        add(f2) ;

                        add(l3) ;
                        add(f3) ;

                        add(l4) ;
                        add(f4) ;

                        add(new Label("X-Section ", Label.RIGHT)) ;
                        add(shapch) ;
                    }
 
                    public void handp () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
                    // pressure
                        V1 = Double.valueOf(f1.getText()) ;
                        v1 = V1.doubleValue() ;
   
                        if(v1 < dpmn) {
                            v1 = dpmn ;
                            fl1 = (float) v1 ;
                            f1.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > dpmx) {
                            v1 = dpmx ;
                            fl1 = (float) v1 ;
                            f1.setText(String.valueOf(fl1)) ;
                         }
                         pres[3] = v1 / pconv ;
                         ivflag = 1 ;
                         i1 = (int) (((v1 - dpmn)/(dpmx-dpmn))*1000.) ;
                         rt.s1.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }

                    public void handv () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
             // velocity
                        V1 = Double.valueOf(f2.getText()) ;
                        v1 = V1.doubleValue() ;
                        if(v1 < dvmn) {
                            v1 = dvmn ;
                            fl1 = (float) v1 ;
                            f2.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > dvmx) {
                            v1 = dvmx ;
                            fl1 = (float) v1 ;
                            f2.setText(String.valueOf(fl1)) ;
                         }
                         vel[3] = v1 / lconv ;
                         ivflag = 2 ;
                         i1 = (int) (((v1 - dvmn)/(dvmx-dvmn))*1000.) ;
                         rt.s2.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }

                    public void handa () {
                        Double V1 ;
                        double v1 ;
                        float fl1 ;
                        int i1  ;
       // mass flow
                        V1 = Double.valueOf(f3.getText()) ;
                        v1 = V1.doubleValue() ;
 
                        if(v1 < damn) {
                            v1 = damn ;
                            fl1 = (float) v1 ;
                            f3.setText(String.valueOf(fl1)) ;
                         }
                         if(v1 > damx) {
                            v1 = damx ;
                            fl1 = (float) v1 ;
                            f3.setText(String.valueOf(fl1)) ;
                         }
                         air[3] = v1 / mconv ;
                         ivflag = 3 ;
                         i1 = (int) (((v1 - damn)/(damx-damn))*1000.) ;
                         rt.s3.setValue(i1) ;
                         loadInput() ;
                         comPute() ;
                    }
                    public boolean handleEvent(Event evt) {
                        Double V4 ;
                        double v4 ;
                        float fl1 ;
                        int i4 ;

                        if(evt.id == Event.ACTION_EVENT) {

                           xsec[3] = shapch.getSelectedIndex() + 1 ;

                           switch (xsec[3]) {
                               case 1:  {    //  rectangular cross-section
                                   laying.show(ing, "fifth")  ;
                                   break ;
                               }
                               case 2:  {    //  square cross-section
                                   if (lunits == 0) ing.difs.lf.l1.setText("Side-ft") ;
                                   if (lunits == 1) ing.difs.lf.l1.setText("Side-m") ;
                                   laying.show(ing, "sixth")  ;
                                   break ;
                               }
                               case 3:  {    //  circular cross-section
                                   if (lunits == 0) ing.difs.lf.l1.setText("Diameter-ft") ;
                                   if (lunits == 1) ing.difs.lf.l1.setText("Diameter-m") ;
                                   laying.show(ing, "sixth")  ;
                                   break ;
                               }
                           }
                           V4 = Double.valueOf(f4.getText()) ;
                           v4 = V4.doubleValue() ;
                       // length
                           if(v4 < dlmn) {
                              v4 = dlmn ;
                              fl1 = (float) v4 ;
                              f4.setText(String.valueOf(fl1)) ;
                           }
                           if(v4 > dlmx) {
                              v4 = dlmx ;
                              fl1 = (float) v4 ;
                              f4.setText(String.valueOf(fl1)) ;
                           }
                           length[3] = v4 / lconv ;

                           i4 = (int) (((v4 - dlmn)/(dlmx-dlmn))*1000.) ;
                           rt.s4.setValue(i4) ;

                           loadInput() ;

                           comPute() ;

                           return true ;
                        }
                        else return false ;
                    }
                } // end left
 
                class Rt extends Panel {
                     Tuno outerparent ;
                     Scrollbar s1,s2,s3,s4 ;

                     Rt (Tuno target) {
                        int i1,i2,i3,i4 ;

                        outerparent = target ;
                        setLayout(new GridLayout(5,1,5,5)) ;
                        AdjustmentListener upres = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handp() ;
                              }
                        } ;
                        AdjustmentListener uvel = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handv() ;
                              }
                        } ;
                        AdjustmentListener uair = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handa() ;
                              }
                        } ;
                        AdjustmentListener ulong = new AdjustmentListener () {
                              public void adjustmentValueChanged(AdjustmentEvent e) {
                                  handl() ;
                              }
                        } ;

                        i1 = (int) (((14.694 - dpmn)/(dpmx-dpmn))*1000.) ;
                        i2 = (int) (((27 - dvmn)/(dvmx-dvmn))*1000.) ;
                        i3 = (int) (((84 - damn)/(damx-damn))*1000.) ;
                        i4 = (int) (((5.0 - dlmn)/(dlmx-dlmn))*1000.) ;

                        s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                        s1.addAdjustmentListener(upres) ;

                        s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);
                        s2.addAdjustmentListener(uvel) ;

                        s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
                        s3.addAdjustmentListener(uair) ;

                        s4 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
                        s4.addAdjustmentListener(ulong) ;

                        add(s1) ;
                        add(s2) ;
                        add(s3) ;
                        add(s4) ;
                        add(new Label(" ", Label.RIGHT)) ;
                    }
                     public void handp () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s1.getValue() ;
                        v1 = i1 * (dpmx - dpmn)/ 1000. + dpmn ;
                        pres[3]   = v1 / pconv ;
                        fl1 = (float) v1 ;
                        lf.f1.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 1 ;
                        comPute() ;
                    }
                    public void handv () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s2.getValue() ;
                        v1 = i1 * (dvmx - dvmn)/ 1000. + dvmn ;
                        vel[3]   = v1 / lconv ;
                        fl1 = (float) v1 ;
                        lf.f2.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 2 ;
                        comPute() ;
                    }
                    public void handa () {
                        int i1 ;
                        Double V1  ;
                        double v1 ;
                        float fl1 ;

                        i1 = s3.getValue() ;
                        v1 = i1 * (damx - damn)/ 1000. + damn ;
                        air[3]   = v1 / mconv ;
                        fl1 = (float) v1 ;
                        lf.f3.setText(String.valueOf(filter3(fl1))) ;

                        ivflag = 3 ;
                        comPute() ;
                    }
                     public void handl() {
                        int i4 ;
                        Double V4 ;
                        double v4 ;
                        float fl4 ;
                        i4 = s4.getValue() ;

                        v4 = i4 * (dlmx - dlmn)/ 1000. + dlmn ;
                        length[3] = v4 / lconv ;

                        fl4 = (float) v4 ;

                        lf.f4.setText(String.valueOf(filter3(fl4))) ;
                        comPute() ;
                     }

                 } // end right
             } // end dif
        }  // end in flow

        class Out extends Panel {
           Tuno outerparent ;
    
           Label l1,l2,l3,l4,l5,l6;
           TextField o1,o2,o3,o4,o5,o6;
           Label ler1,ler2,ler3,ler4;

           Out (Tuno target) {
      
                outerparent = target ;
                setLayout(new GridLayout(5,4,2,2)) ;

                l1 = new Label("Velocity", Label.RIGHT) ;
                o1 = new TextField("83.",5) ;
                o1.setBackground(Color.black) ;
                o1.setForeground(Color.yellow) ;

                l2 = new Label("Mach", Label.RIGHT) ;
                o2 = new TextField("0.075.",5) ;
                o2.setBackground(Color.black) ;
                o2.setForeground(Color.yellow) ;

                l3 = new Label("Pressure", Label.RIGHT) ;
                o3 = new TextField("14.643.",5) ;
                o3.setBackground(Color.black) ;
                o3.setForeground(Color.yellow) ;

                l4 = new Label("Airflow", Label.RIGHT) ;
                o4 = new TextField("5000.",5) ;
                o4.setBackground(Color.black) ;
                o4.setForeground(Color.yellow) ;

                l5 = new Label("Reynolds", Label.RIGHT) ;
                o5 = new TextField("531802.",5) ;
                o5.setBackground(Color.black) ;
                o5.setForeground(Color.yellow) ;

                l6 = new Label("Diff Angle", Label.RIGHT) ;
                o6 = new TextField("5.711.",5) ;
                o6.setBackground(Color.black) ;
                o6.setForeground(Color.yellow) ;
    
                ler1 = new Label("Error: ", Label.RIGHT) ;
                ler1.setBackground(Color.white) ;
                ler1.setForeground(Color.white) ;

                ler2 = new Label("Compressible ", Label.CENTER) ;
                ler2.setBackground(Color.white) ;
                ler2.setForeground(Color.white) ;

                ler3 = new Label("BL Separation", Label.CENTER) ;
                ler3.setBackground(Color.white) ;
                ler3.setForeground(Color.white) ;

                ler4 = new Label(" ", Label.RIGHT) ;
                ler4.setBackground(Color.white) ;
                ler4.setForeground(Color.white) ;

                add(new Label(" ", Label.RIGHT)) ;
                add(new Label("Test Section", Label.RIGHT)) ;
                add(new Label("Conditions", Label.LEFT)) ;
                add(new Label(" ", Label.RIGHT)) ;

                add(l1) ;
                add(o1) ;
                add(l3) ;
                add(o3) ;

                add(l5) ;
                add(o5) ;
                add(l4) ;
                add(o4) ;

                add(l2) ;
                add(o2) ;
                add(l6) ;
                add(o6) ;

                add(ler1) ;
                add(ler2) ;
                add(ler4) ;
                add(ler3) ;
           }

        } // end out
  } // end pan

  class Pic extends Panel {
     Tuno outerparent ;
     Viewer view ;
     Grph grph ;

     Pic (Tuno target) {

        outerparent = target ;
        setLayout(new GridLayout(2,1,5,5)) ;

        view = new Viewer(outerparent) ;
        grph = new Grph(outerparent) ;

        add(view) ;
        add(grph) ;
     }

     class Viewer extends Canvas 
         implements Runnable{
        Tuno outerparent ;
        Thread runner ;
        Point locate,anchor;

        Viewer (Tuno target) {
            setBackground(Color.white) ;
            runner = null ;
        }

        public boolean mouseDown(Event evt, int x, int y) {
           anchor = new Point(x,y) ;
           return true;
        }

        public boolean mouseUp(Event evt, int x, int y) {
           handleb(x,y) ;
           return true;
        }

        public boolean mouseDrag(Event evt, int x, int y) {
           handle(x,y) ;
           return true;
        }

        public void handle(int x, int y) {
            // determine location
            if (y >= 30) {
                if (x >= 30 ) {   // translate
                   locate = new Point(x,y) ;
                   ytv =  ytv + (int) (.2*(locate.y - anchor.y)) ;
                   xtv =  xtv + (int) (.4*(locate.x - anchor.x))  ;
                   if (xtv > 320) xtv = 320 ;
                   if (xtv < -280) xtv = -280 ;
                   if (ytv > 300) ytv = 300 ;
                   if (ytv <-300) ytv = -300 ;
                }
                if (x < 30 ) {   // zoom widget
                   sldloc = y ;
                   if (sldloc < 30) sldloc = 30;
                   if (sldloc > 165) sldloc = 165;
                   factv = 10.0 + (sldloc-30)*1.0 ;
                }

            }
        }

        public void handleb(int x, int y) {
            if (y < 15) {
                if (x <= 30) {   //find
                   factv = 30.0 ;
                   xtv = 55 ;
                   ytv = 80 ;
                   sldloc = 50 ;
                }
            }
            view.repaint() ;
        }
   
        public void start() {
           if (runner == null) {
              runner = new Thread(this) ;
              runner.start() ;
           }
           antim = 0 ;                
           ancol = 1 ;                        
        }
   
        public void run() {
          int timer ;
   
          timer = 100 ;
          while (true) {
             ++ antim ;
             try { Thread.sleep(timer); }
             catch (InterruptedException e) {}
             view.repaint() ;
             if (antim == 3) {
                antim = 0;
                ancol = - ancol ;       
             }
          }
        }
   
        public void update(Graphics g) {
            view.paint(g) ;
        }
   
        public void paint(Graphics g) {
   
          int exes[] = new int[10] ;
          int whys[] = new int[10] ;
          int yorgn = 170 ;
          int xorgn = 150 ;
          int i,j ;
          int npts,yspac,labspac,ysp1,ysp2 ;
          double delx;
   
          offsGg.setColor(Color.black) ;
          offsGg.fillRect(0,0,500,900) ;
          
          xorgn = 150 ;
          yorgn = 0 ;
   
// top view
// centerline
          delx = xwu[50] / 50. ;
            offsGg.setColor(Color.cyan) ;
            whys[0] = (int) (factv * 0.0) + ytv ;
            whys[1] = (int) (factv * 0.0) + ytv ;
          for (i=1 ; i<= 25; ++i) {
            exes[0] = (int) (factv * ((2*i -1) * delx)) + xtv ;
            exes[1] = (int) (factv * (2*i * delx)) + xtv ;
            offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }

// upper surface
//  bellmouth
          exes[1] = (int) (factv * xwu[1]) + xtv ;
          whys[1] = (int) (factv * ywu[1]) + ytv ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 1) offsGg.setColor(Color.yellow) ;
          for (i=2 ; i<= 46; ++i) {
             exes[0] = exes[1] ;
             whys[0] = whys[1] ;
             exes[1] = (int) (factv * xwu[i]) + xtv ;
             whys[1] = (int) (factv * ywu[i]) + ytv ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
// test section
          exes[0] = (int) (factv * xwu[46]) + xtv ;
          whys[0] = (int) (factv * ywu[46]) + ytv ;
          exes[1] = (int) (factv * xwu[47]) + xtv ;
          whys[1] = (int) (factv * ywu[47]) + ytv ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 2) offsGg.setColor(Color.yellow) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
// diffuser
          exes[1] = (int) (factv * xwu[47]) + xtv ;
          whys[1] = (int) (factv * ywu[47]) + ytv ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 3) offsGg.setColor(Color.yellow) ;
          for (i=48 ; i<= 50; ++i) {
             exes[0] = exes[1] ;
             whys[0] = whys[1] ;
             exes[1] = (int) (factv * xwu[i]) + xtv ;
             whys[1] = (int) (factv * ywu[i]) + ytv ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }

// lower surface
//  bellmouth
          exes[1] = (int) (factv * xwl[1]) + xtv ;
          whys[1] = (int) (factv * ywl[1]) + ytv ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 1) offsGg.setColor(Color.yellow) ;
          for (i=2 ; i<= 46; ++i) {
             exes[0] = exes[1] ;
             whys[0] = whys[1] ;
             exes[1] = (int) (factv * xwl[i]) + xtv ;
             whys[1] = (int) (factv * ywl[i]) + ytv ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
// test section
          exes[0] = (int) (factv * xwl[46]) + xtv ;
          whys[0] = (int) (factv * ywl[46]) + ytv ;
          exes[1] = (int) (factv * xwl[47]) + xtv ;
          whys[1] = (int) (factv * ywl[47]) + ytv ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 2) offsGg.setColor(Color.yellow) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
// diffuser
          exes[1] = (int) (factv * xwl[47]) + xtv ;
          whys[1] = (int) (factv * ywl[47]) + ytv ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 3) offsGg.setColor(Color.yellow) ;
          for (i=48 ; i<= 50; ++i) {
             exes[0] = exes[1] ;
             whys[0] = whys[1] ;
             exes[1] = (int) (factv * xwl[i]) + xtv ;
             whys[1] = (int) (factv * ywl[i]) + ytv ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
    
//  motor
          exes[0] = (int) (factv * xwu[50]) + xtv ;
          whys[0] = (int) (factv * ywu[50]) + ytv ;
          exes[1] = (int) (factv * xwl[50]) + xtv ;
          whys[1] = (int) (factv * ywl[50]) + ytv ;
          exes[2] = (int) (factv * (xwl[50]+ .3)) + xtv ;
          whys[2] = (int) (factv * ywl[50]) + ytv ;
          exes[3] = (int) (factv * (xwu[50]+ .3)) + xtv ;
          whys[3] = (int) (factv * ywu[50]) + ytv ;
          offsGg.setColor(Color.green) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          offsGg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
          offsGg.drawLine(exes[2],whys[2],exes[3],whys[3]) ;
          offsGg.drawLine(exes[3],whys[3],exes[0],whys[0]) ;

// side view
// centerline
          delx = xhu[50] / 50. ;
          yspac = (int) (factv * (wdth[1]/2.0 + hite[1]/2.0 + 2.0)) ;
            offsGg.setColor(Color.cyan) ;
            whys[0] = (int) (factv * 0.0) + ytv + yspac ;
            whys[1] = (int) (factv * 0.0) + ytv + yspac ;
          for (i=1 ; i<= 25; ++i) {
            exes[0] = (int) (factv * ((2*i -1) * delx)) + xtv ;
            exes[1] = (int) (factv * (2*i * delx)) + xtv ;
            offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }

// upper surface
//  bellmouth
          exes[1] = (int) (factv * xhu[1]) + xtv ;
          whys[1] = (int) (factv * yhu[1]) + ytv + yspac ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 1) offsGg.setColor(Color.yellow) ;
          for (i=2 ; i<= 46; ++i) {
             exes[0] = exes[1] ;
             whys[0] = whys[1] ;
             exes[1] = (int) (factv * xhu[i]) + xtv ;
             whys[1] = (int) (factv * yhu[i]) + ytv + yspac ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
// test section
          exes[0] = (int) (factv * xhu[46]) + xtv ;
          whys[0] = (int) (factv * yhu[46]) + ytv + yspac ;
          exes[1] = (int) (factv * xhu[47]) + xtv ;
          whys[1] = (int) (factv * yhu[47]) + ytv + yspac;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 2) offsGg.setColor(Color.yellow) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
// diffuser
          exes[1] = (int) (factv * xhu[47]) + xtv ;
          whys[1] = (int) (factv * yhu[47]) + ytv + yspac ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 3) offsGg.setColor(Color.yellow) ;
          for (i=48 ; i<= 50; ++i) {
             exes[0] = exes[1] ;
             whys[0] = whys[1] ;
             exes[1] = (int) (factv * xhu[i]) + xtv ;
             whys[1] = (int) (factv * yhu[i]) + ytv + yspac;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }

// lower surface
//  bellmouth
          exes[1] = (int) (factv * xhl[1]) + xtv ;
          whys[1] = (int) (factv * yhl[1]) + ytv + yspac ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 1) offsGg.setColor(Color.yellow) ;
          for (i=2 ; i<= 46; ++i) {
             exes[0] = exes[1] ;
             whys[0] = whys[1] ;
             exes[1] = (int) (factv * xhl[i]) + xtv ;
             whys[1] = (int) (factv * yhl[i]) + ytv + yspac ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
// test section
          exes[0] = (int) (factv * xhl[46]) + xtv ;
          whys[0] = (int) (factv * yhl[46]) + ytv + yspac ;
          exes[1] = (int) (factv * xhl[47]) + xtv ;
          whys[1] = (int) (factv * yhl[47]) + ytv + yspac;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 2) offsGg.setColor(Color.yellow) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
// diffuser
          exes[1] = (int) (factv * xhl[47]) + xtv ;
          whys[1] = (int) (factv * yhl[47]) + ytv + yspac ;
          offsGg.setColor(Color.cyan) ;
          if(gflag == 3) offsGg.setColor(Color.yellow) ;
          for (i=48 ; i<= 50; ++i) {
             exes[0] = exes[1] ;
             whys[0] = whys[1] ;
             exes[1] = (int) (factv * xhl[i]) + xtv ;
             whys[1] = (int) (factv * yhl[i]) + ytv + yspac;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }    
//  motor
          exes[0] = (int) (factv * xhu[50]) + xtv ;
          whys[0] = (int) (factv * yhu[50]) + ytv + yspac ;
          exes[1] = (int) (factv * xhl[50]) + xtv ;
          whys[1] = (int) (factv * yhl[50]) + ytv + yspac ;
          exes[2] = (int) (factv * (xhl[50]+ .3)) + xtv ;
          whys[2] = (int) (factv * yhl[50]) + ytv + yspac ;
          exes[3] = (int) (factv * (xhu[50]+ .3)) + xtv ;
          whys[3] = (int) (factv * yhu[50]) + ytv + yspac ;
          offsGg.setColor(Color.green) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          offsGg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
          offsGg.drawLine(exes[2],whys[2],exes[3],whys[3]) ;
          offsGg.drawLine(exes[3],whys[3],exes[0],whys[0]) ;

//  labels
//   stations
          offsGg.setColor(Color.white) ;
          labspac = (int) (factv * (wdth[1]/2.0 + 1.0)) ;
          ysp1 = (int) (factv * (wdth[1]/2.0 + .5)) ;
          ysp2 = (int) (factv * (wdth[1]/2.0 + 1.5)) ;
          whys[4] = (int) (factv * 0.0)  + ytv + labspac ;
          whys[2] = (int) (factv * 0.0)  + ytv + ysp2 ;
          whys[1] = (int) (factv * 0.0)  + ytv + ysp1 ;

          exes[0] = (int) (factv * 0.0) + xtv -5 ;
          whys[0] = (int) (factv * ywl[1]) + ytv ;
          whys[3] = (int) (factv * yhu[1]) + ytv + yspac  ;
          offsGg.drawLine(exes[0]+5,whys[0],exes[0]+5,whys[1]-10) ;
          offsGg.drawLine(exes[0]+5,whys[2],exes[0]+5,whys[3]) ;
          offsGg.drawString("1",exes[0],whys[4]) ;

          exes[0] = (int) (factv * x2) + xtv -5 ;
          whys[0] = (int) (factv * yw2) + ytv ;
          whys[3] = (int) (factv * yh2) + ytv + yspac  ;
          offsGg.drawLine(exes[0]+5,whys[0],exes[0]+5,whys[1]-10) ;
          offsGg.drawLine(exes[0]+5,whys[2],exes[0]+5,whys[3]) ;
          offsGg.drawString("2",exes[0],whys[4]) ;

          exes[0] = (int) (factv * x3) + xtv -5 ;
          whys[0] = (int) (factv * yw3) + ytv ;
          whys[3] = (int) (factv * yh3) + ytv + yspac  ;
          offsGg.drawLine(exes[0]+5,whys[0],exes[0]+5,whys[1]-10) ;
          offsGg.drawLine(exes[0]+5,whys[2],exes[0]+5,whys[3]) ;
          offsGg.drawString("3",exes[0],whys[4]) ;

          exes[0] = (int) (factv * xhu[46]) + xtv -5 ;
          whys[0] = (int) (factv * ywl[46]) + ytv ;
          whys[3] = (int) (factv * yhu[46]) + ytv + yspac  ;
          offsGg.drawLine(exes[0]+5,whys[0],exes[0]+5,whys[1]-10) ;
          offsGg.drawLine(exes[0]+5,whys[2],exes[0]+5,whys[3]) ;
          offsGg.drawString("4",exes[0],whys[4]) ;

          exes[0] = (int) (factv * xhu[47]) + xtv -5 ;
          whys[0] = (int) (factv * ywl[47]) + ytv ;
          whys[3] = (int) (factv * yhu[47]) + ytv + yspac  ;
          offsGg.drawLine(exes[0]+5,whys[0],exes[0]+5,whys[1]-10) ;
          offsGg.drawLine(exes[0]+5,whys[2],exes[0]+5,whys[3]) ;
          offsGg.drawString("5",exes[0],whys[4]) ;

          exes[0] = (int) (factv * xhu[48]) + xtv -5 ;
          whys[0] = (int) (factv * ywl[48]) + ytv ;
          whys[3] = (int) (factv * yhu[48]) + ytv + yspac  ;
          offsGg.drawLine(exes[0]+5,whys[0],exes[0]+5,whys[1]-10) ;
          offsGg.drawLine(exes[0]+5,whys[2],exes[0]+5,whys[3]) ;
          offsGg.drawString("6",exes[0],whys[4]) ;

          exes[0] = (int) (factv * xhu[49]) + xtv -5 ;
          whys[0] = (int) (factv * ywl[49]) + ytv ;
          whys[3] = (int) (factv * yhu[49]) + ytv + yspac  ;
          offsGg.drawLine(exes[0]+5,whys[0],exes[0]+5,whys[1]-10) ;
          offsGg.drawLine(exes[0]+5,whys[2],exes[0]+5,whys[3]) ;
          offsGg.drawString("7",exes[0],whys[4]) ;

          exes[0] = (int) (factv * xhu[50]) + xtv -5 ;
          whys[0] = (int) (factv * ywl[50]) + ytv ;
          whys[3] = (int) (factv * yhu[50]) + ytv + yspac  ;
          offsGg.drawLine(exes[0]+5,whys[0],exes[0]+5,whys[1]-10) ;
          offsGg.drawLine(exes[0]+5,whys[2],exes[0]+5,whys[3]) ;
          offsGg.drawString("8",exes[0],whys[4]) ;

// widgets
          offsGg.setColor(Color.yellow) ;
          offsGg.drawString("Find",10,15) ;

          offsGg.setColor(Color.cyan) ;
          offsGg.drawString("Top View",250,15) ;
          offsGg.drawString("Side View",250,250) ;
   
 //  zoom in
           offsGg.setColor(Color.black) ;
           offsGg.fillRect(0,30,30,150) ;
           offsGg.setColor(Color.green) ;
           offsGg.drawLine(15,35,15,165) ;
           offsGg.fillRect(5,sldloc,20,5) ;
          g.drawImage(offscreenImg,0,0,this) ;
       }
     } // end viewer

     class Grph extends Canvas 
         implements Runnable{
        Tuno outerparent ;
        Thread runner ;
        Point locate,anchor;

        Grph (Tuno target) {
            setBackground(Color.white) ;
            runner = null ;
        }

        public boolean mouseDown(Event evt, int x, int y) {
           anchor = new Point(x,y) ;
           return true;
        }

        public boolean mouseUp(Event evt, int x, int y) {
           handleb(x,y) ;
           return true;
        }

        public boolean mouseDrag(Event evt, int x, int y) {
           handle(x,y) ;
           return true;
        }

        public void handle(int x, int y) {
         // determine location
            if (y >= 30) {
                if (x >= 30 ) {   // translate
                   locate = new Point(x,y) ;
                }
                if (x < 30 ) {   // zoom widget
                }
            }
        }

        public void handleb(int x, int y) {
            if (y < 25) {    // buttons
                if (x >= 5 && x <= 85) {  // velocity
                   ovflag = 1 ;
                }
                if (x >= 90 && x <= 170) {  // pressure
                   ovflag = 2 ;
                }
                if (x >= 175 && x <= 255) {  // mach
                   ovflag = 3 ;
                }
                if (x >= 260) {  // area
                   ovflag = 4 ;
                }
                
                comPute() ;
            }
            grph.repaint() ;
        }

        public void start() {
           if (runner == null) {
              runner = new Thread(this) ;
              runner.start() ;
           }
           antg = 0 ;                
           ancg = 1 ;                        
        }

        public void run() {
          int timer ;
   
          timer = 100 ;
          while (true) {
             try { Thread.sleep(timer); }
             catch (InterruptedException e) {}
             grph.repaint() ;
             if (antg == 3) {
                antg = 0;
                ancg = - ancg ;       
             }
          }
        }

        public void update(Graphics g) {
            grph.paint(g) ;
        }

        public void paint(Graphics g) {
   
          int exes[] = new int[10] ;
          int whys[] = new int[10] ;
          int i,j ;
          int npts ;

          offs2Gg.setColor(Color.blue) ;
          offs2Gg.fillRect(0,0,500,900) ;

 // background yellow boxes
          offs2Gg.setColor(Color.yellow) ;
          offs2Gg.fillRect(4,35,40,20) ;
          offs2Gg.fillRect(46,35,40,20) ;
          offs2Gg.fillRect(89,35,40,20) ;
          offs2Gg.fillRect(131,35,40,20) ;
          offs2Gg.fillRect(174,35,40,20) ;
          offs2Gg.fillRect(216,35,40,20) ;
          offs2Gg.fillRect(259,35,40,20) ;
          offs2Gg.fillRect(301,35,40,20) ;

          offs2Gg.fillRect(4,60,40,180) ;
          offs2Gg.fillRect(46,60,40,180) ;
          offs2Gg.fillRect(89,60,40,180) ;
          offs2Gg.fillRect(131,60,40,180) ;
          offs2Gg.fillRect(174,60,40,180) ;
          offs2Gg.fillRect(216,60,40,180) ;
          offs2Gg.fillRect(259,60,40,180) ;
          offs2Gg.fillRect(301,60,40,180) ;

 // values 
          offs2Gg.setColor(Color.black) ;
          if (ovflag == 1) {
            offs2Gg.drawString(String.valueOf(filter0(ypl[1])),5,52) ;
            offs2Gg.drawString(String.valueOf(filter0(ypl[2])),47,52) ;
            offs2Gg.drawString(String.valueOf(filter0(ypl[3])),90,52) ;
            offs2Gg.drawString(String.valueOf(filter0(ypl[4])),132,52) ;
            offs2Gg.drawString(String.valueOf(filter0(ypl[5])),175,52) ;
            offs2Gg.drawString(String.valueOf(filter0(ypl[6])),217,52) ;
            offs2Gg.drawString(String.valueOf(filter0(ypl[7])),260,52) ;
            offs2Gg.drawString(String.valueOf(filter0(ypl[8])),302,52) ;
          }
          if (ovflag == 2) {
            offs2Gg.drawString(String.valueOf(filter2(ypl[1])),5,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[2])),47,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[3])),90,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[4])),132,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[5])),175,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[6])),217,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[7])),260,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[8])),302,52) ;
          }
          if (ovflag == 3) {
            offs2Gg.drawString(String.valueOf(filter3(ypl[1])),5,52) ;
            offs2Gg.drawString(String.valueOf(filter3(ypl[2])),47,52) ;
            offs2Gg.drawString(String.valueOf(filter3(ypl[3])),90,52) ;
            offs2Gg.drawString(String.valueOf(filter3(ypl[4])),132,52) ;
            offs2Gg.drawString(String.valueOf(filter3(ypl[5])),175,52) ;
            offs2Gg.drawString(String.valueOf(filter3(ypl[6])),217,52) ;
            offs2Gg.drawString(String.valueOf(filter3(ypl[7])),260,52) ;
            offs2Gg.drawString(String.valueOf(filter3(ypl[8])),302,52) ;
          }
          if (ovflag == 4) {
            offs2Gg.drawString(String.valueOf(filter2(ypl[1])),5,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[2])),47,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[3])),90,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[4])),132,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[5])),175,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[6])),217,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[7])),260,52) ;
            offs2Gg.drawString(String.valueOf(filter2(ypl[8])),302,52) ;
          }

 //  bar charts
          offs2Gg.setColor(Color.green) ;
          whys[0] =  (int) (factp * (ypl[1] - yplo))  ;      
          offs2Gg.fillRect(6,240 - whys[0],36,whys[0]) ;
          whys[0] =  (int) (factp * (ypl[2] - yplo))  ;      
          offs2Gg.fillRect(48,240 - whys[0],36,whys[0]) ;
          whys[0] =  (int) (factp * (ypl[3] - yplo))  ;      
          offs2Gg.fillRect(91,240 - whys[0],36,whys[0]) ;
          whys[0] =  (int) (factp * (ypl[4] - yplo))  ;      
          offs2Gg.fillRect(133,240 - whys[0],36,whys[0]) ;
          whys[0] =  (int) (factp * (ypl[5] - yplo))  ;      
          offs2Gg.fillRect(176,240 - whys[0],36,whys[0]) ;
          whys[0] =  (int) (factp * (ypl[6] - yplo))  ;      
          offs2Gg.fillRect(218,240 - whys[0],36,whys[0]) ;
          whys[0] =  (int) (factp * (ypl[7] - yplo))  ;      
          offs2Gg.fillRect(261,240 - whys[0],36,whys[0]) ;
          whys[0] =  (int) (factp * (ypl[8] - yplo))  ;      
          offs2Gg.fillRect(303,240 - whys[0],36,whys[0]) ;
//  label
          offs2Gg.setColor(Color.white) ;
          offs2Gg.drawString("1",20,255) ;
          offs2Gg.drawString("2",60,255) ;
          offs2Gg.drawString("3",100,255) ;
          offs2Gg.drawString("4",140,255) ;
          offs2Gg.drawString("5",180,255) ;
          offs2Gg.drawString("6",220,255) ;
          offs2Gg.drawString("7",270,255) ;
          offs2Gg.drawString("8",310,255) ;
          
//  buttons
          offs2Gg.setColor(Color.white) ;
          if (ovflag ==1) offs2Gg.setColor(Color.yellow) ;
          offs2Gg.fillRect(5,5,80,25) ;
          offs2Gg.setColor(Color.blue) ;
          offs2Gg.drawString("Velocity",10,20) ;

          offs2Gg.setColor(Color.white) ;
          if (ovflag ==2) offs2Gg.setColor(Color.yellow) ;
          offs2Gg.fillRect(90,5,80,25) ;
          offs2Gg.setColor(Color.blue) ;
          offs2Gg.drawString("Pressure",95,20) ;

          offs2Gg.setColor(Color.white) ;
          if (ovflag ==3) offs2Gg.setColor(Color.yellow) ;
          offs2Gg.fillRect(175,5,80,25) ;
          offs2Gg.setColor(Color.blue) ;
          offs2Gg.drawString("Mach",187,20) ;

          offs2Gg.setColor(Color.white) ;
          if (ovflag ==4) offs2Gg.setColor(Color.yellow) ;
          offs2Gg.fillRect(260,5,80,25) ;
          offs2Gg.setColor(Color.blue) ;
          offs2Gg.drawString("Area",272,20) ;

          g.drawImage(offscreen2Img,0,0,this) ;
       }
     } // end grph
  } // end pic
}
