#!/usr/local/bin/perl

# Script to convert a genplot file to tecplot ascii format.

  use warnings FATAL => 'uninitialized';    # Stop if uninitialized variables.
  use strict;           # Same as use strict "vars"+"refs"+"subs".
# use strict "vars";    # No undeclared variables.
# use strict "refs";    # No symbolic references.
# use strict "subs";    # No barewords.  "quote them", &sub, or sub().
  use English;          # English names for perl shortcuts
  use File::Basename;   # file name parsing
# use File::Copy;       # perl version of linux cp and mv commands
# use File::Which;      # perl version of linux which command
  use Getopt::Long;     # Process extended command line options

  my $zonenamefile = "";
  my @zonenames = ();
  my $strip_comments = 1;  # (0/1) = (N/Y) to remove comments
  my $fmt = "";            # Perl formatting string for output data. 
                           #   If empty, just copy as a line of text.
  my $fh_src = undef;      # Source      file handle
  my $fh_dst = undef;      # Destination file handle
  my $src = undef;         # Source      file name
  my $dst = undef;         # Destination file name
  my $src_linenum = 0;     # Source      file line number
  my $exit_code = 0;       # Script exit code.  0=No error (yet).

  my @varlist = undef;
  my $varlist = undef;
  my $varlist_old = undef;

# Command line arguments.
  my $status = GetOptions( 'help|h|?'       => \&Usage,
                           'fmt=s'          => \$fmt,
                           'strip_comments' => \$strip_comments,
                           'zonenames'      => \$zonenamefile,
                           '<>'             => \&gen2tec );
  unless ( $status ) { $exit_code = 1; }
  exit $exit_code;

  END_OF_MAIN:
  goto EOF;

# ====================================================================
  sub Usage{
    my $program = basename($0);

print << "USAGE";
$program
  Convert a GENPLOT file to Tecplot ascii format.
Usage:
  $program { [Options] <file1> } ...
Options:
  h | help | ?                 This help message.
  fmt="format"                 Perl format specifier for output data.
  (no)strip_comments           (keep) or strip comments.
  zonenames="zonenames_file"   Filename specifying Tecplot zone names.
USAGE

    exit 1;
  }
# ====================================================================
  sub gen2tec{

#   Read a genplot ($src) file and convert to a tecplot ($dst) file.

    $src = shift(@_);
#   Tecplot filename based on genplot name.
#   Replace ".gen" with ".tec" if found, or else append ".tec".
    $dst = $src;
    unless ( $dst =~ s/\.gen/.tec/ ) { $dst = "$dst.tec" ; }

    my @fmt = $fmt =~ /\%[^\%]+/g;     # array of format specifiers
    my $numfmt =  scalar(@fmt);        # number of format specifiers

    @varlist = undef;
    $varlist = undef;
    $varlist_old = undef;
    
#   Open source and destination files.
    open($fh_src,"<$src") or die "Unable to open file $src.";
    open($fh_dst,">$dst") or die "Unable to open file $dst";
      $src_linenum = 0;

#     Main Plot Title
      my $Main_Plot_Title = &mytrim(&readline(1));
      printf $fh_dst "TITLE=\"%s\"\n",$Main_Plot_Title;

#     Process each subplot
      while ( defined( my $Plot1_Sub_Title =  &mytrim(&readline(0)) ) ) {
        my $X_Axis_Title     = &mytrim(&readline(1));
        my $Y_Axis_Title     = &mytrim(&readline(1));
        my $Number_Of_Curves = &mytrim(&readline(1));

        if ( $Number_Of_Curves > 0 ) {

#         Genplot file with multiple curves of x,y pairs.
#         Similar to a multizone point-formatted Tecplot file.

#         Print Tecplot variable list.
          printf $fh_dst "VARIABLES=\"%s\",\"%s\"\n", $X_Axis_Title, $Y_Axis_Title;

          @varlist = ( $X_Axis_Title, $Y_Axis_Title );
          $varlist = join(",",@varlist);
#         Check whether variable list changes.
          &check_varlist();
          $varlist_old = $varlist;

#         Process each curve.
          for (my $ic=1; $ic<=$Number_Of_Curves; $ic++) {
#           Read curve data.
            my $Curve_Title = &mytrim(&readline(1));
            my $Number_Of_Points = &mytrim(&readline(1));
            my @xylines = ();
            for (my $ip=0; $ip<$Number_Of_Points; $ip++) {
              push @xylines, &readline(1);
            }

#-----------Could calc min/max values or column widths here and
#-----------create a format srting to output this curve data.
#-----------For now, let user provide format via command line.

#           Write curve data.
            printf $fh_dst "ZONE T=\"%s\" I=%s F=POINT\n",$Curve_Title,$Number_Of_Points;
            for (my $ip=0; $ip<$Number_Of_Points; $ip++) {
              if ( $fmt ) {
#               Format specifier provided by user.
                my @data = split /[ ,]+/, $xylines[$ip];
                my $numdat = scalar(@data);
                unless ( $numfmt == $numdat ) { &formaterror($numfmt,$numdat); }
                printf $fh_dst $fmt, @data;
                printf $fh_dst "\n";
              } else {
#               No format specifier provided, treat data as a string.
                printf $fh_dst "%s\n",$xylines[$ip];
              }
            } # $ip-loop
          } # $ic-loop

        } elsif ( $Number_Of_Curves < 0 ) {

#         Genplot file with x,y1,y2,...,yN.
#         Similar to a single-zone point-formatted Tecplot file,
#         where the Tecplot zone title is the subplot name 
#         and the y-variable names are really the effective zone names.

          $Number_Of_Curves *= -1;  # value is now positive

#         Read curve names.
          my @Curve_Title = ();
          my @xylines = ();
          for (my $ic=0; $ic<$Number_Of_Curves; $ic++) {
            push @Curve_Title, &mytrim(&readline(1));
          }

#         Print Tecplot variable list.
          printf $fh_dst "VARIABLES=\"%s\"",$X_Axis_Title;
          foreach (@Curve_Title) {
            printf $fh_dst ",\"%s\"",$_;
          }
          printf $fh_dst "\n";

          @varlist = ( $X_Axis_Title, @Curve_Title );
          $varlist = join(",",@varlist);
#         Check whether variable list changes.
          &check_varlist();
          $varlist_old = $varlist;

#         Read curve data.
          my $Number_Of_Points = &mytrim(&readline(1));
          for (my $ip=0; $ip<$Number_Of_Points; $ip++) {
            push @xylines, &readline(1);
          }

#---------Could calc min/max values or column widths here and
#---------create a format srting to output this curve data.
#---------For now, let user provide format via command line.

#         Write curve data.
          printf $fh_dst "ZONE T=\"%s\"\n",$Plot1_Sub_Title;
          for (my $ip=0; $ip<$Number_Of_Points; $ip++) {
            if ( $fmt ) {
#             Format specifier provided by user.
              my @data = split /[ ,]+/, $xylines[$ip];
              my $numdat = scalar(@data);
              unless ( $numfmt == $numdat ) { &formaterror($numfmt,$numdat); }
              printf $fh_dst $fmt, @data;
              printf $fh_dst "\n";
            } else {
#             No format specifier provided, treat data as a string.
              printf $fh_dst "%s\n",$xylines[$ip];
            }
          } # $ip-loop

        } else {
          die "Number of curves is 0";
        }
      } # end while
    close($fh_dst);
    $fh_dst = undef;
    close($fh_src);
    $fh_src = undef;


    return;
  }
# ====================================================================
  sub readline{

#   Read from the source file.
#   Comment lines copied to destination based on $strip_comments setting.
#   First non-comment line gets returned (or undef).

#   use global variable $fh_src
#   use global variable $fh_dst
#   use global variable $src_linenum
#   use global variable $strip_comments
    my $check_for_error = shift;
    my $line;

    while ( $line = <$fh_src> ) {
      $src_linenum += 1;
      if ( defined( $line ) ) {
        chomp $line;               # strip newlines
        if ( $line =~ /^\s*\#/ ) {
          unless ( $strip_comments ) { printf $fh_dst "%s\n", $line; }
        } else {
          last;  # break the read loop
        }
      }
    } # end while

    if ( $check_for_error ) { unless ( defined($line) ) { &readerror(); } }

    return $line;
  }
# ====================================================================
  sub readerror{
    printf "ERROR reading line %d of genplot file %s.\n",
           $src_linenum+1, $src;
    close($fh_src);
    close($fh_dst);
    exit 1;
  }
# ====================================================================
  sub formaterror{
    my $numfmt = shift;
    my $numdat = shift;

    printf "ERROR:  Format specifier mismatch.\n";
    printf "          Number of formats = %d\n", $numfmt;
    printf "          Number of values  = %d\n", $numdat;
    exit 1;
  }
# ====================================================================
  sub check_varlist{

#   Check variable list to see if it has changed.
#   Tecplot requires all zones in a datafile to have the same
#   set of variable names.

    if ( defined($varlist) && defined($varlist_old) ) {
      unless ( $varlist != $varlist_old ) {
        printf "Warning:  Variable names mismatch.\n";
        printf "          Previous set = %s\n", $varlist_old;
        printf "          Current  set = %s\n", $varlist;
        printf "          Line number %d of %s\n", $src_linenum, $src;
      }
    }
    return;
  }
# ====================================================================
  sub mytrim{

#   Trim leading and trailing space from a string.

    my $str = shift(@_);
    if ( defined ($str) ) {
      $str =~ s/^\s+//;      # trim leading  space
      $str =~ s/\s+$//;      # trim trailing space
    }
    return $str;
  }
# ====================================================================

  EOF:
