/*
                             SoccerNASA
                3D Version  --  includes gravity and drag 
   
                           A Java Applet
            to calculate the flight trajectory of a soccer ball
               Based on FoilSim II - derived from CurveBall

                     Version 1.4c   - 25 Aug 10

                         Written by Tom Benson
                       NASA Glenn Research Center

>                              NOTICE
>This software is in the Public Domain.  It may be freely copied and used in
>non-commercial products, assuming proper credit to the author is given.  IT
>MAY NOT BE RESOLD.  If you want to use the software for commercial
>products, contact the author.
>No copyright is claimed in the United States under Title 17, U. S. Code.
>This software is provided "as is" without any warranty of any kind, either
>express, implied, or statutory, including, but not limited to, any warranty
>that the software will conform to specifications, any implied warranties of
>merchantability, fitness for a particular purpose, and freedom from
>infringement, and any warranty that the documentation will conform to the
>program, or any warranty that the software will be error free.
>In no event shall NASA be liable for any damages, including, but not
>limited to direct, indirect, special or consequential damages, arising out
>of, resulting from, or in any way connected with this software, whether or
>not based on warranty, contract, tort or otherwise, whether or not injury
>was sustained by persons or property or otherwise, and whether or not loss
>was sustained from, or arose out of the results of, or use of, the software
>or services provided hereunder.
 
  New test -
             * improve analysis    (integrated cylinder)
             * add lift coefficient  (Cl = .15) 
             * change computation scheme to flight integration
             *     save pitch option
             * include the drag
             * include 3rd dimension
             *    add vertical equation
             *    vary the spin axis
             * add input panel for spin direction

             * switch to soccer ball (for the world cup)
                    rball = .75 feet
                    weight = 16 oz = 1 lb
                    cd will be different from a baseball - .25 ?
                    cl different from baseball  - .25
                      
              Change the field to look like a soccer stadium  -- 
            *       add goal area
            *       add view from kicker
            *         modify this to have perspective
            *         clean up side view perspective
            * do 3 options of graphics .. with buttons .. 
            *        downfield view - top view - side view
            *        add goal nets

            * change defaults for a launch from the ground
            *       may have to change limits on the sliders
            * change end of calc ... not lpitch but till you hit the ground 
            * put in 3 options;   
                  *  corner kick, 
                  *  free kick, 
                  *  penalty kick
            *             take out height input and slider
            * turn on input boxes
            * change the time for integration ... ball will travel farther .. but faster
            * change the graphic .. find a rotating soccer ball
            * change "Strike" to "Goal"
            * add the stadiums for the world cup ...
            * change layout ... graphics on the left .. inputs on the right (like F-Sim)
            * corrections to force determination - sign on drag term

              Big Change ---  to make corner kick easier
            * Change co-ordinate system to xyz with
                    x = 0 at the left goal line
                    y = 0 at the centerline
                    z = 0 ground level
            * Change beginning x, y, z for type of kick .. 
            *      and direction of kick velocity
            * Change graphic output to reflect this change
            * Allow input for free kick in both x and y ... 
            *      add input for x location
            * Clean-up the graphics
            * Put in corner kick input options 
      
            * Add another view .. directly from the side .. change "side view" to "TV view"
            * Add target box for corner kick .. use criteria from Cook and Goff
            *      add logic to see when you hit the target 
            * Tune the cd and cl from Goff and Care
            *       make cd a variable for input ... educational
            *       make cl a variable for input ... educational

                                           TJB  25 Aug 10

*/

import java.awt.*;
import java.lang.Math ;

public class Ball extends java.applet.Applet {
 
   static double convdr = 3.1415926/180. ;
   static double pid2 = 3.1415926/2.0 ;
   static double rval,ycval,xcval,gamval,alfval,thkval,camval,chrd,cl ;
   static double thkinpt,caminpt ;                 /* MODS 10 Sep 99 */
   static double leg,teg,lem,tem;
   static double usq,vsq,alt,altmax,area,armax,armin ;
   static double chord,span,aspr,arold,chrdold,spnold ; /* Mod 13 Jan 00 */
   static double q0,ps0,pt0,ts0,rho,rlhum,temf,presm ;
   static double lyg,lrg,lthg,lxgt,lygt,lrgt,lthgt;/* MOD 20 Jul */
   static double lxm,lym,lxmt,lymt,vxdir;/* MOD 20 Jul */
   static double deltb,xflow ;             /* MODS  20 Jul 99 */
   static double delx,delt,vfsd,spin,spindr ;
   static double vel,pres,lift,side,omega,radcrv ;
       // kick data
   static double deltim,ptim,spd,wball,rball,cd,drag,wtrat,dist,clift;
   static double relsy,relsz,relsx,yangr,zangr,sangr,relsdr ;
   static double zacc,zvel,zloc ;
   static double yacc,yvel,yloc ;
   static double xacc,xvel,xloc ;
   static double yfor,xfor,zfor;

   int ktype,cdcalc ;

   static double rg[][]  = new double[20][140] ; 
   static double thg[][] = new double[20][140] ; 
   static double xg[][]  = new double[20][140] ; 
   static double yg[][]  = new double[20][140] ; 
   static double xm[][]  = new double[20][140] ; 
   static double ym[][]  = new double[20][140] ; 

   int nptc,npt2,nlnc,nln2,rdflag,browflag;
   int foil,flflag,lunits,lftout,planet,wthopt,place ;
   int conflag,displ,antim1,ancol1,antim2,ancol2; 
       /* units data */
   static double vmn1,vmn2,vmn3,vmn4,vmn5,vmn6,vmn7 ;
   static double vmx1,vmx2,vmx3,vmx4,vmx5,vmx6,vmx7 ;
   static double vconv,vmaxa,vmaxb ;
   static double pconv,pmax,pmin,lconv,fconv,fmax,fmaxb;
   int lflag,gflag,plscale,nond ;
       /*  plot & probe data */
   static double fact,xpval,ypval,pbval;
   static double prg,pthg,pxg,pyg,pxm,pym ;
   int ptchflag,pboflag,xtt,ytt,xts,yts,xtk,ytk,xtv,ytv,ntikx,ntiky,npt,nptb ;
   int lines,nord,nabs,ntr,pitch,comp,call,nsav ;
   int drawview,sldloc ;
   static int[] npts = new int[5];
   static double begx,endx,begy,endy,begz,endz ;
   static String labx,labxu,laby,labyu ;
   static double posx[][]  = new double[5][175] ;
   static double posy[][]  = new double[5][175] ;
   static double posz[][]  = new double[5][175] ;
   static double yoff[] = new double[5] ;
   static double zoff[] = new double[5] ;
   static double pltrg[]  = new double[2] ;
   static double plthg1[]  = new double[2] ;
   static double plthg2[]  = new double[2] ;
   static double pltxg[]  = new double[2] ;
   static double pltyg[]  = new double[2] ;

   Solver solve ;
   View1 view1 ;
   Inp inp;
   Image offscreenImg ;
   Graphics offsGg ;
   Image offImg3 ;
   Graphics off3Gg ;
   Image offImg4 ;
   Graphics off4Gg ;

   public void init() {
     int i;
     Ball a = new Ball() ;
     solve = new Solver() ;

     offscreenImg = createImage(this.size().width,
                      this.size().height) ;
     offsGg = offscreenImg.getGraphics() ;
     offImg3 = createImage(this.size().width,
                      this.size().height) ;
     off3Gg = offImg3.getGraphics() ;
     offImg4 = createImage(this.size().width,
                      this.size().height) ;
     off4Gg = offImg4.getGraphics() ;

     setLayout(new GridLayout(1,2,5,5)) ;

     solve.setDefaults () ;
 
     view1  = new View1(this) ;
     inp = new Inp(this) ;

     add(view1) ;
     add(inp) ;

     solve.getFreeStream ();
     computeFlow () ;
     view1.start() ;
  }
 
  public void computeFlow() { 

     if (flflag == 1) {
         solve.getCirc ();                   /* get circulation */
         solve.genFlow () ;
         solve.getFreeStream () ;
     }
 
     loadOut() ;

     view1.loadPlot() ;
  }

  public int filter0(double inumbr) {
        //  output only to .
       int number ;
       int intermed ;
 
       number = (int) (inumbr);
       return number ;
  }

  public float filter1(double inumbr) {
     //  output only to .1
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 10.) ;
       number = (float) (intermed / 10. );
       return number ;
  }
 
  public float filter2(double inumbr) {
     //  output only to .01
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 100.) ;
       number = (float) (intermed / 100. );
       return number ;
  }
 
  public float filter3(double inumbr) {
     //  output only to .001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 1000.) ;
       number = (float) (intermed / 1000. );
       return number ;
  }
 
  public float filter5(double inumbr) {
     //  output only to .00001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 100000.) ;
       number = (float) (intermed / 100000. );
       return number ;
  }
 
  public void setUnits() {   // Switching Units
       double ovs,chords,spans,aros,chos,spos ;
       double alts,ares ;

       alts = alt / lconv ;
       chords = chord / lconv ;
       spans = span / lconv ;
       ares = area /lconv/lconv ;
       aros = arold /lconv/lconv ;
       chos = chrdold / lconv ;
       spos = spnold / lconv ;
       ovs = vfsd / vconv ;

       switch (lunits) {
          case 0: {                             /* English */
            lconv = 1.;                      /*  feet    */
            vconv = .6818; vmaxa = 100.; vmaxb = 100. ;  /*  mph  */
            fconv = 1.0; fmax = 100000.; fmaxb = .5;  /* pounds   */
            pconv = 14.7  ;                   /* lb/sq in */
            break;
          }
          case 1: {                             /* Metric */
            lconv = .3048;                    /* meters */
            vconv = 1.097;  vmaxa = 167.; vmaxb = 167.;   /* km/hr  */
            fconv = 4.448 ; fmax = 500000.; fmaxb = 2.5; /* newtons */
            pconv = 101.3 ;               /* kilo-pascals */
            break ;
          }
       }
 
       alt = alts * lconv ;
       chord = chords * lconv ;
       span = spans * lconv ;
       area = ares * lconv * lconv ;
       arold = aros * lconv * lconv ;
       chrdold = chos * lconv ;
       spnold = spos * lconv ;
       vfsd  = ovs * vconv;

       return ;
  }

  public void loadProb() {   // load the input panel
       int i1,i2,i3,i4,i5,i6,i7 ;
       double v1,v2,v3,v4,v5,v6,v7 ;
       float fl1,fl2,fl3,fl4,fl5,fl6,fl7 ;
       String outpres,outtemp,outhum,outht ;
       String outspd,outspin,outang ;

       outpres = " in Hg" ;
       outtemp = " deg F" ;
       outhum = " %" ;
       outht = " feet" ;
       outspd = " mph" ;
       outspin = " rpm" ;
       outang = " deg" ;

       v1 = 0.0 ;
       v2 = 0.0 ;
       v3 = 0.0 ;
       v4 = 0.0 ;
       v5 = 0.0 ;
       v6 = 0.0 ;
       v7 = 0.0 ;

       switch (lunits)  {
          case 0: {                             /* English */
              inp.mdl.mdll.o1.setText(String.valueOf(
                                          filter3(presm)) + outpres) ;
              inp.mdl.mdll.o2.setText(String.valueOf(
                                         filter0(ts0 - 460.)) + outtemp) ;
              inp.mdl.mdll.o3.setText(String.valueOf(
                                          filter0(rlhum * 100.)) + outhum) ;
              inp.mdl.mdll.o4.setText(String.valueOf(
                                         filter0(alt)) + outht) ;
              break;
           }
           case 1: {                             /* Metric */
              inp.mdl.mdll.o1.setText(String.valueOf(
                                          filter3(101.3/14.7*ps0/144.))) ;
              inp.mdl.mdll.o2.setText(String.valueOf(
                                         filter0(ts0*5.0/9.0 - 273.1))) ;
              inp.mdl.mdll.o3.setText(String.valueOf(
                                          filter0(rlhum*100.))) ;
              inp.mdl.mdll.o4.setText(String.valueOf(
                                         filter0(alt))) ;
              break ;
           }
       }

       v1 = vfsd ;
       vmn1 = 5.0;   vmx1= 75. ;
       v2 = spin * 60.0 ;
       vmn2 = -700.0; vmx2 = 700.0 ;
       v3 = relsy * lconv / relsdr ;
       vmn3 = -120.0 * lconv;   vmx3 = 120.0 * lconv ;
       v4 = yangr ;
       vmn4 = -135.0;   vmx4 = 135.0 ;
       v5 = relsx * lconv ;
       vmn5 = 0.0;   vmx5 = 100.0 * lconv ;
       v6 = zangr ;
       vmn6 = 0.0;   vmx6 = 60.0 ;
       v7 = sangr ;
       vmn7 = -90.0;   vmx7 = 90.0 ;

       fl1 = (float) v1 ;
       fl2 = (float) v2 ;
       fl3 = (float) v3 ;
       fl4 = (float) v4 ;
       fl5 = (float) v5 ;
       fl6 = (float) v6 ;
       fl7 = (float) v7 ;

       inp.m.lol1.f1.setText(String.valueOf(filter0(v1))) ;
       inp.m.lol1.f2.setText(String.valueOf(filter0(v2))) ;
       inp.m.lol1.f7.setText(String.valueOf(filter2(v7))) ;

       inp.l.lor1.f6.setText(String.valueOf(filter2(v6))) ;
       inp.l.lor1.f3.setText(String.valueOf(filter3(v3))) ;
       inp.l.lor1.f4.setText(String.valueOf(filter2(v4))) ;
       inp.l.lor1.f5.setText(String.valueOf(filter2(v5))) ;

       i1 = (int) (((v1 - vmn1)/(vmx1-vmn1))*1000.) ;
       i2 = (int) (((v2 - vmn2)/(vmx2-vmn2))*1000.) ;
       i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;
       i4 = (int) (((v4 - vmn4)/(vmx4-vmn4))*1000.) ;
       i5 = (int) (((v5 - vmn5)/(vmx5-vmn5))*1000.) ;
       i6 = (int) (((v6 - vmn6)/(vmx6-vmn6))*1000.) ;
       i7 = (int) (((v7 - vmn7)/(vmx7-vmn7))*1000.) ;

       inp.m.lol2.s1.setValue(i1) ;
       inp.m.lol2.s2.setValue(i2) ;
       inp.l.lor2.s3.setValue(i3) ;
       inp.l.lor2.s4.setValue(i4) ;
       inp.l.lor2.s5.setValue(i5) ;
       inp.l.lor2.s6.setValue(i6) ;
       inp.m.lol2.s7.setValue(i7) ;

       computeFlow() ;
       return ;
  }

  public void loadOut() {   // output routine
      String outpres,outtemp,outhum,outht ;

      outpres = " in Hg" ;
      outtemp = " deg F" ;
      outhum = " %" ;
      outht = " feet" ;

      if (conflag == 1) {
        switch (lunits)  {
          case 0: {                             /* English */
              inp.mdl.mdll.o1.setText(String.valueOf(
                                          filter3(presm)) + outpres) ;
              inp.mdl.mdll.o2.setText(String.valueOf(
                                          filter0(ts0 - 460.)) + outtemp) ;
              inp.mdl.mdll.o3.setText(String.valueOf(
                                          filter0(rlhum * 100.)) + outhum) ;
              inp.mdl.mdll.o4.setText(String.valueOf(
                                          filter0(alt)) + outht) ;
              break;
           }
           case 1: {                             /* Metric */
              inp.mdl.mdll.o1.setText(String.valueOf(
                                          filter3(101.3/14.7*ps0/144.))) ;
              inp.mdl.mdll.o2.setText(String.valueOf(
                                          filter0(ts0*5.0/9.0 - 273.1))) ;
              inp.mdl.mdll.o3.setText(String.valueOf(
                                          filter0(rlhum*100.))) ;
              inp.mdl.mdll.o4.setText(String.valueOf(
                                          filter0(alt))) ;
              break ;
           }
        }
     }
     return ;
  }

  class Solver {
 
     Solver () {
     }

     public void setDefaults() {

        planet = 0 ;
        wtrat = 1.0 ;
        wthopt = 0 ;
        place = 0 ;
        lunits = 0 ;
        lftout = 0 ;
        nlnc = 15 ;
        nln2 = nlnc/2 + 1 ;
        nptc = 37 ;
        npt2 = nptc/2 + 1 ;
        deltb = .5 ;
        foil = 0 ;
        flflag = 1;
        thkval = .5 ;
        thkinpt = 12.5 ;                   /* MODS 10 SEP 99 */
        camval = 0.0 ;
        caminpt = 0.0 ;
        alfval = 0.0 ;
        gamval = 0.0 ;
        spin = 5. ;
        spindr = -1.0 ;
        rval = 1.0 ;
        ycval = 0.0 ;
        xcval = 0.0 ;
        conflag = 2 ;                             /* MODS  2 Apr 99 */
        displ = 0 ;                              /* MODS  22 Apr 99 */
        relsy = 0.0 ;
        relsz = 0.0 ;
        relsx = 36.0 ;
        relsdr = 1.0 ;
        yangr = 0.0 ;
        zangr = 10.0 ;
        sangr = 0.0 ;
 
        xpval = 2.1;
        ypval = -.5 ;
        pboflag = 0 ;
        xflow = -10.0;                             /* MODS  20 Jul 99 */

        pconv = 14.7;
        pmin = .5 ;
        pmax = 1.0 ;
        fconv = 1.0 ;
        fmax = 100000. ;
        fmaxb = .50 ;
        vconv = .6818 ;
        vfsd = 50. ;
        vmaxa = 100. ;
        vmaxb = 100. ;
        lconv = 1.0 ;
//                        Soccer defaults    
        wball = 1.0 ;             // weight of soccer ball = 16 oz
        rball = .375 ;            // soccer radius = 4.5 inch = .375 feet
        cdcalc = 1 ;              // cd and cl are defaulted
        cd = .25 ;
        clift = .25 ;

        area = 3.14159 * rball * rball ;

        drawview = 1 ; 
        ktype = 1;                // penalty kick
        dist = 0.;               // distance from ball to goal line
        sldloc = 250;
        fact = 50. ;

        nsav = 0 ;
        pitch = 0 ;
        comp = 0 ;
        call = 0 ;
        zacc = -32.2 ; zvel = 0.0; zloc = relsz;
        yacc = 0.0;   yvel = 0.0; yloc = relsy;
        xacc = 0.0;   xvel = -vfsd; xloc = relsx ;

        alt = 700.0 ;
        altmax = 50000. ;
        chrdold = chord = 2.0 ;
        spnold = span = 10.0 ;
        aspr = 5.0 ;
        armax = 1000.01 ;
        armin = .01 ;                 /* MODS 9 SEP 99 */
        presm = 29.172 ;
        temf  = 60.0 ;
        ts0 = temf + 459.6 ;
        rlhum = 0.0 ;

        xtt = 400 ;
        ytt = 300 ;
        xts = 400 ;
        yts = 430 ;
        xtk = 250 ;
        ytk = 400 ;
        xtv = 400 ;
        ytv = 430 ;

        pltxg[1] = 0.0 ;
        pltyg[1] = 1.0 ;
        plthg1[1] = 0.0 ;
        plthg2[1] = 0.0 ;
        pltrg[1] = 1.0 ;
        begy= 0.0 ; endy=65. ; 
        begx= 0.0 ; endx=65. ; 
        begz= 0.0 ; endz=65. ; 
        nptb = 1;
 
        vmn1 = 5.0;     vmx1 = 75.0 ;
        vmn2 = -700.0;   vmx2 = 700.0 ;
        vmn3 = -120.0;   vmx3 = 120.0 ;
        vmn4 = -135.0;   vmx4 = 135.0 ;
        vmn5 = 0.0;   vmx5 = 100.0 ;
        vmn6 = 0.0;   vmx6 = 60.0 ;
        vmn7 = -90.0;   vmx7 = 90.0 ;

        return ;
     }

     public void getFreeStream() {    //  free stream conditions
       double hite,pvap,rgas,gama ;       /* MODS  19 Jan 00  whole routine*/

       rgas = 1718. ;                /* ft2/sec2 R */
       gama = 1.4 ;
       hite = alt/lconv ;
       if (planet == 0) {    // Earth  standard day
         if (conflag == 1) {
           if (hite <= 36152.) {           // Troposphere
              ts0 = 518.6 - 3.56 * hite/1000. ;
              ps0 = 2116. * Math.pow(ts0/518.6,5.256) ;
           }
           if (hite >= 36152. && hite <= 82345.) {   // Stratosphere
              ts0 = 389.98 ;
              ps0 = 2116. * .2236 *
                 Math.exp((36000.-hite)/(53.35*389.98)) ;
           }
           if (hite >= 82345.) {
              ts0 = 389.98 + 1.645 * (hite-82345)/1000. ;
              ps0 = 2116. *.02456 * Math.pow(ts0/389.98,-11.388) ;
           }
           if (place == 0) {
             if(wthopt == 0) ts0 = 519.6 ;
             if(wthopt == 1) ts0 = 549.6 ;
             if(wthopt == 2) ts0 = 494.6 ;
             if(wthopt == 3) ts0 = 549.6 ;
           }
           if (wthopt <= 2) {
               rlhum = 0.0 ;
           }
           if (wthopt == 3) {
               rlhum = 1.0 ;
           }
           temf = ts0 - 459.6 ;
           if (temf <= 0.0) temf = 0.0 ;                    
           presm = ps0 * 29.92 / 2116. ;
         }
         if (conflag == 2) {
            ts0 = temf + 459.6 ;
            if (temf < 0.0) {
                  temf = 0.0 ;
                  rlhum = 0.0 ;
            }
             ps0 = presm * 2116. / 29.92 ;
         }
         pvap = rlhum*(2.685+.00353*Math.pow(temf,2.245));/* Eq 1:6A  Domasch */
         rho = (ps0 - .379*pvap)/(rgas * ts0) ;  /* effect of humidty */
       }

       if (planet == 1) {   // Mars - curve fit of orbiter data
         rgas = 1149. ;                /* ft2/sec2 R */
         gama = 1.29 ;

         if (hite <= 22960.) {
            ts0 = 434.02 - .548 * hite/1000. ;
            ps0 = 14.62 * Math.pow(2.71828,-.00003 * hite) ;
         }
         if (hite > 22960.) {
            ts0 = 449.36 - 1.217 * hite/1000. ;
            ps0 = 14.62 * Math.pow(2.71828,-.00003 * hite) ;
         }
         rho = ps0/(rgas*ts0) ;
         presm = ps0 * 29.92 / 2116. ;
       }

       q0  = .5 * rho * vfsd * vfsd / (vconv * vconv) ;
       pt0 = ps0 + q0 ;

       return ;
     }

     public void getCirc() {   // circulation from Kutta condition
       double thet,rdm,thtm ;
       double beta;
       int index;

       xcval = 0.0 ;
       switch (foil)  {
          case 0: {         /* get circulation from spin for baseball */
              gamval = 4.0 * 3.1415926 * 3.1415926 *spin * rball * rball
                                 / (vfsd/vconv) ;
              gamval = gamval * spindr ;
              ycval = .0001 ;
              break ;
          }
          case 1:  {                  /* Juokowski geometry*/
              ycval = camval / 2.0 ;
              rval = thkval/4.0 +Math.sqrt(thkval*thkval/16.0+ycval*ycval +1.0);
              xcval = 1.0 - Math.sqrt(rval*rval - ycval*ycval) ;
              beta = Math.asin(ycval/rval)/convdr ;     /* Kutta condition */
              gamval = 2.0*rval*Math.sin((alfval+beta)*convdr) ;
              break ;
          }
          case 2:  {                  /* Elliptical geometry*/
              ycval = camval / 2.0 ;
              rval = thkval/4.0 + Math.sqrt(thkval*thkval/16.0+ycval*ycval+1.0);
              beta = Math.asin(ycval/rval)/convdr ;    /* Kutta condition */
              gamval = 2.0*rval*Math.sin((alfval+beta)*convdr) ;
              break ;
          }
       }
                                                   /* geometry */
       for (index =1; index <= nptc; ++index) {
           thet = (index -1)*360./(nptc-1) ;
           xg[0][index] = rval * Math.cos(convdr * thet) + xcval ;
           yg[0][index] = rval * Math.sin(convdr * thet) + ycval ;
           rg[0][index] = Math.sqrt(xg[0][index]*xg[0][index] +
                                    yg[0][index]*yg[0][index])  ;
           thg[0][index] = Math.atan2(yg[0][index],xg[0][index])/convdr;
           xm[0][index] = (rg[0][index] + 1.0/rg[0][index])*
                        Math.cos(convdr*thg[0][index]) ;
           ym[0][index] = (rg[0][index] - 1.0/rg[0][index])*
                        Math.sin(convdr*thg[0][index]) ;
           rdm = Math.sqrt(xm[0][index]*xm[0][index] +
                           ym[0][index]*ym[0][index])  ;
           thtm = Math.atan2(ym[0][index],xm[0][index])/convdr;
           xm[0][index] = rdm * Math.cos((thtm - alfval)*convdr);
           ym[0][index] = rdm * Math.sin((thtm - alfval)*convdr);
           getVel(rval,thet) ;
       }

       return ;
     }

     public void genFlow() {   // generate flowfield
       double rnew,thet,psv,fxg;
       int k,index;
                              /* all lines of flow  except stagnation line*/
       for (k=1; k<=nlnc; ++k) {
         psv = -.5*(nln2-1) + .5*(k-1) ;
         fxg = xflow ;
         for (index =1; index <=nptc; ++ index) {
           solve.getPoints (fxg,psv) ;
           xg[k][index]  = lxgt ;
           yg[k][index]  = lygt ;
           rg[k][index]  = lrgt ;
           thg[k][index] = lthgt ;
           xm[k][index]  = lxmt ;
           ym[k][index]  = lymt ;
           solve.getVel(lrg,lthg) ;
           fxg = fxg + vxdir*deltb ;
         }
       }
                                              /*  stagnation line */
       k = nln2 ;
       psv = 0.0 ;
                                              /*  incoming flow */
       for (index =1; index <= npt2; ++ index) {
           rnew = 10.0 - (10.0 - rval)*Math.sin(pid2*(index-1)/(npt2-1)) ;
           thet = Math.asin(.999*(psv - gamval*Math.log(rnew/rval))/
                                   (rnew - rval*rval/rnew)) ;
           fxg =  - rnew * Math.cos(thet) ;
           solve.getPoints (fxg,psv) ;
           xg[k][index]  = lxgt ;
           yg[k][index]  = lygt ;
           rg[k][index]  = lrgt ;
           thg[k][index] = lthgt ;
           xm[k][index]  = lxmt ;
           ym[k][index]  = lymt ;
       }
                                              /*  downstream flow */
       for (index = 1; index <= npt2; ++ index) {
           rnew = 10.0 + .01 - (10.0 - rval)*Math.cos(pid2*(index-1)/(npt2-1)) ;
           thet = Math.asin(.999*(psv - gamval*Math.log(rnew/rval))/
                                      (rnew - rval*rval/rnew)) ;
           fxg =   rnew * Math.cos(thet) ;
           solve.getPoints (fxg,psv) ;
           xg[k][npt2+index]  = lxgt ;
           yg[k][npt2+index]  = lygt ;
           rg[k][npt2+index]  = lrgt ;
           thg[k][npt2+index] = lthgt ;
           xm[k][npt2+index]  = lxmt ;
           ym[k][npt2+index]  = lymt ;
       }
                                              /*  stagnation point */
       xg[k][npt2]  = xcval ;
       yg[k][npt2]  = ycval ;
       rg[k][npt2]  = Math.sqrt(xcval*xcval+ycval*ycval) ;
       thg[k][npt2] = Math.atan2(ycval,xcval)/convdr ;
       xm[k][npt2]  = (xm[k][npt2+1] + xm[k][npt2-1])/2.0 ;
       ym[k][npt2]  = (ym[0][nptc/4+1] + ym[0][nptc/4*3+1])/2.0 ;
                                /*  compute lift coefficient */
       leg = xcval - Math.sqrt(rval*rval - ycval*ycval) ;
       teg = xcval + Math.sqrt(rval*rval - ycval*ycval) ;
       lem = leg + 1.0/leg ;
       tem = teg + 1.0/teg ;
       chrd = tem - lem ;
       cl = gamval*4.0*3.1415926/chrd ;

       return ;
     }

     public void getPoints(double fxg, double psv) {   // flow in x-psi
       double radm,thetm ;                /* MODS  20 Jul 99  whole routine*/
       double fnew,ynew,yold,rfac,deriv ;
       double xold,xnew,thet ;
       double rmin,rmax ;
       int iter,isign;
                       /* get variables in the generating plane */
                           /* iterate to find value of yg */
       ynew = 10.0 ;
       yold = 10.0 ;
       if (psv < 0.0) ynew = -10.0 ;
       if (Math.abs(psv) < .001 && alfval < 0.0) ynew = rval ;
       if (Math.abs(psv) < .001 && alfval >= 0.0) ynew = -rval ;
       fnew = 0.1 ;
       iter = 1 ;
       while (Math.abs(fnew) >= .00001 && iter < 25) {
           ++iter ;
           rfac = fxg*fxg + ynew*ynew ;
           if (rfac < rval*rval) rfac = rval*rval + .01 ;
           fnew = psv - ynew*(1.0 - rval*rval/rfac)
                  - gamval*Math.log(Math.sqrt(rfac)/rval) ;
           deriv = - (1.0 - rval*rval/rfac)
               - 2.0 * ynew*ynew*rval*rval/(rfac*rfac)
               - gamval * ynew / rfac ;
           yold = ynew ;
           ynew = yold  - .5*fnew/deriv ;
       }
       lyg = yold ;
                                     /* rotate for angle of attack */
       lrg = Math.sqrt(fxg*fxg + lyg*lyg) ;
       lthg = Math.atan2(lyg,fxg)/convdr ;
       lxgt = lrg * Math.cos(convdr*(lthg + alfval)) ;
       lygt = lrg * Math.sin(convdr*(lthg + alfval)) ;
                              /* translate cylinder to generate airfoil */
       lxgt = lxgt + xcval ;
       lygt = lygt + ycval ;
       lrgt = Math.sqrt(lxgt*lxgt + lygt*lygt) ;
       lthgt = Math.atan2(lygt,lxgt)/convdr ;
                               /*  Kutta-Joukowski mapping */
       lxm = (lrgt + 1.0/lrgt)*Math.cos(convdr*lthgt) ;
       lym = (lrgt - 1.0/lrgt)*Math.sin(convdr*lthgt) ;
                              /* tranforms for view fixed with free stream */
                /* take out rotation for angle of attack mapped and cylinder */
       radm = Math.sqrt(lxm*lxm+lym*lym) ;
       thetm = Math.atan2(lym,lxm)/convdr ;
       lxmt = radm*Math.cos(convdr*(thetm-alfval)) ;
       lymt = radm*Math.sin(convdr*(thetm-alfval)) ;

       lxgt = lxgt - xcval ;
       lygt = lygt - ycval ;
       lrgt = Math.sqrt(lxgt*lxgt + lygt*lygt)  ;
       lthgt = Math.atan2(lygt,lxgt)/convdr;
       lxgt = lrgt * Math.cos((lthgt - alfval)*convdr);
       lygt = lrgt * Math.sin((lthgt - alfval)*convdr);

       return ;
     }
 
     public void getVel(double radius, double theta) {  //velocity and pressure 
      double ur,uth,jake1,jake2,jakesq ;
      double xlocv,ylocv,thrad,alfrad ;

      thrad = convdr * theta ;
      alfrad = convdr * alfval ;
                                /* get x, y location in cylinder plane */
      xlocv = radius * Math.cos(thrad) ;
      ylocv = radius * Math.sin(thrad) ;
                                /* velocity in cylinder plane */
      ur  = Math.cos(thrad-alfrad)*(1.0-(rval*rval)/(radius*radius)) ;
      uth = -Math.sin(thrad-alfrad)*(1.0+(rval*rval)/(radius*radius))
                            - gamval/radius;
      usq = ur*ur + uth*uth ;
      vxdir = ur * Math.cos(thrad) - uth * Math.sin(thrad) ; // MODS  20 Jul 99 
                                /* translate to generate airfoil  */
      xlocv = xlocv + xcval ;
      ylocv = ylocv + ycval ;
                                   /* compute new radius-theta  */
      radius = Math.sqrt(xlocv*xlocv + ylocv*ylocv) ;
      thrad  = Math.atan2(ylocv,xlocv) ;
                                   /* compute Joukowski Jacobian  */
      jake1 = 1.0 - Math.cos(2.0*thrad)/(radius*radius) ;
      jake2 = Math.sin(2.0*thrad)/(radius*radius) ;
      jakesq = jake1*jake1 + jake2*jake2 ;
      if (Math.abs(jakesq) <= .01) jakesq = .01 ;  /* protection */
      vsq = usq / jakesq ;
          /* vel is velocity ratio - pres is coefficient  (p-p0)/q0   */
      if (foil > 0) {
           vel = Math.sqrt(vsq) ;
           pres = 1.0 - vsq ;
      }
      if (foil == 0) {
           vel = Math.sqrt(usq) ;
           pres = 1.0 - usq ;
      }
      return ;
    }
  }
 
  class Inp extends Panel {
     Ball outerparent ;
     M m;
     L l;
     Mdl mdl ;
     Mdr mdr ;

     Inp (Ball target) {
          outerparent = target ;
          setLayout(new GridLayout(4,1,0,0)) ;

          m = new M(outerparent) ;
          l = new L(outerparent) ;
          mdl = new Mdl(outerparent) ;
          mdr = new Mdr(outerparent) ;

          add(mdr) ;  
          add(mdl) ;  
          add(m) ;
          add(l) ;
     }

     class Mdl extends Panel {
        Ball outerparent ;
        Mdlu mdlu ;
        Mdll mdll ;

        Mdl (Ball target) { 
          outerparent = target ;
          setLayout(new GridLayout(2,1,5,5)) ;
 
          mdll = new Mdll(outerparent) ;
          mdlu = new Mdlu(outerparent) ;

          add(mdlu) ;  
          add(mdll) ;  
        }

        class Mdlu extends Panel {
           Ball outerparent ;
           Choice placeh,weatch ;
           Label l1,l2 ;

           Mdlu (Ball target) { 
             outerparent = target ;
             setLayout(new GridLayout(2,2,0,0)) ;
    
             placeh = new Choice() ;
             placeh.setForeground(Color.red) ;
             placeh.addItem("Cleveland") ;
             placeh.addItem("Denver") ;
             placeh.addItem("New York") ;
             placeh.addItem("Chicago") ;
             placeh.addItem("Johannsburg") ;
             placeh.addItem("Cape Town") ;
             placeh.addItem("Pretoria") ;
             placeh.addItem("Polokwana");
             placeh.addItem("Rustenburg") ;
             placeh.addItem("Mangaung") ;
             placeh.addItem("Durban") ;
             placeh.addItem("Port Elizabeth") ;
             placeh.addItem("Nelspruit") ;

             placeh.addItem("Mt.Everest") ;
             placeh.addItem("Mars");
             placeh.select(0) ;
 
             weatch = new Choice() ;
             weatch.setForeground(Color.red) ;
             weatch.addItem("Average Day") ;
             weatch.addItem("Hot Day") ;
             weatch.addItem("Cold Day") ;
             weatch.addItem("Hot, Humid Day") ;

             l1 = new Label("Set Stadium Location:", Label.RIGHT) ;
             l1.setForeground(Color.red) ;

             l2 = new Label("Set Weather Conditions:", Label.RIGHT) ;
             l2.setForeground(Color.red) ;

             add(l1) ;
             add(placeh) ;

             add(l2) ;
             add(weatch) ;
           }

           public boolean action(Event evt, Object arg) {
              if(evt.target instanceof Choice) {
                  this.handleProb(arg) ;
                  return true ;
              }
              else return false ;
           }

           public void handleProb(Object obj) {
              int i2,i1 ;
    
              planet  = 0 ;
              wthopt  = weatch.getSelectedIndex() ;
              i1 = 0 ;
              i2  = placeh.getSelectedIndex() ;
              switch (i2) {
                 case 0:{                            /* Cleveland */
                        place = 0 ;
                        alt = 700.0 * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 1:{                            /* Denver */
                        place = 0 ;
                        alt = 5280.0 * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 2:{                            /* New York */
                        place = 0 ;
                        alt = 0. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 3:{                            /* Chicago */
                        place = 0 ;
                        alt = 568. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 4:{                            /* Johannesburg */
                        place = 0 ;
                        alt = 5558. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 5:{                            /* Cape Town*/
                        place = 0 ;
                        alt = 138. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 6:{                            /* Pretoria */
                        place = 0 ;
                        alt = 4364. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 7:{                            /* Polokwana */
                        place = 0 ;
                        alt = 4035. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 8:{                            /* Rustenburg */
                        place = 0 ;
                        alt = 3783. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 9:{                            /* Mangaung */
                        place = 0 ;
                        alt = 4432. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 10:{                            /* Durban*/
                        place = 0 ;
                        alt = 26. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 11:{                            /* Port Elizabeth */
                        place = 0 ;
                        alt = 197. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 12:{                            /* Nelspruit */
                        place = 0 ;
                        alt = 2165. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }

                 case 13:{                        /* Mt. Everest */
                        place = 1 ;
                        alt = 29028. * lconv ;
                        wtrat = 1.0 ;
                        break ;
                 }
                 case 14:{                          /* Mars */
                        planet = 1 ;
                        alt = 0.0 * lconv ;
                        wtrat = .333 ;
                        break ;
                 }
              }

              switch (i1) {
                 case 0:{                            /* Standard Day */
                        conflag = 1 ;
                        break ;
                 }
                 case 1:{                            /* Set Conditions */
                        conflag = 2 ;
                        break ;
                 }
              }

              computeFlow() ;
           }
        }

        class Mdll extends Panel {
           Ball outerparent ;
           TextField o1,o2,o3,o4 ;
           Label l1,l2,l3,l4 ;

           Mdll (Ball target) { 
             outerparent = target ;
             setLayout(new GridLayout(3,4,0,0)) ;
 
             l1 = new Label("Pressure", Label.CENTER) ;
             o1 = new TextField("29.172 in Hg",5) ;
             l2 = new Label("Temperature", Label.CENTER) ;
             o2 = new TextField("59 deg F",5) ;
             l3 = new Label("Humidity", Label.CENTER) ;
             o3 = new TextField("0.0  %",5) ;
             l4 = new Label("Altitude", Label.CENTER) ;
             o4 = new TextField("700.0  feet",5) ;
             add(l1) ;
             add(o1) ;
             add(l3) ;
             add(o3) ;
    
             add(l2) ;
             add(o2) ;
             add(l4) ;
             add(o4) ;
    
             add(new Label(" ", Label.RIGHT)) ;
             add(new Label(" ", Label.RIGHT)) ;
             add(new Label(" ", Label.RIGHT)) ;
             add(new Label(" ", Label.RIGHT)) ;
           }

           public boolean handleEvent(Event evt) {
             if(evt.id == Event.ACTION_EVENT) {
                this.handleText(evt) ;
                return true ;
             }
             else return false ;
           }
 
           public void handleText(Event evt) {
              Double V1,V2,V3 ;
              double v1,v2,v3 ;
              float fl1 ;

              V1 = Double.valueOf(o1.getText()) ;
              v1 = V1.doubleValue() ;
              V2 = Double.valueOf(o2.getText()) ;
              v2 = V2.doubleValue() ;
              V3 = Double.valueOf(o3.getText()) ;
              v3 = V3.doubleValue() ;
 
              if (conflag == 2) {
                 if(v1 < 5.0) {
                    v1 = 5.0 ;
                    fl1 = (float) v1 ;
                    o1.setText(String.valueOf(fl1)) ;
                 }
                 if(v1 > 50.0) {
                    v1 = 50.0 ;
                    fl1 = (float) v1 ;
                    o1.setText(String.valueOf(fl1)) ;
                 }
                 presm = v1 ;
                 temf = v2 ;
                 if(v3 < 1.0) {
                    v3 = 0.0 ;
                    fl1 = (float) v3 ;
                    o3.setText(String.valueOf(fl1)) ;
                 }
                 if(v3 > 100.0) {
                    v3 = 100.0 ;
                    fl1 = (float) v3 ;
                    o3.setText(String.valueOf(fl1)) ;
                 }
                 rlhum = v3 /100. ;
                 computeFlow() ;
              }
           }
        }
     }

     class Mdr extends Panel {
        Ball outerparent ;
        Button ptchbt,butclr,butsav,resbt ;
        Checkbox ch1,ch2,cbpen,cbcorn,cbfree,chin,chcal ;
        CheckboxGroup cbg,cbtyp,cbdrag ;
        TextField o1,o2,o3,o4,o5,o6 ;
        Label l1,l2,l3,l4,l5,l6 ;
        Label lu4,la1,la2,latyp ;

        Mdr (Ball target) { 
          int i1 ;

          outerparent = target ;
          setLayout(new GridLayout(5,6,5,5)) ;
 
          la1 = new Label("Results", Label.CENTER) ;
          la1.setForeground(Color.blue) ;

          ptchbt = new Button("KICK") ;
          ptchbt.setBackground(Color.green) ;
          ptchbt.setForeground(Color.black) ;

          butclr = new Button("Clear") ;
          butclr.setBackground(Color.white) ;
          butclr.setForeground(Color.blue) ;

          butsav = new Button("Save") ;
          butsav.setBackground(Color.blue) ;
          butsav.setForeground(Color.white) ;

          resbt = new Button("Reset") ;
          resbt.setBackground(Color.orange) ;
          resbt.setForeground(Color.black) ;

          cbg = new CheckboxGroup() ;
          ch1 = new Checkbox("Right Foot",cbg,true) ;
          ch2 = new Checkbox("Left Foot",cbg,false) ;

          cbdrag = new CheckboxGroup() ;
          chcal = new Checkbox("Default",cbdrag,true) ;
          chin = new Checkbox("Input",cbdrag,false) ;

          latyp = new Label("Type:", Label.RIGHT) ;
          latyp.setForeground(Color.blue) ;

          cbtyp = new CheckboxGroup() ;
          cbpen = new Checkbox("Penalty",cbtyp,true) ;
          cbfree = new Checkbox("Free",cbtyp,false) ;
          cbcorn = new Checkbox("Corner",cbtyp,false) ;

          l1 = new Label("Speed", Label.CENTER) ;
          o1 = new TextField("0.0 mph ",5) ;
          l2 = new Label("Cd ", Label.CENTER) ;
          o2 = new TextField(".25 ",5) ;
 
          l3 = new Label("Time", Label.CENTER) ;
          o3 = new TextField("0.0 sec ",5) ;
          l4 = new Label("Cl", Label.CENTER) ;
          o4 = new TextField(".25",5) ;
 
          l5 = new Label("Side Force", Label.CENTER) ;
          o5 = new TextField("0.0 oz ",5) ;
          l6 = new Label("Drag", Label.CENTER) ;
          o6 = new TextField("0 oz",5) ;

          add(new Label(" ", Label.CENTER)) ;
          add(ptchbt) ; 
          add(butsav) ;   
          add(butclr) ;
          add(new Label(" ", Label.CENTER)) ;
          add(resbt) ;

          add(latyp) ;
          add(cbpen) ;
          add(cbfree) ;
          add(cbcorn) ;
          add(new Label(" ", Label.CENTER)) ;
          add(new Label(" ", Label.CENTER)) ;

          add(ch2) ;
          add(ch1) ;
          add(new Label(" ", Label.CENTER)) ;
          add(new Label("Cl and Cd: ", Label.RIGHT)) ;
          add(chcal) ;
          add(chin) ;

          add(l3) ;
          add(o3) ;
          add(l6) ;
          add(o6) ;
          add(l2) ;
          add(o2) ;

          add(l1) ;
          add(o1) ;
          add(l5) ;
          add(o5) ;
          add(l4) ;
          add(o4) ;
        }
 
        public boolean action(Event evt, Object arg) {
           String outfor,outtim,outspd ;
           Double V2,V1 ;
           double v2,v1 ;
           int i,j ;

           outspd = " mph" ;
           outfor = " oz" ;
           outtim = " sec" ;

           if(evt.target instanceof Checkbox) {
               this.handleCbox(evt) ;
               return true ;
           }

           if(evt.target instanceof Button) {
               String label = (String)arg ;
               if(label.equals("KICK")){
                     nptb = 1 ;
                     pitch = 1 ;
                     comp = 1 ;
       // magnitude of forces
                     side = gamval * rho * vfsd*vfsd/vconv/vconv; /* lift/ft */
                     side = side * rball * 4.0/3.0  ; // lift integration 

                     if (cdcalc == 0) {   // read in cd value
                          V2 = Double.valueOf(o2.getText()) ;
                          v2 = V2.doubleValue() ; 
                          if(v2 < .1) v2 = .1 ;
                          if(v2 > .8) v2 = .8 ;
                          cd = v2 ;
                     }
                     else  cd = .25 ;

                     if (cdcalc == 0) {   // read in cl value
                          V1 = Double.valueOf(o4.getText()) ;
                          v1 = V1.doubleValue() ; 
                          if(v1 < .1) v1 = .1 ;
                          if(v1 > .8) v1 = .8 ;
                          clift = v1 ;
                     }
                     else  clift = .25 ;

                     drag = cd * .5 * rho * area * vfsd * vfsd/vconv/vconv;
                     side = clift * side  ;
       // forces
                     xfor = side*Math.cos(sangr*convdr)*Math.sin(yangr*convdr) 
                        - drag*Math.cos(zangr*convdr)*Math.cos(yangr*convdr) ;
                     yfor = -side*Math.cos(sangr*convdr)*Math.cos(yangr*convdr)
                        - drag*Math.cos(zangr*convdr)*Math.sin(yangr*convdr)  ;
                     zfor = -wball - side*Math.sin(sangr*convdr) - drag*Math.sin(zangr*convdr) ;

                     xloc = relsx ;
                     xvel = -vfsd * Math.cos(convdr*yangr) * Math.cos(convdr*zangr) / vconv ;
                     xacc =  xfor / wball * 32.2 ; 

                     yloc = relsy ;
                     yvel = vfsd * Math.cos(convdr*zangr) * Math.sin(convdr*yangr) / vconv ;
                     yacc =  yfor / wball * 32.2 ;

                     zloc = relsz ;
                     zvel = vfsd * Math.sin(convdr*zangr) / vconv ;
                     zacc =  zfor / wball * 32.2  ;

                     call = 0 ;
                     npt = 0 ;
                     ptim = 0.0 ;
                     posx[nsav][nptb] = xloc ;
                     posy[nsav][nptb] = yloc ;
                     posz[nsav][nptb] = zloc ;
                     posx[nsav][0] = xloc ;
                     posy[nsav][0] = yloc ;
                     posz[nsav][0] = zloc ;
               }

               if(label.equals("Clear")) {
                 nptb = 1 ;
                 pitch = 0 ;
                 comp = 0 ;
                 nsav = 0 ;
                 for (j=0; j<=4; ++j) {
                   for (i=0; i<=npts[j]; ++i){
                     posx[j][i] = 0.0 ;
                     posy[j][i] = 0.0 ;
                     posz[j][i] = 0.0 ;
                   }
                   npts[j] = 0 ;
                 }
                o5.setText(String.valueOf(filter3(0.0)) + outfor) ;
                o6.setText(String.valueOf(filter3(0.0)) + outfor) ;
                o3.setText(String.valueOf(filter2(0.0)) + outtim) ;
                o1.setText(String.valueOf(filter0(0.0)) + outspd) ;
                o2.setText(String.valueOf(filter3(cd))) ;
                o4.setText(String.valueOf(filter3(clift))) ;
               }
               if(label.equals("Save")) {
                  pitch = 0 ;
                  comp = 0 ;
                  npts[nsav] = npt ;
                  nsav = nsav + 1 ;
                  if (nsav == 5) nsav = 4 ;
                  for (i=0; i<=npts[nsav]; ++i){
                      posx[nsav][i] = 0.0 ;
                      posy[nsav][i] = 0.0 ;
                      posz[nsav][i] = 0.0 ;
                   }
                   npts[nsav] = 0 ;
               }
               if(label.equals("Reset")) {
                  solve.setDefaults() ;
                  inp.mdr.ch1.setState(true) ;
                  inp.mdr.ch2.setState(false) ;
                  inp.mdr.chcal.setState(true) ;
                  inp.mdr.chin.setState(false) ;
                  inp.mdr.cbpen.setState(true) ;
                  inp.mdr.cbcorn.setState(false) ;
                  inp.mdr.cbfree.setState(false) ;
                  inp.mdl.mdlu.placeh.select(0) ;
                  inp.mdl.mdlu.weatch.select(0) ;
                  pitch = 0 ;
                  comp = 0 ;
                  nsav = 0 ;
                  for (j=0; j<=4; ++j) {
                      for (i=0; i<=npts[j]; ++i){
                         posx[j][i] = 0.0 ;
                         posy[j][i] = 0.0 ;
                         posz[j][i] = 0.0 ;
                      }
                      npts[j] = 0 ;
                  }
                  o5.setText(String.valueOf(filter3(0.0)) + outfor) ;
                  o6.setText(String.valueOf(filter3(0.0)) + outfor) ;
                  o3.setText(String.valueOf(filter2(0.0)) + outtim) ;
                  o1.setText(String.valueOf(filter0(0.0)) + outspd) ;
                  o2.setText(String.valueOf(filter3(cd))) ;
                  o4.setText(String.valueOf(filter3(clift))) ;

                  loadProb() ;
               }

               computeFlow() ;
               return true ;
            }
            else return false ;
        } 

        public void handleCbox(Object obj) {
            int i3,i5 ;
            Double V3,V5 ;
            double v3,v5 ;
            double fl1 ;
            String outht ;

            outht = " feet" ;

            if (ch1.getState()) {    // righty
                 spindr = -1.0 ;
                 relsdr = 1.0 ;
            }
            if (ch2.getState()) {    // lefty
                 spindr = 1.0 ;
                 relsdr = -1.0 ;
            }

            if (chcal.getState()) {    // default cd and cl
                 cdcalc = 1 ;
            }
            if (chin.getState()) {    // input cd and cl
                 cdcalc = 0 ;
            }

            v3 = 0.0 ;
            v5 = 36.0 ;
            if (cbpen.getState()) {    // penalty kick
                 ktype = 1 ;
                 relsy = v3 = 0.0 ;
                 relsx = v5 = 36.0 ;
            }
            if (cbfree.getState()) {    // free kick
                 ktype = 2 ;
                 relsy = v3 = 30.0 ;
                 relsx = v5 = 60.0 ;
            }
            if (cbcorn.getState()) {    // corner kick
                 ktype = 3 ;
                 relsy = v3 = -120.0 ;
                 relsx = v5 = 0.0 ;
            }
            dist = 0.0 ;
            fl1 = (float) v3 ;
            l.lor1.f3.setText(String.valueOf(fl1)) ;
            i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;
            l.lor2.s3.setValue(i3) ;
            fl1 = (float) v5 ;
            l.lor1.f5.setText(String.valueOf(fl1)) ;
            i5 = (int) (((v5 - vmn5)/(vmx5-vmn5))*1000.) ;
            l.lor2.s5.setValue(i5) ;

            computeFlow() ;
        }
     }

  class M extends Panel {
     Ball outerparent ;
     Lol1 lol1 ;
     Lol2 lol2 ;
 
     M (Ball target) { 
        outerparent = target ;
        setLayout(new GridLayout(1,2,5,5)) ;

        lol1 = new Lol1(outerparent) ;
        lol2 = new Lol2(outerparent) ;

        add(lol1) ;  
        add(lol2) ;  
     }
     class Lol1 extends Panel {
        Ball outerparent ;
        TextField f1,f2,f7 ;
        Label l1,l2,l7,lab1 ;

        Lol1 (Ball target) {

         outerparent = target ;
         setLayout(new GridLayout(5,2,5,5)) ;

         lab1 = new Label("Kick Input", Label.CENTER) ;
         lab1.setForeground(Color.blue) ;

         l1 = new Label("Speed - mph ", Label.CENTER) ;
         f1 = new TextField("50.0",5) ;

         l2 = new Label("Spin - rpm ", Label.CENTER) ;
         f2 = new TextField("300.0",5) ;

         l7 = new Label("Spin Axis - deg", Label.CENTER) ;
         f7 = new TextField("0.0",5) ;

         add(lab1) ;
         add(new Label(" ", Label.CENTER)) ;

         add(l1) ;
         add(f1) ;

         add(l2) ;
         add(f2) ;

         add(l7) ;
         add(f7) ;

         add(new Label(" ", Label.CENTER)) ;
         add(new Label(" ", Label.CENTER)) ;
       }

        public boolean handleEvent(Event evt) {
         if(evt.id == Event.ACTION_EVENT) {
            this.handleText(evt) ;
            return true ;
         }
         else return false ;
       }
 
       public void handleText(Event evt) {
          Double V1,V2,V7 ;
          double v1,v2,v7 ;
          float fl1 ;
          int i1,i2,i7 ;

          V1 = Double.valueOf(f1.getText()) ;
          v1 = V1.doubleValue() ;
          V2 = Double.valueOf(f2.getText()) ;
          v2 = V2.doubleValue() ;
          V7 = Double.valueOf(f7.getText()) ;
          v7 = V7.doubleValue() ;
 
          vfsd = v1 ;
          if(v1 < vmn1) {
             vfsd = v1 = vmn1 ;
             fl1 = (float) v1 ;
             f1.setText(String.valueOf(fl1)) ;
          }
          if(v1 > vmx1) {
             vfsd = v1 = vmx1 ;
             fl1 = (float) v1 ;
             f1.setText(String.valueOf(fl1)) ;
          }

          if(v2 < vmn2) {
             v2 = vmn2 ;
             fl1 = (float) v2 ;
             f2.setText(String.valueOf(fl1)) ;
          }
          if(v2 > vmx2) {
             v2 = vmx2 ;
             fl1 = (float) v2 ;
             f2.setText(String.valueOf(fl1)) ;
          }
          spin = v2 ;
          spin = spin/60. ;

          sangr = v7 ;
          if(v7 < vmn7) {
             sangr = v7 = vmn7 ;
             fl1 = (float) v7 ;
             f7.setText(String.valueOf(fl1)) ;
          }
          if(v7 > vmx7) {
             sangr = v7 = vmx7 ;
             fl1 = (float) v7 ;
             f7.setText(String.valueOf(fl1)) ;
          }

          i1 = (int) (((v1 - vmn1)/(vmx1-vmn1))*1000.) ;
          i2 = (int) (((v2 - vmn2)/(vmx2-vmn2))*1000.) ;
          i7 = (int) (((v7 - vmn7)/(vmx7-vmn7))*1000.) ;

          lol2.s1.setValue(i1) ;
          lol2.s2.setValue(i2) ;
          lol2.s7.setValue(i7) ;
 
          computeFlow() ;
       }
     }

     class Lol2 extends Panel {
        Ball outerparent ;
        Scrollbar s1,s2,s7;
        Label lab1;

        Lol2 (Ball target) {
         int i1,i2,i7 ;

         outerparent = target ;
         setLayout(new GridLayout(5,1,5,5)) ;

         lab1 = new Label("Set Conditions:", Label.CENTER) ;
         lab1.setForeground(Color.red) ;

         i1 = (int) (((50.0 - vmn1)/(vmx1-vmn1))*1000.) ;
         i2 = (int) (((300.0 - vmn2)/(vmx2-vmn2))*1000.) ;
         i7 = (int) (((0.0 - vmn7)/(vmx7-vmn7))*1000.) ;

         s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
         s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);
         s7 = new Scrollbar(Scrollbar.HORIZONTAL,i7,10,0,1000);

         add(lab1) ;
         add(s1) ;
         add(s2) ;
         add(s7) ;
         add(new Label(" ", Label.CENTER)) ;
       }
 
       public boolean handleEvent(Event evt) {
            if(evt.id == Event.ACTION_EVENT) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_ABSOLUTE) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_DOWN) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_UP) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_DOWN) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_UP) {
               this.handleBar(evt) ;
               return true ;
            }
            else return false ;
       }

       public void handleBar(Event evt) {
         int i1,i2,i7 ;
         double v1,v2,v7 ;
         float fl1,fl2,fl7 ;
         String outspd,outspin,outang ;

         outspd = " mph" ;
         outspin = " rpm" ;
         outang = " deg" ;

// Input for computations
         i1 = s1.getValue() ;
         i2 = s2.getValue() ;
         i7 = s7.getValue() ;

         v1 = i1 * (vmx1 - vmn1)/ 1000. + vmn1 ;
         v2 = i2 * (vmx2 - vmn2)/ 1000. + vmn2 ;
         v7 = i7 * (vmx7 - vmn7)/ 1000. + vmn7 ;
 
         vfsd = v1 ;
         spin  = v2 ;
         spin = spin/60.0 ;
         sangr = v7 ;

         fl1 = (float) v1 ;
         fl2 = (float) v2 ;
         fl7 = (float) v7 ;
 
         lol1.f1.setText(String.valueOf(filter0(vfsd))) ;
         lol1.f2.setText(String.valueOf(filter0(spin*60.0))) ;
         lol1.f7.setText(String.valueOf(filter2(sangr))) ;

         computeFlow() ;
       }
     }
  } // end of M

  class L extends Panel {
     Ball outerparent ;
     Lor1 lor1 ;
     Lor2 lor2 ;

     L (Ball target) { 
          outerparent = target ;
          setLayout(new GridLayout(1,2,5,5)) ;
 
          lor1 = new Lor1(outerparent) ;
          lor2 = new Lor2(outerparent) ;

          add(lor1) ;  
          add(lor2) ;  
     }
     class Lor1 extends Panel {
        Ball outerparent ;
        TextField f6,f3,f4, f5 ;
        Label l6,l3,l4, l5 ;

        Lor1 (Ball target) {

         outerparent = target ;
         setLayout(new GridLayout(5,2,5,5)) ;

         l6 = new Label("Vert Angle - deg", Label.CENTER) ;
         f6 = new TextField("10.0",5) ;

         l3 = new Label("Y Side - feet", Label.CENTER) ;
         f3 = new TextField("0.0",5) ;

         l4 = new Label("Side Angle - deg", Label.CENTER) ;
         f4 = new TextField("0.0",5) ;

         l5 = new Label("X Distance - feet", Label.CENTER) ;
         f5 = new TextField("36.0",5) ;

         add(l6) ;
         add(f6) ;

         add(l4) ;
         add(f4) ;

         add(l3) ;
         add(f3) ;

         add(l5) ;
         add(f5) ;

         add(new Label(" ", Label.CENTER)) ;
         add(new Label(" ", Label.CENTER)) ;
       }
 
       public boolean handleEvent(Event evt) {
         if(evt.id == Event.ACTION_EVENT) {
            this.handleText(evt) ;
            return true ;
         }
         else return false ;
       }
 
       public void handleText(Event evt) {
          Double V6,V3,V4,V5 ;
          double v6,v3,v4,v5 ;
          float fl1 ;
          int i6,i3,i4,i5 ;

          V6 = Double.valueOf(f6.getText()) ;
          v6 = V6.doubleValue() ;
          V3 = Double.valueOf(f3.getText()) ;
          v3 = V3.doubleValue() ;
          V4 = Double.valueOf(f4.getText()) ;
          v4 = V4.doubleValue() ;
          V5 = Double.valueOf(f5.getText()) ;
          v5 = V5.doubleValue() ;
 
          zangr = v6 ;
          if(v6 < vmn6) {
             zangr = v6 = vmn6 ;
             fl1 = (float) v6 ;
             f6.setText(String.valueOf(fl1)) ;
          }
          if(v6 > vmx6) {
             zangr = v6 = vmx6 ;
             fl1 = (float) v6 ;
             f6.setText(String.valueOf(fl1)) ;
          }

          yangr = v4 ;
          if(v4 < vmn4) {
             yangr = v4 = vmn4 ;
             fl1 = (float) v4 ;
             f4.setText(String.valueOf(fl1)) ;
          }
          if(v4 > vmx4) {
             yangr = v4 = vmx4 ;
             fl1 = (float) v4 ;
             f4.setText(String.valueOf(fl1)) ;
          }

          i6 = (int) (((v6 - vmn6)/(vmx6-vmn6))*1000.) ;
          i4 = (int) (((v4 - vmn4)/(vmx4-vmn4))*1000.) ;

          lor2.s6.setValue(i6) ;
          lor2.s4.setValue(i4) ;

          if(ktype == 1) {          // penalty kick         
             relsy = v3 = 0.0 ;
             fl1 = (float) v3 ;
             f3.setText(String.valueOf(fl1)) ;
             i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;
             lor2.s3.setValue(i3) ;
             relsx = v5 = 36.0 ;
             fl1 = (float) v5 ;
             f5.setText(String.valueOf(fl1)) ;
             i5 = (int) (((v5 - vmn5)/(vmx5-vmn5))*1000.) ;
             lor2.s5.setValue(i5) ;
          }

          if(ktype == 2) {          // free kick
         // y location for free kick
             relsy = v3 ;
             if(v3 < vmn3) {
                relsy = v3 = vmn3 ;
                fl1 = (float) v3 ;
                f3.setText(String.valueOf(fl1)) ;
             }
             if(v3 > vmx3) {
                relsy = v3 = vmx3 ;
                fl1 = (float) v3 ;
                f3.setText(String.valueOf(fl1)) ;
             }
             relsy = relsy / lconv * relsdr ;

             i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;

             lor2.s3.setValue(i3) ;
       // x location for free kick
             relsx = v5 ;
             if(v5 < vmn5) {
                relsx = v5 = vmn5 ;
                fl1 = (float) v5 ;
                f5.setText(String.valueOf(fl1)) ;
             }
             if(v5 > vmx5) {
                relsx = v5 = vmx5 ;
                fl1 = (float) v5 ;
                f5.setText(String.valueOf(fl1)) ;
             }
             relsx = relsx / lconv * relsdr ;

             i5 = (int) (((v5 - vmn5)/(vmx5-vmn5))*1000.) ;

             lor2.s5.setValue(i5) ;
          }

          if(ktype == 3) {          // corner kick         
             relsy = v3 = -120. ;
             fl1 = (float) v3 ;
             f3.setText(String.valueOf(fl1)) ;
             i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;
             lor2.s3.setValue(i3) ;
             relsx = v5 = 0.0 ;
             fl1 = (float) v5 ;
             f5.setText(String.valueOf(fl1)) ;
             i5 = (int) (((v5 - vmn5)/(vmx5-vmn5))*1000.) ;
             lor2.s5.setValue(i5) ;
          }

          computeFlow() ;
       }
     }

     class Lor2 extends Panel {
        Ball outerparent ;
        Scrollbar s6,s3,s4,s5;
        Label lab1;

        Lor2 (Ball target) {
         int i5,i6,i3,i4 ;

         outerparent = target ;
         setLayout(new GridLayout(5,1,5,5)) ;

         i6 = (int) (((10.0 - vmn6)/(vmx6-vmn6))*1000.) ;
         i3 = (int) (((0.0 - vmn3)/(vmx3-vmn3))*1000.) ;
         i4 = (int) (((0.0 - vmn4)/(vmx4-vmn4))*1000.) ;
         i5 = (int) (((36.0 - vmn5)/(vmx5-vmn5))*1000.) ;

         s6 = new Scrollbar(Scrollbar.HORIZONTAL,i6,10,0,1000);
         s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
         s4 = new Scrollbar(Scrollbar.HORIZONTAL,i4,10,0,1000);
         s5 = new Scrollbar(Scrollbar.HORIZONTAL,i5,10,0,1000);

         add(s6) ;
         add(s4) ;
         add(s3) ;
         add(s5) ;
         add(new Label(" ", Label.CENTER)) ;
       }
 
       public boolean handleEvent(Event evt) {
            if(evt.id == Event.ACTION_EVENT) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_ABSOLUTE) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_DOWN) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_UP) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_DOWN) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_UP) {
               this.handleBar(evt) ;
               return true ;
            }
            else return false ;
       }

       public void handleBar(Event evt) {
         int i6,i3,i4,i5 ;
         double v6,v3,v4,v5 ;
         float fl6,fl3,fl4,fl5 ;
         String outspd,outspin,outang,outht ;

         outht = " feet" ;
         outang = " deg" ;

// Input for computations
         i6 = s6.getValue() ;
         i4 = s4.getValue() ;

         v6 = i6 * (vmx6 - vmn6)/ 1000. + vmn6 ;
         v4 = i4 * (vmx4 - vmn4)/ 1000. + vmn4 ;
 
         zangr = v6 ;
         yangr = v4 ;

         fl6 = (float) v6 ;
         fl4 = (float) v4 ;
 
         lor1.f6.setText(String.valueOf(filter2(zangr))) ;
         lor1.f4.setText(String.valueOf(filter2(yangr))) ;

         if(ktype == 1) {  // penalty kick
             relsy = v3 = 0.0 ;
             fl3 = (float) v3 ;
             lor1.f3.setText(String.valueOf(fl3)) ;
             i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;
             s3.setValue(i3) ;
             relsx = v5 = 36.0 ;
             fl5 = (float) v5 ;
             lor1.f5.setText(String.valueOf(fl5)) ;
             i5 = (int) (((v5 - vmn5)/(vmx5-vmn5))*1000.) ;
             s5.setValue(i5) ;
         }

         if(ktype == 2) {  // free kick
       // y side location
            i3 = s3.getValue() ;
            v3 = i3 * (vmx3 - vmn3)/ 1000. + vmn3 ;
            relsy = v3 / lconv * relsdr ;
            fl3 = (float) v3 ;
            lor1.f3.setText(String.valueOf(filter3(relsy))) ;
       // x dist location
            i5 = s5.getValue() ;
            v5 = i5 * (vmx5 - vmn5)/ 1000. + vmn5 ;
            relsx = v5 / lconv * relsdr ;
            fl5 = (float) v5 ;
            lor1.f5.setText(String.valueOf(filter3(relsx))) ;
         }

         if(ktype == 3) {  // corner kick
             relsy = v3 = -120.0 ;
             fl3 = (float) v3 ;
             lor1.f3.setText(String.valueOf(fl3)) ;
             i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;
             s3.setValue(i3) ;
             relsx = v5 = 0.0 ;
             fl5 = (float) v5 ;
             lor1.f5.setText(String.valueOf(fl5)) ;
             i5 = (int) (((v5 - vmn5)/(vmx5-vmn5))*1000.) ;
             s5.setValue(i5) ;
         }

         computeFlow() ;
       }
     }
   }
  }

  class View1 extends Canvas  
         implements Runnable{
     Ball outerparent ;
     Thread runner ;
     Image displimg,nasaimg ;
     Point locate,anchor;

     View1 (Ball target) {
         setBackground(Color.black) ;
         runner = null ;
         displimg = getImage(getCodeBase(),"soccerball.gif") ;
         nasaimg = getImage(getCodeBase(),"mball3.gif") ;
     }

     public Insets insets() {
        return new Insets(0,10,0,10) ;
     }

     public boolean mouseDown(Event evt, int x, int y) {
        anchor = new Point(x,y) ;
        return true;
     }

     public boolean mouseUp(Event evt, int x, int y) {
         handle(x,y) ;
         return true;
     }

     public boolean mouseDrag(Event evt, int x, int y) {
        handle(x,y) ;
        return true;
     }

     public void handle(int x, int y) {
         // determine location and set button parameters

         if (y >= 100 && y <= 125) {      //  view buttons
           if (x >= 0 && x <= 100) {      //  downfield view
             drawview = 1 ;
             view1.repaint();
             return ;
           }

           if (x >= 110 && x <= 190) {      //  top view
             drawview = 2 ;
             view1.repaint();
             return ;
           }

           if (x >= 200 && x <= 280) {      //  side view
             drawview = 4 ;
             view1.repaint();
             return ;
           }

           if (x >= 290 && x <= 370) {      //  tv view
             drawview = 3 ;
             view1.repaint();
             return ;
           }

           if (x >= 380 && x <= 450) {      //  find button
              xtt = 400 ;
              ytt = 300 ;
              xts = 400 ;
              yts = 430 ;
              xtk = 250 ;
              ytk = 400 ;
              xtv = 400 ;
              ytv = 430 ;

              view1.repaint();
              return ;
            }
         }

         if (y > 450 ) {   // zoom widget
            sldloc = x ;
            if (sldloc < 50) sldloc = 50;
            if (sldloc > 440) sldloc = 440;
            fact = 10.0 + (sldloc-50) * .2 ;
            view1.repaint() ;
            return;
         }

         if (y >= 125 && y <= 449) {      // move the drawing
            locate = new Point(x,y) ;
            ytt =  ytt + (int) (.2*(locate.y - anchor.y)) ;
            xtt =  xtt + (int) (.4*(locate.x - anchor.x))  ;
            if (xtt > 400) xtt = 400 ;
            if (xtt < 0) xtt = 0 ;
            if (ytt > 500) ytt = 500 ;
            if (ytt < 0) ytt = 0 ;
            yts =  yts + (int) (.2*(locate.y - anchor.y)) ;
            xts =  xts + (int) (.4*(locate.x - anchor.x))  ;
            if (xts > 700) xts = 700 ;
            if (xts < -100) xts = -100 ;
            if (yts > 500) yts = 500 ;
            if (yts < 0) yts = 0 ;
            ytk =  ytk + (int) (.2*(locate.y - anchor.y)) ;
            xtk =  xtk + (int) (.4*(locate.x - anchor.x))  ;
            if (xtk > 500) xtk = 500 ;
            if (xtk < -280) xtk = -280 ;
            if (ytk > 520) ytk = 520 ;
            if (ytk < 300) ytk = 300 ;
            ytv =  ytv + (int) (.2*(locate.y - anchor.y)) ;
            xtv =  xtv + (int) (.4*(locate.x - anchor.x))  ;
            if (xtv > 700) xtv = 700 ;
            if (xtv < -100) xtv = -100 ;
            if (ytv > 500) ytv = 500 ;
            if (ytv < 0) ytv = 0 ;

            view1.repaint() ;
            return;
         }
     }

     public void start() {
        if (runner == null) {
           runner = new Thread(this) ;
           runner.start() ;
        }
        antim1 = 0 ;                              /* MODS  21 JUL 99 */
        ancol1 = 1 ;                              /* MODS  27 JUL 99 */
     }

     public void run() {
       int timer ;
       String outfor,outtim,outspd ;
       double yvld,xvld,zvld,hangle,vangle,deltim;
       double yald,xald,zald;

       outspd = " mph" ;
       outfor = " oz" ;
       outtim = " sec" ;
       timer = 100 ;
       while (true) {
          ++ antim1 ;
          try { Thread.sleep(timer); }
          catch (InterruptedException e) {}
          if (antim1 == 3) {
             antim1 = 0;
             ancol1 = - ancol1 ;               /* MODS 27 JUL 99 */
          }
          timer = 135 - (int) (.227 *vfsd/vconv) ;
                                            // make the ball spin
          plthg1[1] = plthg1[1] - spin*spindr*1.489 ;
          if (plthg1[1] < -360.0) {
                plthg1[1] = plthg1[1] + 360.0 ;
          }
          if (plthg1[1] > 360.0) {
                plthg1[1] = plthg1[1] - 360.0 ;
          }
          plthg1[0] = plthg1[1] + 180.0 ;
          if (plthg1[0] > 270.0) {
                plthg1[0] = plthg1[0] - 360.0 ;
          }

   // new integration scheme
          if (pitch == 1) {
             deltim = .05 ;
             if (comp == 1) {
                zvld = zvel ;
                yvld = yvel ;
                xvld = xvel ;
                zald = zacc ;
                yald = yacc ;
                xald = xacc ;

                hangle = Math.atan2(yvel,xvel) ;
                vangle = Math.atan2(zvel,(xvel*xvel + yvel*yvel)) ;

                spd = Math.sqrt(yvel*yvel + xvel*xvel + zvel*zvel) ;
       // magnitude of forces
                side = gamval * rho * spd*spd; /* lift/ft */
                side = side * rball * 4.0/3.0  ; // lift integration

                drag = cd * .5 * rho * area * spd * spd ;
                side = clift * side  ; 

       // forces
                xfor = side*Math.cos(sangr*convdr)*Math.sin(hangle) 
                        - drag*Math.cos(vangle)*Math.cos(hangle) ;
                yfor = -side*Math.cos(sangr*convdr)*Math.cos(hangle)
                        - drag*Math.cos(vangle)*Math.sin(hangle)  ;
                zfor = -wball - side*Math.sin(sangr*convdr) - drag*Math.sin(vangle) ;

       // acceleration
                xacc =  xfor / wball * 32.2 ; 
                yacc =  yfor / wball * 32.2 ;
                zacc =  zfor / wball * 32.2 * wtrat ;

                xvel = xvel + .5 * (xacc + xald) * deltim ;
                yvel = yvel + .5 * (yacc + yald) * deltim ;
                zvel = zvel + .5 * (zacc + zald) * deltim ;

                xloc = xloc + .5 * (xvel + xvld) * deltim ;
                yloc = yloc + .5 * (yvel + yvld) * deltim ;
                zloc = zloc + .5 * (zvel + zvld) * deltim ;
    
                npt = npt + 1 ;
                ptim = ptim + deltim ;
   
                inp.mdr.o5.setText(String.valueOf(
                               filter3(Math.abs(side*fconv*16.0))) + outfor) ;
                inp.mdr.o6.setText(String.valueOf(
                               filter3(Math.abs(drag*fconv*16.0))) + outfor) ;
                inp.mdr.o3.setText(String.valueOf(
                                          filter2(ptim)) + outtim) ;
                inp.mdr.o1.setText(String.valueOf(
                                          filter0(spd*vconv)) + outspd) ;
                inp.mdr.o2.setText(String.valueOf(
                                          filter3(cd))) ;
                inp.mdr.o4.setText(String.valueOf(
                                          filter3(clift))) ;

                if (zloc <= 0.0) {    // stop calculation - ball hits the ground
                   comp = 0 ;
                   call = 2 ;
                   zoff[nsav] = zloc ;
                }

                if (xloc <= 0.0) {  // stop calculation - ball passes end line
                   comp = 0 ;
                   yoff[nsav] = yloc ;
                   zoff[nsav] = zloc ;
                   call = 1 ;
                }

                if (ktype == 3) {   // corner kick
                   if (yloc > -9.0 && yloc < 15.0 ) { 
                       if (xloc > 24.0 && xloc < 30.0) {
                          if (zloc > 5.5 && zloc < 8.0) {  // stop calculation - ball in target box
                              comp = 0 ;
                              call = 3 ;
                              yoff[nsav] = yloc ;
                              zoff[nsav] = zloc ;
                          }
                       }
                   }
                }
             }
          }
   // reset integration
          if (pitch == 0) {
              xloc = relsx ;
              xvel = -vfsd/vconv ;
              xacc = 0.0 ;
              yloc = relsy ;
              yvel = 0.0 ;
              yacc = 0.0 ;
              ptim = 0.0 ;
              call = 0 ;
          }
   // animate path of ball

          nptb = nptb + 1 ;
          if (nptb >= npt) nptb = npt ;

          posx[nsav][nptb] = xloc ;
          posy[nsav][nptb] = yloc ;
          posz[nsav][nptb] = zloc ;

          view1.repaint() ;
       }
     }

     public void loadPlot() {
       return ;
     }

     public void update(Graphics g) {
        view1.paint(g) ;
     }
 
    public void paint(Graphics g) {
      int i,j,k,n,tbgn,tend ;
      int nump,xpic,ypic,xpicg ;
      int xlabel,ylabel,ind,inmax,inmin ;
      int ifact ;
      int exes[] = new int[8] ;
      int whys[] = new int[8] ;
      int hite,wdth ;
      double offx,scalex;
      double offy,scaley,scalecy;
      double offz,scalez,scalecz;
      double length,perspec;
      Color col1,col2,col3 ;

      col1 = new Color(0,121,0) ;   // dark green
      col2 = new Color(254,121,0) ; // dark orange
      col3 = new Color(0,220,220) ; // sky blue

      offx = 0.0 - begx ;
      scalex = 8.0/(endx-begx) ;
      offy = 0.0 - begy ;
      scaley = 8.0/(endy-begy) ;
      offz = 0.0 - begz ;
      scalez = 8.0/(endz-begz) ;
      scalecz = 20.0/(endz-begz) ;
      scalecy = 20.0/(endy-begy) ;

  //  Top View
     if(drawview == 2) {
         offsGg.setColor(col1) ;
         offsGg.fillRect(0,0,500,600) ;
 // Playing Field
                 // goal line
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy - 120.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy + 120.0 ))) + ytt ;
          offsGg.setColor(Color.white) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 // side lines
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy - 120.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(offx -dist + 360.0))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy - 120.0 ))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy + 120.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(offx -dist + 360.0))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy + 120.0 ))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              // center line
          exes[0] = (int) (fact* (scalex*(offx - dist + 180.0 ))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy - 120.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(offx -dist + 180.0))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy + 120.0 ))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 // center circle
          exes[0] = (int) (fact* (scalex*(offx - dist + 150.0 ))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy + 30.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(60.0))) ;
          whys[1] = (int) (fact* (scaley*(60.0)));
          offsGg.drawArc(exes[0],whys[0],exes[1],whys[1],0,360) ;
                // goal box
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy - 30.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(offx - dist + 18.0))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy - 30.0 ))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx - dist + 18.0))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy + 30.0))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[1] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy + 30.0))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                // penalty box
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy - 66.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(offx -dist + 54.0))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy - 66.0 ))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx -dist + 54.0))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy + 66.0))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[1] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy + 66.0))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 // penalty circle
          exes[0] = (int) (fact* (scalex*(offx -dist + 6.0))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy + 30.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(60.0))) ;
          whys[1] = (int) (fact* (scaley*(60.0)));
          offsGg.drawArc(exes[0],whys[0],exes[1],whys[1],-53,106) ;
               // goal net
          offsGg.setColor(Color.yellow) ;
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy - 12.0))) + ytt ;
          exes[1] = (int) (fact* (scalex*(offx - dist - 9.0))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy - 12.0 ))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx - dist - 9.0))) + xtt ;
          whys[0] = (int) (fact* (-scaley*(offy + 12.0))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[1] = (int) (fact* (scalex*(offx - dist))) + xtt ;
          whys[1] = (int) (fact* (-scaley*(offy + 12.0))) + ytt ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          for (i=0; i<=24; ++i) {
             exes[0] = (int) (fact* (scalex*(offx - dist))) + xtt ;
             whys[0] = (int) (fact* (-scaley*(offy - 12.0 + 1.0 * i))) + ytt ;
             exes[1] = (int) (fact* (scalex*(offx - dist - 9.0))) + xtt ;
             whys[1] = (int) (fact* (-scaley*(offy - 12.0 + 1.0 * i))) + ytt ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
          for (i=0; i<=9; ++i) {
             exes[0] = (int) (fact* (scalex*(offx - dist - 1.0 * i))) + xtt ;
             whys[0] = (int) (fact* (-scaley*(offy - 12.0))) + ytt ;
             exes[1] = (int) (fact* (scalex*(offx - dist - 1.0 * i))) + xtt ;
             whys[1] = (int) (fact* (-scaley*(offy + 12.0))) + ytt ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
              // posts
          offsGg.fillOval((int) (fact*scalex*(offx - dist)) + xtt,
          (int) (fact*(-scaley*(offy - 12.0))) + ytt - 2,5,5);
          offsGg.fillOval((int) (fact*scalex*(offx - dist)) + xtt,
          (int) (fact*(-scaley*(offy + 12.0))) + ytt - 2,5,5);
 
          if (ktype == 3) {   // target box for corner kick
             offsGg.setColor(Color.magenta) ;
             exes[0] = (int) (fact* (scalex*(offx + 24.0))) + xtt ;
             whys[0] = (int) (fact* (-scaley*(offy - 9.0))) + ytt ;
             exes[1] = (int) (fact* (scalex*(offx + 24.0))) + xtt ;
             whys[1] = (int) (fact* (-scaley*(offy + 15.0))) + ytt ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             exes[0] = (int) (fact* (scalex*(offx + 30.0))) + xtt ;
             whys[0] = (int) (fact* (-scaley*(offy - 9.0))) + ytt ;
             exes[1] = (int) (fact* (scalex*(offx + 30.0))) + xtt ;
             whys[1] = (int) (fact* (-scaley*(offy + 15.0))) + ytt ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             exes[0] = (int) (fact* (scalex*(offx + 24.0))) + xtt ;
             whys[0] = (int) (fact* (-scaley*(offy - 9.0))) + ytt ;
             exes[1] = (int) (fact* (scalex*(offx + 30.0))) + xtt ;
             whys[1] = (int) (fact* (-scaley*(offy - 9.0))) + ytt ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             exes[0] = (int) (fact* (scalex*(offx + 24.0))) + xtt ;
             whys[0] = (int) (fact* (-scaley*(offy + 15.0))) + ytt ;
             exes[1] = (int) (fact* (scalex*(offx + 30.0))) + xtt ;
             whys[1] = (int) (fact* (-scaley*(offy + 15.0))) + ytt ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }  
 // ball and vector
         if (comp == 0) {
            offsGg.fillOval((int) (fact*scalex*(offx+relsx)) + xtt,
             (int) (fact*(-scaley*(offy+relsy))) + ytt - 2,5,5);
            exes[0] = (int) (fact* (scalex*(offx+relsx))) + xtt ;
            whys[0] = (int) (fact* (-scaley*(offy+relsy))) + ytt ;
            length = 10.0 * (vfsd / vmaxa) ;
            exes[1] = (int) (fact* (scalex*(offx+relsx - 
                     length * Math.cos(convdr*yangr)))) + xtt ;
            whys[1] = (int) (fact* (-scaley*(offy+relsy +
                     length * Math.sin(convdr*yangr)))) + ytt ;
            offsGg.setColor(Color.cyan) ;
            offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
            offsGg.setColor(Color.white) ;
            offsGg.fillOval((int) (fact*scalex*(offx+relsx )) + xtt ,
             (int) (fact*(-scaley*(offy+relsy))) + ytt - 5,10,10);
            exes[0] = (int) (fact* (scalex*(offx+relsx + 
                     0.5 * Math.cos(convdr*(plthg1[1] + 180.))))) + xtt + 5 ;
            whys[0] = (int) (fact* (-scaley*(offy+relsy +
                     0.5 * Math.sin(convdr*(plthg1[1] + 180.))))) + ytt ;
            exes[1] = (int) (fact* (scalex*(offx+relsx + 
                     0.5 * Math.cos(convdr*plthg1[1])))) + xtt + 5 ;
            whys[1] = (int) (fact* (-scaley*(offy+relsy +
                     0.5 * Math.sin(convdr*plthg1[1])))) + ytt ;
            offsGg.setColor(Color.red) ;
            offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
        }
 // animated kick
   // saved kicks
         if (nsav >= 1) {
           for (i=0; i<= nsav-1; ++ i) {
              exes[0] = (int) (fact* (scalex*(offx+posx[i][0]))) + xtt ;
              whys[0] = (int) (fact* (-scaley*(offy+posy[i][0]))) + ytt ;
              if(i==0) {
                 offsGg.setColor(Color.yellow) ;
//                 offsGg.drawString("inches",350,195) ;
              }
              if(i==1) offsGg.setColor(Color.orange) ;
              if(i==2) offsGg.setColor(Color.red) ;
              if(i==3) offsGg.setColor(Color.magenta) ;
              if(i==4) offsGg.setColor(Color.orange) ;
              offsGg.fillOval(exes[0],whys[0],5,5);
              nump = npts[i] ;
              for (j=1; j<=nump; ++j){
                 exes[1] = exes[0] ;
                 whys[1] = whys[0] ;
                 exes[0] = (int) (fact*scalex*(offx+posx[i][j])) + xtt ;
                 whys[0] = (int) (fact*(-scaley*(offy+posy[i][j]))) + ytt ;
                 offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 offsGg.fillOval(exes[0],whys[0]-2,5,5);
              }
//              offsGg.drawString(String.valueOf(filter1(yoff[i])),320,195+i*15) ;
           }
         }

         if (pitch == 1) {
            exes[0] = (int) (fact* (scalex*(offx+posx[nsav][0]))) + xtt ;
            whys[0] = (int) (fact* (-scaley*(offy+posy[nsav][0]))) + ytt ;
            offsGg.setColor(Color.white) ;
            for (i=1; i<=nptb; ++i) {                /* plot trajectory */
               exes[1] = exes[0] ;
               whys[1] = whys[0] ;
               exes[0] = (int) (fact*scalex*(offx+posx[nsav][i])) + xtt ;
               whys[0] = (int) (fact*(-scaley*(offy+posy[nsav][i]))) + ytt ;
               offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
               offsGg.fillOval(exes[0],whys[0]-2,5,5);
            }
            offsGg.setColor(Color.white) ;
            offsGg.fillOval((int) (fact*scalex*(offx+posx[nsav][nptb])) + xtt ,
             (int) (fact*(-scaley*(offy+posy[nsav][nptb]))) + ytt - 5,10,10);
         }
//Buttons 
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(0,100,100,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(0,100,100,25) ;
         offsGg.drawString("Downfield View",10,120) ;
         offsGg.setColor(Color.yellow) ;
         offsGg.fillRect(110,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(110,100,80,25) ;
         offsGg.drawString("Top View",120,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(200,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(200,100,80,25) ;
         offsGg.drawString("Side View",210,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(290,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(290,100,80,25) ;
         offsGg.drawString("TV View",300,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(380,100,70,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(380,100,70,25) ;
         offsGg.drawString("Find",390,120) ;
      }

 //  TV View
     if(drawview == 3) {
       offsGg.setColor(col1) ;
       offsGg.fillRect(0,0,500,600) ;
 // Playing Field
       perspec = 20. * convdr ;
                 // goal line
       offsGg.setColor(Color.white) ;
       exes[0] = (int) (fact* (scalex*(offx - dist - 
                         120.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz -
                         120.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist +
                         120.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz + 
                         120.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 // side lines
       exes[0] = (int) (fact* (scalex*(offx - dist - 
                          120.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz - 
                          120.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist + 360. -
                          120.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz - 
                          120.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist + 
                          120.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz + 
                          120.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist + 360. +
                          120.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz + 
                          120.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                  // goal box
       exes[0] = (int) (fact* (scalex*(offx - dist + 18.0 - 
                          30.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz - 
                          30.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist + 18.0 + 
                          30.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz +
                          30.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist - 
                          30.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz - 
                          30.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist + 18.0 -
                          30.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz -
                          30.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist + 
                          30.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz + 
                          30.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist + 18.0 +
                          30.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz +
                          30.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                  // penalty box
       exes[0] = (int) (fact* (scalex*(offx - dist + 54.0 - 
                          66.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz - 
                          66.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist + 54.0 + 
                          66.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz +
                          66.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist - 
                         66.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz - 
                         66.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist + 54.0 -
                         66.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz -
                         66.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist + 
                         66.* Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz + 
                         66.*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist + 54.0 +
                         66.*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz +
                         66.*Math.cos(perspec)))) + ytv;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 // penalty circle
       exes[0] = (int) (fact* (scalex*(offx - dist + 6.0))) + xtv ;
       whys[0] = (int) (fact* (-scaley*(offy + 30.0))) + ytv ;
       exes[1] = (int) (fact* (scalex*(60.0))) ;
       whys[1] = (int) (fact* (scaley*(60.0)));
       offsGg.drawArc(exes[0],whys[0],exes[1],whys[1],-75,110) ;
                  // goal post
       offsGg.setColor(Color.yellow) ;
       exes[0] = (int) (fact* (scalex*(offx - dist -
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz -
                         12.0*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist -
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz + 8.0 -
                         12.0*Math.cos(perspec)))) + ytv ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist +
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz +
                         12.0*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist +
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                         12.0*Math.cos(perspec)))) + ytv ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist -
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz +8.0 -
                         12.0*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist +
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                         12.0*Math.cos(perspec)))) + ytv ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                  // goal 
       exes[0] = (int) (fact* (scalex*(offx - dist -
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz -
                         12.0*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist - 8.0 -
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz -
                         12.0*Math.cos(perspec)))) + ytv ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist +
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz +
                         12.0*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist - 8.0 +
                         12.0*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz +
                         12.0*Math.cos(perspec)))) + ytv ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist - 8.0 -
                        12.0*Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz -
                        12.0*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist - 8.0 +
                        12.0*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz +
                        12.0*Math.cos(perspec)))) + ytv ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist - 8.0  -
                        12.0*Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz -
                        12.0*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist  -
                        12.0*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz +8.0 -
                        12.0*Math.cos(perspec)))) + ytv ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (scalex*(offx - dist - 8.0 +
                        12.0*Math.sin(perspec)))) + xtv ;
       whys[0] = (int) (fact* (-scalez*(offz +
                        12.0*Math.cos(perspec)))) + ytv ;
       exes[1] = (int) (fact* (scalex*(offx - dist +
                        12.0*Math.sin(perspec)))) + xtv ;
       whys[1] = (int) (fact* (-scalez*(offz +8.0 +
                        12.0*Math.cos(perspec)))) + ytv ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
//net
       for( i=0; i<=8; ++ i) {
          exes[0] = (int) (fact* (scalex*(offx - dist - 8.0 + 1.0 * i -
                         12.0*Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 1.0 * i -
                         12.0*Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx - dist - 8.0 + 1.0 * i +
                         12.0*Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 1.0 * i +
                         12.0*Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }
       for( i=0; i<=8; ++ i) {
          exes[0] = (int) (fact* (scalex*(offx - dist - 8.0 + 1.0 * i -
                         12.0*Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 1.0 * i -
                         12.0*Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx - dist -
                          12.0*Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 1.0 * i -
                          12.0*Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }
       for( i=0; i<=8; ++ i) {
          exes[0] = (int) (fact* (scalex*(offx - dist - 8.0 + 1.0 * i +
                          12.0*Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 1.0 * i +
                          12.0*Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx - dist +
                          12.0*Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 1.0 * i +
                          12.0*Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }
       for( i=0; i<=8; ++ i) {
          exes[0] = (int) (fact* (scalex*(offx - dist -8.0 + 1.0*i  -
                          12.0*Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz -
                          12.0*Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx - dist  -
                          12.0*Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz +8.0 -
                          12.0*Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }
       for( i=0; i<=8; ++ i) {
          exes[0] = (int) (fact* (scalex*(offx - dist -8.0 + 1.0*i  +
                          12.0*Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz +
                          12.0*Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx - dist  +
                          12.0*Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz +8.0 +
                          12.0*Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }
       for( i=0; i<=24; ++ i) {
          exes[0] = (int) (fact* (scalex*(offx - dist - 8.0 + 
                          (-12.0 + 1.0 *i)*Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 
                          (-12.0 + 1.0 *i)*Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx - dist + 
                          (-12.0 + 1.0*i)*Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 + 
                          (-12.0 + 1.0*i)*Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }

       if (ktype == 3) {   // target box for corner kick
          offsGg.setColor(Color.magenta) ;
             // bottom
          exes[0] = (int) (fact* (scalex*(offx + 24.0 +
                          (-9.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 5.5 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 24.0 +
                            15.0 * Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 5.5 +
                            15.0 * Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 30.0 +
                          (-9.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 5.5 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 30.0 +
                            15.0 * Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 5.5 +
                            15.0 * Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 24.0 +
                          (-9.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 5.5 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 30.0 +
                           (-9.0)* Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 5.5 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 24.0 +
                          (15.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 5.5 +
                           (15.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 30.0 +
                           (15.0)* Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 5.5 +
                           (15.0)* Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                   //top
          exes[0] = (int) (fact* (scalex*(offx + 24.0 +
                          (-9.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 8.0 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 24.0 +
                            15.0 * Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                            15.0 * Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 30.0 +
                          (-9.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 8.0 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 30.0 +
                            15.0 * Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                            15.0 * Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 24.0 +
                          (-9.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 8.0 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 30.0 +
                           (-9.0)* Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 24.0 +
                          (15.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 8.0 +
                           (15.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 30.0 +
                           (15.0)* Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                           (15.0)* Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 // posts
          exes[0] = (int) (fact* (scalex*(offx + 24.0 +
                          (15.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 5.5 +
                           (15.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 24.0 +
                           (15.0)* Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                           (15.0)* Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 30.0 +
                          (15.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 5.5 +
                           (15.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 30.0 +
                           (15.0)* Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                           (15.0)* Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 24.0 +
                          (-9.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 5.5 +
                          (-9.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 24.0 +
                          (-9.0)* Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                          (-9.0)* Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx + 30.0 +
                          (-9.0) * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + 5.5 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + 30.0 +
                           (-9.0)* Math.sin(perspec)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + 8.0 +
                           (-9.0)* Math.cos(perspec)))) + ytv ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

       }  

 // ball and vector
       if (comp == 0) {
          offsGg.fillOval((int) (fact*scalex*(offx + relsx +
                   relsy * Math.sin(perspec))) + xtv,
             (int) (fact*(-scalez*(offz + relsz +
                   relsy * Math.cos(perspec)))) + ytv - 2,5,5);
          exes[0] = (int) (fact* (scalex*(offx + relsx +
                    relsy * Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz + relsz +
                   relsy * Math.cos(perspec)))) + ytv ;
          length = 10.0 * (vfsd / vmaxa) ;
          exes[1] = (int) (fact* (scalex*(offx + relsx +
                     relsy * Math.sin(perspec) - 
                     length * Math.cos(convdr*zangr)))) + xtv ;
          whys[1] = (int) (fact* (-scalez*(offz + relsz +
                     relsy * Math.cos(perspec) +
                     length * Math.sin(convdr*zangr)))) + ytv ;
//          offsGg.setColor(Color.cyan) ;
 //         offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          offsGg.setColor(Color.white) ;
          offsGg.fillOval((int) (fact*scalex*(offx + relsx +
                     relsy * Math.sin(perspec) )) + xtv ,
             (int) (fact*(-scalez*(offz+relsz +
                     relsy * Math.cos(perspec)))) + ytv - 5,10,10);
          exes[0] = (int) (fact* (scalex*(offx + relsx + 
                     relsy * Math.sin(perspec) + 
                     0.5 * Math.cos(convdr*(plthg1[1] + 180.))))) + xtv + 5 ;
          whys[0] = (int) (fact* (-scalez*(offz+relsz +
                     relsy * Math.cos(perspec) +
                     0.5 * Math.sin(convdr*(plthg1[1] + 180.))))) + ytv ;
          exes[1] = (int) (fact* (scalex*(offx + relsx +
                     relsy * Math.sin(perspec) + 
                     0.5 * Math.cos(convdr*plthg1[1])))) + xtv + 5 ;
          whys[1] = (int) (fact* (-scalez*(offz+relsz +
                      relsy * Math.cos(perspec)   +
                     0.5 * Math.sin(convdr*plthg1[1])))) + ytv ;
          offsGg.setColor(Color.red) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }
   // saved kicks
       if (nsav >= 1) {
         for (i=0; i<= nsav-1; ++ i) {
            exes[0] = (int) (fact* (scalex*(offx+posx[i][0] +
                      posy[i][0]*Math.sin(perspec)))) + xtv ;
            whys[0] = (int) (fact* (-scalez*(offz+posz[i][0] +
                      posy[i][0] * Math.cos(perspec)))) + ytv ;
            if(i==0) {
               offsGg.setColor(Color.yellow) ;
               offsGg.drawString("feet",360,15) ;
            }
            if(i==1) offsGg.setColor(Color.orange) ;
            if(i==2) offsGg.setColor(Color.red) ;
            if(i==3) offsGg.setColor(Color.magenta) ;
            if(i==4) offsGg.setColor(Color.orange) ;
            offsGg.fillOval(exes[0],whys[0],5,5);
            nump = npts[i] ;
            for (j=1; j<=nump; ++j){
               exes[1] = exes[0] ;
               whys[1] = whys[0] ;
               exes[0] = (int) (fact*scalex*(offx+posx[i][j] +
                       posy[i][j]*Math.sin(perspec))) + xtv ;
               whys[0] = (int) (fact*(-scalez*(offz+posz[i][j] +
                       posy[i][j]*Math.cos(perspec)))) + ytv ;
               offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
               offsGg.fillOval(exes[0]-2,whys[0]-2,5,5);
            }
            offsGg.drawString(String.valueOf(filter3(zoff[i])),320,15 + i*15) ;
         }
       }
  // kick
       if (pitch == 1) {
          exes[0] = (int) (fact* (scalex*(offx+posx[nsav][0] +
                      posy[nsav][0]*Math.sin(perspec)))) + xtv ;
          whys[0] = (int) (fact* (-scalez*(offz+posz[nsav][0] +
                      posy[nsav][0]*Math.cos(perspec)))) + ytv ;
          offsGg.setColor(Color.white) ;
          for (i=1; i<=nptb; ++i) {                /* plot trajectory */
             exes[1] = exes[0] ;
             whys[1] = whys[0] ;
             exes[0] = (int) (fact*scalex*(offx+posx[nsav][i] +
                        posy[nsav][i]*Math.sin(perspec))) + xtv ;
             whys[0] = (int) (fact*(-scalez*(offz+posz[nsav][i] +
                        posy[nsav][i]*Math.cos(perspec)))) + ytv ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             offsGg.fillOval(exes[0]-2,whys[0]-2,5,5);
          }
          offsGg.setColor(Color.white) ;
          offsGg.fillOval((int) (fact*scalex*(offx+posx[nsav][nptb] +
                      posy[nsav][nptb]*Math.sin(perspec))) + xtv-5 ,
             (int) (fact*(-scalez*(offz+posz[nsav][nptb] +
                      posy[nsav][nptb]*Math.cos(perspec)))) + ytv - 5,10,10);
       }
//Buttons
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(0,100,100,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(0,100,100,25) ;
         offsGg.drawString("Downfield View",10,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(110,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(110,100,80,25) ;
         offsGg.drawString("Top View",120,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(200,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(200,100,80,25) ;
         offsGg.drawString("Side View",210,120) ;
         offsGg.setColor(Color.yellow) ;
         offsGg.fillRect(290,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(290,100,80,25) ;
         offsGg.drawString("TV View",300,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(380,100,70,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(380,100,70,25) ;
         offsGg.drawString("Find",390,120) ;
     }

//  Downfield View
     if(drawview == 1) {
       perspec = 5.0 * convdr ;
       offsGg.setColor(col1) ;
       offsGg.fillRect(0,280,500,360) ;
       offsGg.setColor(col3) ;
       offsGg.fillRect(0,100,500,180) ;  
 // Playing Field
                 // goal line
       offsGg.setColor(Color.white) ;
       exes[0] = (int) (fact* (-scalecy*(offy - 120.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 120.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             // side lines
       offsGg.setColor(Color.white) ;
       exes[0] = (int) (fact* (-scalecy*(offy - 120.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy - 125.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz -
                   200 * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       offsGg.setColor(Color.white) ;
       exes[0] = (int) (fact* (-scalecy*(offy + 120.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 125.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz -
                   200 * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
         // goal box
       exes[0] = (int) (fact* (-scalecy*(offy - 31.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   (dist - 18.0) * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 31.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz +
                   (dist - 18.0) * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (-scalecy*(offy - 31.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   (dist - 18.0) * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy - 30.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (-scalecy*(offy + 31.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   (dist - 18.0) * Math.sin( perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 30.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
            // penalty box
       exes[0] = (int) (fact* (-scalecy*(offy + 66.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin( perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 67.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz +
                   (dist - 54.0) * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (-scalecy*(offy - 66.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy - 67.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz +
                   (dist - 54.0) * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       exes[0] = (int) (fact* (-scalecy*(offy - 67.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   (dist - 54.0) * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 67.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz +
                   (dist - 54.0) * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             // goal posts
       offsGg.setColor(Color.yellow) ;
       exes[0] = (int) (fact* (-scalecy*(offy - 12.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin( perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy - 12.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                   dist * Math.sin(perspec)))) + ytk ;
       hite = (whys[0] - whys[1]) ;
       wdth = 5 ; 
       offsGg.fillRect(exes[0],whys[1],wdth,hite) ;
       exes[0] = (int) (fact* (-scalecy*(offy + 12.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz +
                   dist * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 12.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                   dist * Math.sin(perspec)))) + ytk ;
       hite = (whys[0] - whys[1]) ;
       wdth = 5 ; 
       offsGg.fillRect(exes[0],whys[1],wdth,hite) ;
       exes[0] = (int) (fact* (-scalecy*(offy - 12.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz + 8.0 +
                   dist * Math.sin( perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 12.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                   dist * Math.sin(perspec)))) + ytk ;
       hite = 5 ;
       wdth = (exes[0]-exes[1]) ; 
       offsGg.fillRect(exes[1],whys[1],wdth,hite) ;

       exes[0] = (int) (fact* (-scalecy*(offy - 11.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz  +
                   (dist + 4.0) * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 11.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz  +
                   (dist + 4.0) * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

       exes[0] = (int) (fact* (-scalecy*(offy - 11.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz  +
                   (dist + 4.0) * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy - 12.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz  + 8.0 +
                    dist * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

       exes[0] = (int) (fact* (-scalecy*(offy + 11.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz  +
                   (dist + 4.0) * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 12.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz  + 8.0 +
                   dist * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

       exes[0] = (int) (fact* (-scalecy*(offy + 11.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz  +
                   (dist + 4.0) * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy + 12.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz  +
                   dist * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

       exes[0] = (int) (fact* (-scalecy*(offy - 11.0))) + xtk ;
       whys[0] = (int) (fact* (-scalecz*(offz  +
                   (dist + 4.0) * Math.sin(perspec)))) + ytk ;
       exes[1] = (int) (fact* (-scalecy*(offy - 12.0))) + xtk ;
       whys[1] = (int) (fact* (-scalecz*(offz  +
                   dist * Math.sin(perspec)))) + ytk ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
//  net
       for (i=0; i<= 24 ; ++ i) {
           exes[0] = (int) (fact* (-scalecy*(offy - 12.0 + 1.0 * i))) + xtk ;
           whys[0] = (int) (fact* (-scalecz*(offz  +
                  (dist + 4.0) * Math.sin(perspec)))) + ytk ;
           exes[1] = (int) (fact* (-scalecy*(offy - 12.0 + 1.0 * i))) + xtk ;
           whys[1] = (int) (fact* (-scalecz*(offz  + 8.0 +
                  dist * Math.sin(perspec)))) + ytk ;
           offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }
       for (i=0; i<= 8 ; ++ i) {
           exes[0] = (int) (fact* (-scalecy*(offy - 12.0))) + xtk ;
           whys[0] = (int) (fact* (-scalecz*(offz  + 1.0 * i +
                  dist * Math.sin(perspec)))) + ytk ;
           exes[1] = (int) (fact* (-scalecy*(offy + 12.0))) + xtk ;
           whys[1] = (int) (fact* (-scalecz*(offz  + 1.0 * i +
                  dist * Math.sin(perspec)))) + ytk ;
           offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       }

       if (ktype == 3) {   // target box for corner kick
          offsGg.setColor(Color.magenta) ;
          exes[0] = (int) (fact* (scalecy*(offy - 9.0 ))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz + 5.5 +
                           24.0 * Math.sin(perspec)))) + ytk ;
          exes[1] = (int) (fact* (scalecy*(offy + 15.0 ))) + xtk ;
          whys[1] = (int) (fact* (-scalecz*(offz + 5.5 +
                           24.0 * Math.sin(perspec)))) + ytk ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalecy*(offy - 9.0 ))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz + 8.0 +
                           24.0 * Math.sin(perspec)))) + ytk ;
          exes[1] = (int) (fact* (scalecy*(offy + 15.0 ))) + xtk ;
          whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                           24.0 * Math.sin(perspec)))) + ytk ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalecy*(offy - 9.0 ))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz + 5.5 +
                           24.0 * Math.sin(perspec)))) + ytk ;
          exes[1] = (int) (fact* (scalecy*(offy - 9.0 ))) + xtk ;
          whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                           24.0 * Math.sin(perspec)))) + ytk ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalecy*(offy + 15.0 ))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz + 5.5 +
                           24.0 * Math.sin(perspec)))) + ytk ;
          exes[1] = (int) (fact* (scalecy*(offy + 15.0 ))) + xtk ;
          whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                           24.0 * Math.sin(perspec)))) + ytk ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalecy*(offy - 9.0 ))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz + 5.5 +
                           30.0 * Math.sin(perspec)))) + ytk ;
          exes[1] = (int) (fact* (scalecy*(offy + 15.0 ))) + xtk ;
          whys[1] = (int) (fact* (-scalecz*(offz + 5.5 +
                           30.0 * Math.sin(perspec)))) + ytk ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalecy*(offy - 9.0 ))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz + 8.0 +
                           30.0 * Math.sin(perspec)))) + ytk ;
          exes[1] = (int) (fact* (scalecy*(offy + 15.0 ))) + xtk ;
          whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                           30.0 * Math.sin(perspec)))) + ytk ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalecy*(offy - 9.0 ))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz + 5.5 +
                           30.0 * Math.sin(perspec)))) + ytk ;
          exes[1] = (int) (fact* (scalecy*(offy - 9.0 ))) + xtk ;
          whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                           30.0 * Math.sin(perspec)))) + ytk ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalecy*(offy + 15.0 ))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz + 5.5 +
                           30.0 * Math.sin(perspec)))) + ytk ;
          exes[1] = (int) (fact* (scalecy*(offy + 15.0 ))) + xtk ;
          whys[1] = (int) (fact* (-scalecz*(offz + 8.0 +
                           30.0 * Math.sin(perspec)))) + ytk ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

       }

 // ball 
      if (comp == 0) {
        offsGg.setColor(Color.white) ;
        offsGg.fillOval((int) (fact*(-scalecy*(offy-relsy))) + xtk -5,
             (int) (fact*(-scalecz*(offz+relsz
                       - relsx*Math.sin(perspec)))) + ytk - 5,10,10);
      }
   // saved kicks
       if (nsav >= 1) {
         for (i=0; i<= nsav-1; ++ i) {
            exes[0] = (int) (fact* (-scalecy*(offy-posy[i][0]))) + xtk ;
            whys[0] = (int) (fact* (-scalecz*(offz+posz[i][0] -
                          posx[i][0]* Math.sin(perspec)))) + ytk ;
            if(i==0) offsGg.setColor(Color.yellow) ;
            if(i==1) offsGg.setColor(Color.orange) ;
            if(i==2) offsGg.setColor(Color.red) ;
            if(i==3) offsGg.setColor(Color.magenta) ;
            if(i==4) offsGg.setColor(Color.orange) ;
            offsGg.fillOval(exes[0],whys[0],5,5);
            nump = npts[i] ;
            for (j=1; j<=nump; ++j){
               exes[1] = exes[0] ;
               whys[1] = whys[0] ;
               exes[0] = (int) (fact*(-scalecy*(offy-posy[i][j]))) + xtk ;
               whys[0] = (int) (fact*(-scalecz*(offz+posz[i][j] -
                        posx[i][j]*Math.sin(perspec)))) + ytk ;
               offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
               offsGg.fillOval(exes[0]-2,whys[0]-2,5,5);
            }
         }
       }
    // kick
       if (pitch == 1) {
          exes[0] = (int) (fact* (-scalecy*(offy-posy[nsav][0]))) + xtk ;
          whys[0] = (int) (fact* (-scalecz*(offz+posz[nsav][0] -
                      posx[nsav][0] * Math.sin(perspec)))) + ytk ;
          offsGg.setColor(Color.white) ;
          for (i=1; i<=nptb; ++i) {                /* plot trajectory */
             exes[1] = exes[0] ;
             whys[1] = whys[0] ;
             exes[0] = (int) (fact*(-scalecy*(offy-posy[nsav][i]))) + xtk ;
             whys[0] = (int) (fact*(-scalecz*(offz+posz[nsav][i] -
                       posx[nsav][i] * Math.sin(perspec)))) + ytk;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             offsGg.fillOval(exes[0]-2,whys[0]-2,5,5);
          }
          offsGg.fillOval((int) (fact*(-scalecy*(offy-posy[nsav][nptb]))) +  xtk -5,
             (int) (fact*(-scalecz*(offz+posz[nsav][nptb] -
                     posx[nsav][nptb] * Math.sin(perspec)))) + ytk  - 5,10,10);
       }
//Buttons 
         offsGg.setColor(Color.yellow) ;
         offsGg.fillRect(0,100,100,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(0,100,100,25) ;
         offsGg.drawString("Downfield View",10,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(110,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(110,100,80,25) ;
         offsGg.drawString("Top View",120,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(200,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(200,100,80,25) ;
         offsGg.drawString("Side View",210,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(290,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(290,100,80,25) ;
         offsGg.drawString("TV View",300,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(380,100,70,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(380,100,70,25) ;
         offsGg.drawString("Find",390,120) ;
    }
  //  Side View
     if(drawview == 4) {
         perspec = 2.0 * convdr ;
         offsGg.setColor(col1) ;
         offsGg.fillRect(0,0,500,600) ;
 // Playing Field
                 // goal line
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy - 
                           120.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx - dist))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy + 
                           120.0  * Math.sin(perspec)))) + yts ;
          offsGg.setColor(Color.white) ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 // side lines
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy - 
                           120.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx -dist + 360.0))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy - 
                           120.0 * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy + 
                           120.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx -dist + 360.0))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy + 
                           120.0  * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              // center line
          exes[0] = (int) (fact* (scalex*(offx - dist + 180.0 ))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy - 
                           120.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx -dist + 180.0))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy + 
                           120.0  * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

                // center circle
/*
          exes[0] = (int) (fact* (scalex*(offx - dist + 150.0 ))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy + 30.0))) + yts ;
          exes[1] = (int) (fact* (scalex*(60.0))) ;
          whys[1] = (int) (fact* (scaley*(60.0)));
          offsGg.drawArc(exes[0],whys[0],exes[1],whys[1],0,360) ;
*/
                // goal box
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy - 
                          30.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx - dist + 18.0))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy - 
                          30.0  * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx - dist + 18.0))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy + 
                          30.0 * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[1] = (int) (fact* (scalex*(offx - dist))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy + 
                          30.0 * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                // penalty box
          exes[0] = (int) (fact* (scalex*(offx - dist))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy - 
                          66.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx -dist + 54.0))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy - 
                          66.0  * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx -dist + 54.0))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy + 
                          66.0 * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[1] = (int) (fact* (scalex*(offx - dist))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy + 
                          66.0 * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 // penalty circle
/*
          exes[0] = (int) (fact* (scalex*(offx -dist + 6.0))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy + 30.0))) + yts ;
          exes[1] = (int) (fact* (scalex*(60.0))) ;
          whys[1] = (int) (fact* (scaley*(60.0)));
          offsGg.drawArc(exes[0],whys[0],exes[1],whys[1],-53,106) ;
*/
               // goal net
          offsGg.setColor(Color.yellow) ;
          exes[0] = (int) (fact* (scalex*(offx ))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy - 
                                12.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx - 8.0))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy - 
                                12.0 * Math.sin(perspec) ))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy + 8.0 - 
                                12.0  * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx ))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy + 
                                12.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx - 8.0))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy + 
                                12.0 * Math.sin(perspec) ))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy + 8.0 + 
                                12.0  * Math.sin(perspec)))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = (int) (fact* (scalex*(offx - 8.0))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy - 
                                12.0 * Math.sin(perspec) ))) + yts ;
          exes[1] = (int) (fact* (scalex*(offx - 8.0))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offy + 
                                12.0 * Math.sin(perspec) ))) + yts ;
          offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          for( i=0; i<=8; ++ i) {
             exes[0] = (int) (fact* (scalex*(offx -8.0 + 1.0*i ))) + xts ;
             whys[0] = (int) (fact* (-scaley*(offy +
                          12.0*Math.sin(perspec)))) + yts ;
             exes[1] = (int) (fact* (scalex*(offx))) + xts ;
             whys[1] = (int) (fact* (-scaley*(offy +8.0 +
                          12.0*Math.sin(perspec)))) + yts ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
          for( i=0; i<=8; ++ i) {
             exes[0] = (int) (fact* (scalex*(offx -8.0 + 1.0*i ))) + xts ;
             whys[0] = (int) (fact* (-scaley*(offy -
                          12.0*Math.sin(perspec)))) + yts ;
             exes[1] = (int) (fact* (scalex*(offx))) + xts ;
             whys[1] = (int) (fact* (-scaley*(offy +8.0 -
                          12.0*Math.sin(perspec)))) + yts ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
          for( i=0; i<=8; ++ i) {
             exes[0] = (int) (fact* (scalex*(offx - 8.0 + 1.0 * i))) + xts ;
             whys[0] = (int) (fact* (-scaley*(offz + 1.0 * i -
                         12.0*Math.sin(perspec)))) + yts ;
             exes[1] = (int) (fact* (scalex*(offx))) + xtv ;
             whys[1] = (int) (fact* (-scalez*(offz + 1.0 * i -
                         12.0*Math.sin(perspec)))) + ytv ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
          for( i=0; i<=8; ++ i) {
             exes[0] = (int) (fact* (scalex*(offx - 8.0 + 1.0 * i))) + xts ;
             whys[0] = (int) (fact* (-scaley*(offz + 1.0 * i +
                         12.0*Math.sin(perspec)))) + yts ;
             exes[1] = (int) (fact* (scalex*(offx))) + xtv ;
             whys[1] = (int) (fact* (-scalez*(offz + 1.0 * i +
                         12.0*Math.sin(perspec)))) + ytv ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
              // posts
          offsGg.setColor(Color.yellow) ;
          exes[0] = (int) (fact* (-scalex*(offx ))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy -
                      12.0 * Math.sin( perspec)))) + yts ;
          exes[1] = (int) (fact* (-scalex*(offx ))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offz + 8.0 -
                      12.0 * Math.sin(perspec)))) + yts ;
          hite = (whys[0] - whys[1]) ;
          wdth = 5 ; 
          offsGg.fillRect(exes[0],whys[1],wdth,hite) ;
          exes[0] = (int) (fact* (-scalex*(offx ))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offy +
                      12.0 * Math.sin( perspec)))) + yts ;
          exes[1] = (int) (fact* (-scalex*(offx ))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offz + 8.0 +
                      12.0 * Math.sin(perspec)))) + yts ;
          hite = (whys[0] - whys[1]) ;
          wdth = 5 ; 
          offsGg.fillRect(exes[0],whys[1],wdth,hite) ;
          offsGg.setColor(Color.orange) ;
          exes[0] = (int) (fact* (-scalex*(offx ))) + xts ;
          whys[0] = (int) (fact* (-scaley*(offz + 8.0 -
                      12.0 * Math.sin(perspec)))) + yts ;
          exes[1] = (int) (fact* (-scalex*(offx ))) + xts ;
          whys[1] = (int) (fact* (-scaley*(offz + 8.0 +
                      12.0 * Math.sin(perspec)))) + yts ;
          hite = (whys[0] - whys[1]) ;
          offsGg.fillRect(exes[1],whys[1],wdth,hite) ;

          if (ktype == 3) {   // target box for corner kick
             offsGg.setColor(Color.magenta) ;
             exes[0] = (int) (fact* (scalex*(offx + 24.0 ))) + xts ;
             whys[0] = (int) (fact* (-scaley*(offz + 5.5 +
                           (-9.0) * Math.sin(perspec)))) + yts ;
             exes[1] = (int) (fact* (scalex*(offx + 30.0 ))) + xts ;
             whys[1] = (int) (fact* (-scaley*(offz + 5.5 +
                           (-9.0) * Math.sin(perspec)))) + yts ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             exes[0] = (int) (fact* (scalex*(offx + 24.0 ))) + xts ;
             whys[0] = (int) (fact* (-scaley*(offz + 8.0 +
                           (-9.0) * Math.sin(perspec)))) + yts ;
             exes[1] = (int) (fact* (scalex*(offx + 30.0 ))) + xts ;
             whys[1] = (int) (fact* (-scaley*(offz + 8.0 +
                           (-9.0) * Math.sin(perspec)))) + yts ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             exes[0] = (int) (fact* (scalex*(offx + 24.0 ))) + xts ;
             whys[0] = (int) (fact* (-scaley*(offz + 5.5 +
                           (-9.0) * Math.sin(perspec)))) + yts ;
             exes[1] = (int) (fact* (scalex*(offx + 24.0 ))) + xts ;
             whys[1] = (int) (fact* (-scaley*(offz + 8.0 +
                           (-9.0) * Math.sin(perspec)))) + yts ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             exes[0] = (int) (fact* (scalex*(offx + 30.0 ))) + xts ;
             whys[0] = (int) (fact* (-scaley*(offz + 5.5 +
                           (-9.0) * Math.sin(perspec)))) + yts ;
             exes[1] = (int) (fact* (scalex*(offx + 30.0 ))) + xts ;
             whys[1] = (int) (fact* (-scaley*(offz + 8.0 +
                           (-9.0) * Math.sin(perspec)))) + yts ;
             offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          }

 // ball and vector
         if (comp == 0) {
            offsGg.fillOval((int) (fact*scalex*(offx+relsx)) + xts,
             (int) (fact*(-scaley*(offy+relsz +
                          relsy * Math.sin(perspec)))) + yts - 2,5,5);
            exes[0] = (int) (fact* (scalex*(offx+relsx))) + xts ;
            whys[0] = (int) (fact* (-scaley*(offy+relsz +
                          relsy * Math.sin(perspec)))) + yts ;
            length = 10.0 * (vfsd / vmaxa) ;
            exes[1] = (int) (fact* (scalex*(offx+relsx - 
                     length * Math.cos(convdr*zangr)))) + xts ;
            whys[1] = (int) (fact* (-scaley*(offy+relsz +
                     relsy * Math.sin(perspec) +
                     length * Math.sin(convdr*zangr)))) + yts ;
            offsGg.setColor(Color.cyan) ;
            offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
            offsGg.setColor(Color.white) ;
            offsGg.fillOval((int) (fact*scalex*(offx+relsx )) + xts ,
             (int) (fact*(-scaley*(offy+relsz +
                      relsy * Math.sin(perspec)))) + yts - 5,10,10);
            exes[0] = (int) (fact* (scalex*(offx+relsx + 
                     0.5 * Math.cos(convdr*(plthg1[1] + 180.))))) + xts + 5 ;
            whys[0] = (int) (fact* (-scaley*(offy+relsz +
                      relsy * Math.sin(perspec) +
                     0.5 * Math.sin(convdr*(plthg1[1] + 180.))))) + yts ;
            exes[1] = (int) (fact* (scalex*(offx+relsx + 
                     0.5 * Math.cos(convdr*plthg1[1])))) + xts + 5 ;
            whys[1] = (int) (fact* (-scaley*(offy+relsz +
                      relsy * Math.sin(perspec) +
                     0.5 * Math.sin(convdr*plthg1[1])))) + yts ;
            offsGg.setColor(Color.red) ;
            offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
        }
 // animated kick
   // saved kicks
         if (nsav >= 1) {
           for (i=0; i<= nsav-1; ++ i) {
              exes[0] = (int) (fact* (scalex*(offx+posx[i][0]))) + xts ;
              whys[0] = (int) (fact* (-scaley*(offy+
                         posy[i][0]*Math.sin(perspec) +
                         posz[i][0]))) + yts ;
              if(i==0) {
                 offsGg.setColor(Color.yellow) ;
//                 offsGg.drawString("inches",350,195) ;
              }
              if(i==1) offsGg.setColor(Color.orange) ;
              if(i==2) offsGg.setColor(Color.red) ;
              if(i==3) offsGg.setColor(Color.magenta) ;
              if(i==4) offsGg.setColor(Color.orange) ;
              offsGg.fillOval(exes[0],whys[0],5,5);
              nump = npts[i] ;
              for (j=1; j<=nump; ++j){
                 exes[1] = exes[0] ;
                 whys[1] = whys[0] ;
                 exes[0] = (int) (fact*scalex*(offx+posx[i][j])) + xts ;
                 whys[0] = (int) (fact*(-scaley*(offy+
                                    posy[i][j]*Math.sin(perspec) +
                                    posz[i][j]))) + yts ;
                 offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 offsGg.fillOval(exes[0],whys[0]-2,5,5);
              }
//              offsGg.drawString(String.valueOf(filter1(yoff[i])),320,195+i*15) ;
           }
         }

         if (pitch == 1) {
            exes[0] = (int) (fact* (scalex*(offx+posx[nsav][0]))) + xts ;
            whys[0] = (int) (fact* (-scaley*(offy+
                                    posy[nsav][0]*Math.sin(perspec) +
                                    posz[nsav][0]))) + yts ;
            offsGg.setColor(Color.white) ;
            for (i=1; i<=nptb; ++i) {                /* plot trajectory */
               exes[1] = exes[0] ;
               whys[1] = whys[0] ;
               exes[0] = (int) (fact*scalex*(offx+posx[nsav][i])) + xts ;
               whys[0] = (int) (fact*(-scaley*(offy+
                                       posy[nsav][i]*Math.sin(perspec) +
                                       posz[nsav][i]))) + yts ;
               offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
               offsGg.fillOval(exes[0],whys[0]-2,5,5);
            }
            offsGg.setColor(Color.white) ;
            offsGg.fillOval((int) (fact*scalex*(offx+posx[nsav][nptb])) + xts ,
             (int) (fact*(-scaley*(offy+
                                    posy[nsav][nptb]*Math.sin(perspec) +
                                    posz[nsav][nptb]))) + yts - 5,10,10);
         }
//Buttons 
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(0,100,100,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(0,100,100,25) ;
         offsGg.drawString("Downfield View",10,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(110,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(110,100,80,25) ;
         offsGg.drawString("Top View",120,120) ;
         offsGg.setColor(Color.yellow) ;
         offsGg.fillRect(200,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(200,100,80,25) ;
         offsGg.drawString("Side View",210,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(290,100,80,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(290,100,80,25) ;
         offsGg.drawString("TV View",300,120) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillRect(380,100,70,25) ;
         offsGg.setColor(Color.black) ;
         offsGg.drawRect(380,100,70,25) ;
         offsGg.drawString("Find",390,120) ;
      }

 //  Labels

  // spin axis graphic
       offsGg.setColor(Color.white) ;
       xpic = 280;
       xpicg = 300;
       ypic = 30 ;
       offsGg.fillRect(100,0,400,100) ;

       exes[0] = xpic + 54 ;
       whys[0] = ypic + 36 ;
       exes[1] = xpic + 54 + (int)(50*Math.sin(convdr*sangr)) ; 
       whys[1] = ypic + 36 - (int)(50*Math.cos(convdr*sangr)) ;
       offsGg.setColor(Color.black) ;
       offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       offsGg.drawString("Bend",320,10) ;
       offsGg.setColor(Color.blue) ;
       offsGg.drawString("Spin Axis",370,30) ;
       offsGg.setColor(Color.black) ;
       if (spindr <= 0.0) {
          if (spin >= 0.0) {
             offsGg.drawString("Dive",240,70) ;
             offsGg.drawString("Lift",390,70) ;
          }
          if (spin < 0.0) {
             offsGg.drawString("Lift",240,70) ;
             offsGg.drawString("Dive",390,70) ;
          }
       }
       if (spindr >= 0.0) {
          if (spin >= 0.0) {
             offsGg.drawString("Lift",240,70) ;
             offsGg.drawString("Dive",390,70) ;
          }
          if (spin < 0.0) {
             offsGg.drawString("Dive",240,70) ;
             offsGg.drawString("Lift",390,70) ;
          }
       }
       offsGg.drawImage(displimg,xpicg,ypic,this) ;
//  outcome
       offsGg.setColor(Color.blue) ;
       offsGg.drawString("SoccerNASA",120,10) ;
       offsGg.drawString("Version 1.4c",120,30) ;
       if (call == 1) {
         if (zoff[nsav] <= 8.0 && 
             (yoff[nsav] <= 12.0 && yoff[nsav] >= -12.0 )) {
            if(antim1 == 0) {
               offsGg.setColor(Color.red) ;
               offsGg.fillRect(110,50,90,30) ;
               offsGg.setColor(Color.white) ;
               offsGg.drawString("GOAL !",140,70) ;
            }
            if(antim1 == 1) {
               offsGg.setColor(Color.red) ;
               offsGg.fillRect(110,50,90,30) ;
               offsGg.setColor(Color.white) ;
               offsGg.drawString("GOAL !",140,70) ;
            }
         }
       }
       if (call == 3) {       // hit target box on corner kick
            if(antim1 == 0) {
               offsGg.setColor(Color.blue) ;
               offsGg.fillRect(110,50,90,30) ;
               offsGg.setColor(Color.white) ;
               offsGg.drawString("GOOD",140,70) ;
            }
            if(antim1 == 1) {
               offsGg.setColor(Color.blue) ;
               offsGg.fillRect(110,50,90,30) ;
               offsGg.setColor(Color.white) ;
               offsGg.drawString("GOOD",140,70) ;
            }
       }

//      nasa meatball graphic
       offsGg.setColor(Color.white) ;
       offsGg.fillRect(0,0,100,100) ;
       offsGg.drawImage(nasaimg,10,10,this) ;

 // zoom widget
        offsGg.setColor(Color.yellow) ;
        offsGg.drawString("Zoom",2,530) ;
        offsGg.drawLine(50,525,440,525) ;
        offsGg.fillRect(sldloc,515,5,20) ;

       g.drawImage(offscreenImg,0,0,this) ;   
     }
   }
}
