/*
                      FoilSim II  - Baseball  mode
   
                           A Java Applet
               to perform Kutta-Joukowski Airfoil analysis

                     Version 1.5c   - 8 May 12

                             Written by 
                            Tom Benson
                       NASA Glenn Research Center

                             Peter Sherman
                       Lakeside School, Seattle, WA

>                              NOTICE
>This software is in the Public Domain.  It may be freely copied and used in
>non-commercial products, assuming proper credit to the author is given.  IT
>MAY NOT BE RESOLD.  If you want to use the software for commercial
>products, contact the author.
>No copyright is claimed in the United States under Title 17, U. S. Code.
>This software is provided "as is" without any warranty of any kind, either
>express, implied, or statutory, including, but not limited to, any warranty
>that the software will conform to specifications, any implied warranties of
>merchantability, fitness for a particular purpose, and freedom from
>infringement, and any warranty that the documentation will conform to the
>program, or any warranty that the software will be error free.
>In no event shall NASA be liable for any damages, including, but not
>limited to direct, indirect, special or consequential damages, arising out
>of, resulting from, or in any way connected with this software, whether or
>not based on warranty, contract, tort or otherwise, whether or not injury
>was sustained by persons or property or otherwise, and whether or not loss
>was sustained from, or arose out of the results of, or use of, the software
>or services provided hereunder.
 
  New test -
             * add second graphics
             * re-organize input panels
             * correct ball size   (rball = .12)
             * improve analysis    (integrated cylinder)
             * add lift coefficient  (Cl = .15)
             * improve graphics
             * add softball
             * add any ball
             * add softball diamond
             *   make lpit a global variable
               add slow pitch option
 
                                           TJB  8 May 12

*/

import java.awt.*;
import java.lang.Math ;

public class Ball extends java.applet.Applet {
 
   static double convdr = 3.1415926/180. ;
   static double pid2 = 3.1415926/2.0 ;
   static double rval,ycval,xcval,gamval,alfval,thkval,camval,chrd,cl ;
   static double thkinpt,caminpt ;                 /* MODS 10 Sep 99 */
   static double leg,teg,lem,tem;
   static double usq,vsq,alt,altmax,area,armax,armin ;
   static double chord,span,aspr,arold,chrdold,spnold ; /* Mod 13 Jan 00 */
   static double q0,ps0,pt0,ts0,rho,rlhum,temf,presm ;
   static double lyg,lrg,lthg,lxgt,lygt,lrgt,lthgt;/* MOD 20 Jul */
   static double lxm,lym,lxmt,lymt,vxdir;/* MOD 20 Jul */
   static double deltb,xflow ;             /* MODS  20 Jul 99 */
   static double delx,delt,vfsd,spin,spindr,yoff ;
   static double vel,pres,lift,side,omega,radcrv,relsy,angr,relsdr ;
   static double rball,mball,diameter,weight,lpit ;
   int balflag ;

   static double rg[][]  = new double[20][40] ; 
   static double thg[][] = new double[20][40] ; 
   static double xg[][]  = new double[20][40] ; 
   static double yg[][]  = new double[20][40] ; 
   static double xm[][]  = new double[20][40] ; 
   static double ym[][]  = new double[20][40] ; 

   int nptc,npt2,nlnc,nln2,rdflag,browflag;
   int foil,flflag,lunits,lftout,planet,wthopt,place ;
   int conflag,displ,antim1,ancol1,antim2,ancol2; 
       /* units data */
   static double vmn1,vmn2,vmn3,vmn4,vmx1,vmx2,vmx3,vmx4 ;
   static double vconv,vmaxa,vmaxb ;
   static double pconv,pmax,pmin,lconv,fconv,fmax,fmaxb;
   int lflag,gflag,plscale,nond;
       /*  plot & probe data */
   static double fact,fact2,xpval,ypval,pbval;
   static double prg,pthg,pxg,pyg,pxm,pym ;
   int ptchflag,pboflag,xt,yt,xt2,yt2,ntikx,ntiky,npt,nptb ;
   int lines,nord,nabs,ntr,pitch ;
   static double begx,endx,begy,endy ;
   static String labx,labxu,laby,labyu ;
   static double pltx[]  = new double[75] ;
   static double plty[]  = new double[75] ;
   static double pltrg[]  = new double[2] ;
   static double plthg1[]  = new double[2] ;
   static double plthg2[]  = new double[2] ;
   static double pltxg[]  = new double[2] ;
   static double pltyg[]  = new double[2] ;

   Solver solve ;
   View1 view1 ;
   View2 view2 ;
   M m ;
   L l ;
   Image offscreenImg ;
   Graphics offsGg ;
   Image offImg3 ;
   Graphics off3Gg ;
   Image offImg4 ;
   Graphics off4Gg ;

   public void init() {
     int i;
     Ball a = new Ball() ;
     solve = new Solver() ;

     offscreenImg = createImage(this.size().width,
                      this.size().height) ;
     offsGg = offscreenImg.getGraphics() ;
     offImg3 = createImage(this.size().width,
                      this.size().height) ;
     off3Gg = offImg3.getGraphics() ;
     offImg4 = createImage(this.size().width,
                      this.size().height) ;
     off4Gg = offImg4.getGraphics() ;

     setLayout(new GridLayout(4,1,5,5)) ;

     solve.setDefaults () ;
 
     view1  = new View1(this) ;
     view2  = new View2(this) ;
     m = new M(this) ;
     l = new L(this) ;

     add(m) ;
     add(view1) ;
     add(view2) ;
     add(l) ;

     solve.getFreeStream ();
     computeFlow () ;
     view1.start() ;
     view2.start() ;
  }
 
  public Insets insets() {
     return new Insets(10,10,10,10) ;
  }

  public void computeFlow() { 

     if (flflag == 1) {
         solve.getCirc ();                   /* get circulation */
         solve.genFlow () ;
         solve.getFreeStream () ;
     }
 
     solve.getProbe() ;
 
     loadOut() ;
     solve.computePitch() ;

     view1.loadPlot() ;
     view2.loadPlot() ;

  }

  public int filter0(double inumbr) {
        //  output only to .
       int number ;
       int intermed ;
 
       number = (int) (inumbr);
       return number ;
  }

  public float filter3(double inumbr) {
     //  output only to .001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 1000.) ;
       number = (float) (intermed / 1000. );
       return number ;
  }
 
  public float filter5(double inumbr) {
     //  output only to .00001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 100000.) ;
       number = (float) (intermed / 100000. );
       return number ;
  }
 
  public void setUnits() {   // Switching Units
       double ovs,chords,spans,aros,chos,spos ;
       double alts,ares ;

       alts = alt / lconv ;
       chords = chord / lconv ;
       spans = span / lconv ;
       ares = area /lconv/lconv ;
       aros = arold /lconv/lconv ;
       chos = chrdold / lconv ;
       spos = spnold / lconv ;
       ovs = vfsd / vconv ;

       switch (lunits) {
          case 0: {                             /* English */
            lconv = 1.;                      /*  feet    */
            vconv = .6818; vmaxa = 100.; vmaxb = 100. ;  /*  mph  */
            fconv = 1.0; fmax = 100000.; fmaxb = .5;  /* pounds   */
            pconv = 14.7  ;                   /* lb/sq in */
            break;
          }
          case 1: {                             /* Metric */
            lconv = .3048;                    /* meters */
            vconv = 1.097;  vmaxa = 167.; vmaxb = 167.;   /* km/hr  */
            fconv = 4.448 ; fmax = 500000.; fmaxb = 2.5; /* newtons */
            pconv = 101.3 ;               /* kilo-pascals */
            break ;
          }
       }
 
       alt = alts * lconv ;
       chord = chords * lconv ;
       span = spans * lconv ;
       area = ares * lconv * lconv ;
       arold = aros * lconv * lconv ;
       chrdold = chos * lconv ;
       spnold = spos * lconv ;
       vfsd  = ovs * vconv;

       return ;
  }

  public void loadProb() {   // load the input panel
       int i1,i2,i3,i4 ;
       double v1,v2,v3,v4 ;
       float fl1,fl2,fl3,fl4 ;

       v1 = 0.0 ;
       v2 = 0.0 ;
       v3 = 0.0 ;
       v4 = 0.0 ;

       switch (lunits)  {
          case 0: {                             /* English */
              m.mdl.mdll.o1.setText(String.valueOf(
                                          filter3(presm))) ;
              m.mdl.mdll.lu1.setText("in Hg") ;
              m.mdl.mdll.o2.setText(String.valueOf(
                                         filter0(ts0 - 460.))) ;
              m.mdl.mdll.lu2.setText(" F ") ;
              m.mdl.mdll.o3.setText(String.valueOf(
                                          filter0(rlhum * 100.))) ;
              m.mdl.mdll.o4.setText(String.valueOf(
                                         filter0(alt))) ;
              m.mdl.mdll.lu4.setText("ft ") ;
              break;
           }
           case 1: {                             /* Metric */
              m.mdl.mdll.o1.setText(String.valueOf(
                                          filter3(101.3/14.7*ps0/144.))) ;
              m.mdl.mdll.lu1.setText("k-Pa") ;
              m.mdl.mdll.o2.setText(String.valueOf(
                                         filter0(ts0*5.0/9.0 - 273.1))) ;
              m.mdl.mdll.lu2.setText(" C ") ;
              m.mdl.mdll.o3.setText(String.valueOf(
                                          filter0(rlhum*100.))) ;
              m.mdl.mdll.o4.setText(String.valueOf(
                                         filter0(alt))) ;
              m.mdl.mdll.lu4.setText("m ") ;
              break ;
           }
       }

       v1 = vfsd ;
       vmn1 = 30.0;   vmx1= vmaxa ;
       v2 = spin * 60.0 ;
       vmn2 = -2000.0; vmx2 = 2000.0 ;
       v3 = relsy * lconv / relsdr ;
       vmn3 = 0.0;   vmx3 = 4.0 * lconv ;
       v4 = angr ;
       vmn4 = -15.0;   vmx4 = 15.0 ;

       fl1 = (float) v1 ;
       fl2 = (float) v2 ;
       fl3 = (float) v3 ;
       fl4 = (float) v4 ;

       l.lol1.f1.setText(String.valueOf(fl1)) ;
       l.lol1.f2.setText(String.valueOf(fl2)) ;
       l.lol1.f3.setText(String.valueOf(fl3)) ;
       l.lol1.f4.setText(String.valueOf(fl4)) ;

       i1 = (int) (((v1 - vmn1)/(vmx1-vmn1))*1000.) ;
       i2 = (int) (((v2 - vmn2)/(vmx2-vmn2))*1000.) ;
       i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;
       i4 = (int) (((v4 - vmn4)/(vmx4-vmn4))*1000.) ;

       l.lol2.s1.setValue(i1) ;
       l.lol2.s2.setValue(i2) ;
       l.lol2.s3.setValue(i3) ;
       l.lol2.s4.setValue(i4) ;

       computeFlow() ;
       return ;
  }

  public void loadOut() {   // output routine
      String outfor ;

      outfor = " oz" ;
      if (lunits == 1) outfor = " N" ;
                                                /* force on baseball */
      side = gamval * rho * (vfsd / vconv) * (vfsd / vconv) ; /* lift/ft */
      side = side * rball * 4.0/3.0 * .15 ; /*lift -> rball = .12-- new correction */
      side = side * fconv ;

      l.lor1.out.setText(String.valueOf(filter3(Math.abs(side * 16.0))) + outfor) ;

      if (conflag == 1) {
        m.mdl.mdll.o1.setBackground(Color.black) ;
        m.mdl.mdll.o1.setForeground(Color.yellow) ;
        m.mdl.mdll.o2.setBackground(Color.black) ;
        m.mdl.mdll.o2.setForeground(Color.yellow) ;
        m.mdl.mdll.o3.setBackground(Color.black) ;
        m.mdl.mdll.o3.setForeground(Color.yellow) ;
        switch (lunits)  {
          case 0: {                             /* English */
              m.mdl.mdll.o1.setText(String.valueOf(
                                          filter3(presm))) ;
              m.mdl.mdll.lu1.setText("in Hg") ;
              m.mdl.mdll.o2.setText(String.valueOf(
                                          filter0(ts0 - 460.))) ;
              m.mdl.mdll.lu2.setText(" F ") ;
              m.mdl.mdll.o3.setText(String.valueOf(
                                          filter0(rlhum * 100.))) ;
              m.mdl.mdll.o4.setText(String.valueOf(
                                          filter0(alt))) ;
              m.mdl.mdll.lu4.setText("ft ") ;
              break;
           }
           case 1: {                             /* Metric */
              m.mdl.mdll.o1.setText(String.valueOf(
                                          filter3(101.3/14.7*ps0/144.))) ;
              m.mdl.mdll.lu1.setText("k-Pa") ;
              m.mdl.mdll.o2.setText(String.valueOf(
                                          filter0(ts0*5.0/9.0 - 273.1))) ;
              m.mdl.mdll.lu2.setText(" C ") ;
              m.mdl.mdll.o3.setText(String.valueOf(
                                          filter0(rlhum*100.))) ;
              m.mdl.mdll.o4.setText(String.valueOf(
                                          filter0(alt))) ;
              m.mdl.mdll.lu4.setText("m ") ;
              break ;
           }
        }
     }
     return ;
  }

  public void loadProbe() {   // probe output routine

     pbval = 0.0 ;
     if (pboflag == 1) pbval = vel * vfsd ;           // velocity
     if (pboflag == 2) pbval = ((ps0 + pres * q0)/2116.) * pconv ; // pressure
 
     l.lor2.lor2a.repaint() ;
     return ;
  }

  class Solver {
 
     Solver () {
     }

     public void setDefaults() {

        planet = 0 ;
        wthopt = 0 ;
        place = 0 ;
        lunits = 0 ;
        lftout = 0 ;
        nlnc = 15 ;
        nln2 = nlnc/2 + 1 ;
        nptc = 37 ;
        npt2 = nptc/2 + 1 ;
        deltb = .5 ;
        foil = 0 ;
        flflag = 1;
        thkval = .5 ;
        thkinpt = 12.5 ;                   /* MODS 10 SEP 99 */
        camval = 0.0 ;
        caminpt = 0.0 ;
        alfval = 0.0 ;
        gamval = 0.0 ;
        spin = 0.0 ;
        spindr = -1.0 ;
        rval = 1.0 ;
        ycval = 0.0 ;
        xcval = 0.0 ;
        conflag = 1 ;                             /* MODS  2 Apr 99 */
        displ = 0 ;                              /* MODS  22 Apr 99 */
        relsy = 0.0 ;
        relsdr = 1.0 ;
        angr = 0.0 ;
        diameter = 2.88 ;
        rball = diameter / 24. ;
        weight = 5.0 ;
        mball = weight / 16.0 ;
        balflag = 0 ;
        lpit = 60.5 ;
 
        xpval = 2.1;
        ypval = -.5 ;
        pboflag = 0 ;
        xflow = -10.0;                             /* MODS  20 Jul 99 */

        pconv = 14.7;
        pmin = .5 ;
        pmax = 1.0 ;
        fconv = 1.0 ;
        fmax = 100000. ;
        fmaxb = .50 ;
        vconv = .6818 ;
        vfsd = 100. ;
        vmaxa = 100. ;
        vmaxb = 100. ;
        lconv = 1.0 ;

        alt = 700.0 ;
        altmax = 50000. ;
        chrdold = chord = 2.0 ;
        spnold = span = 10.0 ;
        aspr = 5.0 ;
        arold = area = 20.0 ;
        armax = 1000.01 ;
        armin = .01 ;                 /* MODS 9 SEP 99 */
        presm = 29.172 ;
        temf  = 59.0 ;
        ts0 = temf + 459.6 ;
        rlhum = 0.0 ;
 
        xt = 180 ;
        yt = 160 ;
        fact = 50. ;
        xt2 = 320 ;
        yt2 = 55 ;
        fact2 = 25. ;
        pltxg[1] = 0.0 ;
        pltyg[1] = 1.0 ;
        plthg1[1] = 0.0 ;
        plthg2[1] = 0.0 ;
        pltrg[1] = 1.0 ;
        begy=-10.; endy=10. ; ntiky=2 ;
        begx=-65. ; endx=0. ; ntikx=2;
        nptb = 1;
        pitch = 1 ;
 
        vmn1 = 30.0;     vmx1 = 100.0 ;
        vmn2 = -2000.0;   vmx2 = 2000.0 ;
        vmn3 = 0.0;   vmx3 = 4.0 ;
        vmn4 = -15.0;   vmx4 = 15.0 ;

        return ;
     }

     public void getFreeStream() {    //  free stream conditions
       double hite,pvap,rgas,gama ;       /* MODS  19 Jan 00  whole routine*/

       rgas = 1718. ;                /* ft2/sec2 R */
       gama = 1.4 ;
       hite = alt/lconv ;
       if (planet == 0) {    // Earth  standard day
         if (conflag == 1) {
           if (hite <= 36152.) {           // Troposphere
              ts0 = 518.6 - 3.56 * hite/1000. ;
              ps0 = 2116. * Math.pow(ts0/518.6,5.256) ;
           }
           if (hite >= 36152. && hite <= 82345.) {   // Stratosphere
              ts0 = 389.98 ;
              ps0 = 2116. * .2236 *
                 Math.exp((36000.-hite)/(53.35*389.98)) ;
           }
           if (hite >= 82345.) {
              ts0 = 389.98 + 1.645 * (hite-82345)/1000. ;
              ps0 = 2116. *.02456 * Math.pow(ts0/389.98,-11.388) ;
           }
           if (place == 0) {
             if(wthopt == 0) ts0 = 519.6 ;
             if(wthopt == 1) ts0 = 549.6 ;
             if(wthopt == 2) ts0 = 494.6 ;
           }
           rlhum = 0.0 ;
           temf = ts0 - 459.6 ;
           if (temf <= 0.0) temf = 0.0 ;                    
           presm = ps0 * 29.92 / 2116. ;
         }
         if (conflag == 2) {
            ts0 = temf + 459.6 ;
            if (temf < 0.0) {
                  temf = 0.0 ;
                  rlhum = 0.0 ;
            }
             ps0 = presm * 2116. / 29.92 ;
         }
         pvap = rlhum*(2.685+.00353*Math.pow(temf,2.245));/* Eq 1:6A  Domasch */
         rho = (ps0 - .379*pvap)/(rgas * ts0) ;  /* effect of humidty */
       }

       if (planet == 1) {   // Mars - curve fit of orbiter data
         rgas = 1149. ;                /* ft2/sec2 R */
         gama = 1.29 ;

         if (hite <= 22960.) {
            ts0 = 434.02 - .548 * hite/1000. ;
            ps0 = 14.62 * Math.pow(2.71828,-.00003 * hite) ;
         }
         if (hite > 22960.) {
            ts0 = 449.36 - 1.217 * hite/1000. ;
            ps0 = 14.62 * Math.pow(2.71828,-.00003 * hite) ;
         }
         rho = ps0/(rgas*ts0) ;
         presm = ps0 * 29.92 / 2116. ;
       }

       q0  = .5 * rho * vfsd * vfsd / (vconv * vconv) ;
       pt0 = ps0 + q0 ;

       return ;
     }

     public void getCirc() {   // circulation from Kutta condition
       double thet,rdm,thtm ;
       double beta;
       int index;

       xcval = 0.0 ;
       switch (foil)  {
          case 0: {         /* get circulation from spin for baseball */
              gamval = 4.0 * 3.1415926 * 3.1415926 *spin * rball * rball
                                 / (vfsd/vconv) ;
              gamval = gamval * spindr ;
              ycval = .0001 ;
              break ;
          }
          case 1:  {                  /* Juokowski geometry*/
              ycval = camval / 2.0 ;
              rval = thkval/4.0 +Math.sqrt(thkval*thkval/16.0+ycval*ycval +1.0);
              xcval = 1.0 - Math.sqrt(rval*rval - ycval*ycval) ;
              beta = Math.asin(ycval/rval)/convdr ;     /* Kutta condition */
              gamval = 2.0*rval*Math.sin((alfval+beta)*convdr) ;
              break ;
          }
          case 2:  {                  /* Elliptical geometry*/
              ycval = camval / 2.0 ;
              rval = thkval/4.0 + Math.sqrt(thkval*thkval/16.0+ycval*ycval+1.0);
              beta = Math.asin(ycval/rval)/convdr ;    /* Kutta condition */
              gamval = 2.0*rval*Math.sin((alfval+beta)*convdr) ;
              break ;
          }
       }
                                                   /* geometry */
       for (index =1; index <= nptc; ++index) {
           thet = (index -1)*360./(nptc-1) ;
           xg[0][index] = rval * Math.cos(convdr * thet) + xcval ;
           yg[0][index] = rval * Math.sin(convdr * thet) + ycval ;
           rg[0][index] = Math.sqrt(xg[0][index]*xg[0][index] +
                                    yg[0][index]*yg[0][index])  ;
           thg[0][index] = Math.atan2(yg[0][index],xg[0][index])/convdr;
           xm[0][index] = (rg[0][index] + 1.0/rg[0][index])*
                        Math.cos(convdr*thg[0][index]) ;
           ym[0][index] = (rg[0][index] - 1.0/rg[0][index])*
                        Math.sin(convdr*thg[0][index]) ;
           rdm = Math.sqrt(xm[0][index]*xm[0][index] +
                           ym[0][index]*ym[0][index])  ;
           thtm = Math.atan2(ym[0][index],xm[0][index])/convdr;
           xm[0][index] = rdm * Math.cos((thtm - alfval)*convdr);
           ym[0][index] = rdm * Math.sin((thtm - alfval)*convdr);
           getVel(rval,thet) ;
       }

       return ;
     }

     public void genFlow() {   // generate flowfield
       double rnew,thet,psv,fxg;
       int k,index;
                              /* all lines of flow  except stagnation line*/
       for (k=1; k<=nlnc; ++k) {
         psv = -.5*(nln2-1) + .5*(k-1) ;
         fxg = xflow ;
         for (index =1; index <=nptc; ++ index) {
           solve.getPoints (fxg,psv) ;
           xg[k][index]  = lxgt ;
           yg[k][index]  = lygt ;
           rg[k][index]  = lrgt ;
           thg[k][index] = lthgt ;
           xm[k][index]  = lxmt ;
           ym[k][index]  = lymt ;
           solve.getVel(lrg,lthg) ;
           fxg = fxg + vxdir*deltb ;
         }
       }
                                              /*  stagnation line */
       k = nln2 ;
       psv = 0.0 ;
                                              /*  incoming flow */
       for (index =1; index <= npt2; ++ index) {
           rnew = 10.0 - (10.0 - rval)*Math.sin(pid2*(index-1)/(npt2-1)) ;
           thet = Math.asin(.999*(psv - gamval*Math.log(rnew/rval))/
                                   (rnew - rval*rval/rnew)) ;
           fxg =  - rnew * Math.cos(thet) ;
           solve.getPoints (fxg,psv) ;
           xg[k][index]  = lxgt ;
           yg[k][index]  = lygt ;
           rg[k][index]  = lrgt ;
           thg[k][index] = lthgt ;
           xm[k][index]  = lxmt ;
           ym[k][index]  = lymt ;
       }
                                              /*  downstream flow */
       for (index = 1; index <= npt2; ++ index) {
           rnew = 10.0 + .01 - (10.0 - rval)*Math.cos(pid2*(index-1)/(npt2-1)) ;
           thet = Math.asin(.999*(psv - gamval*Math.log(rnew/rval))/
                                      (rnew - rval*rval/rnew)) ;
           fxg =   rnew * Math.cos(thet) ;
           solve.getPoints (fxg,psv) ;
           xg[k][npt2+index]  = lxgt ;
           yg[k][npt2+index]  = lygt ;
           rg[k][npt2+index]  = lrgt ;
           thg[k][npt2+index] = lthgt ;
           xm[k][npt2+index]  = lxmt ;
           ym[k][npt2+index]  = lymt ;
       }
                                              /*  stagnation point */
       xg[k][npt2]  = xcval ;
       yg[k][npt2]  = ycval ;
       rg[k][npt2]  = Math.sqrt(xcval*xcval+ycval*ycval) ;
       thg[k][npt2] = Math.atan2(ycval,xcval)/convdr ;
       xm[k][npt2]  = (xm[k][npt2+1] + xm[k][npt2-1])/2.0 ;
       ym[k][npt2]  = (ym[0][nptc/4+1] + ym[0][nptc/4*3+1])/2.0 ;
                                /*  compute lift coefficient */
       leg = xcval - Math.sqrt(rval*rval - ycval*ycval) ;
       teg = xcval + Math.sqrt(rval*rval - ycval*ycval) ;
       lem = leg + 1.0/leg ;
       tem = teg + 1.0/teg ;
       chrd = tem - lem ;
       cl = gamval*4.0*3.1415926/chrd ;

       return ;
     }

     public void getPoints(double fxg, double psv) {   // flow in x-psi
       double radm,thetm ;                /* MODS  20 Jul 99  whole routine*/
       double fnew,ynew,yold,rfac,deriv ;
       double xold,xnew,thet ;
       double rmin,rmax ;
       int iter,isign;
                       /* get variables in the generating plane */
                           /* iterate to find value of yg */
       ynew = 10.0 ;
       yold = 10.0 ;
       if (psv < 0.0) ynew = -10.0 ;
       if (Math.abs(psv) < .001 && alfval < 0.0) ynew = rval ;
       if (Math.abs(psv) < .001 && alfval >= 0.0) ynew = -rval ;
       fnew = 0.1 ;
       iter = 1 ;
       while (Math.abs(fnew) >= .00001 && iter < 25) {
           ++iter ;
           rfac = fxg*fxg + ynew*ynew ;
           if (rfac < rval*rval) rfac = rval*rval + .01 ;
           fnew = psv - ynew*(1.0 - rval*rval/rfac)
                  - gamval*Math.log(Math.sqrt(rfac)/rval) ;
           deriv = - (1.0 - rval*rval/rfac)
               - 2.0 * ynew*ynew*rval*rval/(rfac*rfac)
               - gamval * ynew / rfac ;
           yold = ynew ;
           ynew = yold  - .5*fnew/deriv ;
       }
       lyg = yold ;
                                     /* rotate for angle of attack */
       lrg = Math.sqrt(fxg*fxg + lyg*lyg) ;
       lthg = Math.atan2(lyg,fxg)/convdr ;
       lxgt = lrg * Math.cos(convdr*(lthg + alfval)) ;
       lygt = lrg * Math.sin(convdr*(lthg + alfval)) ;
                              /* translate cylinder to generate airfoil */
       lxgt = lxgt + xcval ;
       lygt = lygt + ycval ;
       lrgt = Math.sqrt(lxgt*lxgt + lygt*lygt) ;
       lthgt = Math.atan2(lygt,lxgt)/convdr ;
                               /*  Kutta-Joukowski mapping */
       lxm = (lrgt + 1.0/lrgt)*Math.cos(convdr*lthgt) ;
       lym = (lrgt - 1.0/lrgt)*Math.sin(convdr*lthgt) ;
                              /* tranforms for view fixed with free stream */
                /* take out rotation for angle of attack mapped and cylinder */
       radm = Math.sqrt(lxm*lxm+lym*lym) ;
       thetm = Math.atan2(lym,lxm)/convdr ;
       lxmt = radm*Math.cos(convdr*(thetm-alfval)) ;
       lymt = radm*Math.sin(convdr*(thetm-alfval)) ;

       lxgt = lxgt - xcval ;
       lygt = lygt - ycval ;
       lrgt = Math.sqrt(lxgt*lxgt + lygt*lygt)  ;
       lthgt = Math.atan2(lygt,lxgt)/convdr;
       lxgt = lrgt * Math.cos((lthgt - alfval)*convdr);
       lygt = lrgt * Math.sin((lthgt - alfval)*convdr);

       return ;
     }
 
     public void getVel(double radius, double theta) {  //velocity and pressure 
      double ur,uth,jake1,jake2,jakesq ;
      double xloc,yloc,thrad,alfrad ;

      thrad = convdr * theta ;
      alfrad = convdr * alfval ;
                                /* get x, y location in cylinder plane */
      xloc = radius * Math.cos(thrad) ;
      yloc = radius * Math.sin(thrad) ;
                                /* velocity in cylinder plane */
      ur  = Math.cos(thrad-alfrad)*(1.0-(rval*rval)/(radius*radius)) ;
      uth = -Math.sin(thrad-alfrad)*(1.0+(rval*rval)/(radius*radius))
                            - gamval/radius;
      usq = ur*ur + uth*uth ;
      vxdir = ur * Math.cos(thrad) - uth * Math.sin(thrad) ; // MODS  20 Jul 99 
                                /* translate to generate airfoil  */
      xloc = xloc + xcval ;
      yloc = yloc + ycval ;
                                   /* compute new radius-theta  */
      radius = Math.sqrt(xloc*xloc + yloc*yloc) ;
      thrad  = Math.atan2(yloc,xloc) ;
                                   /* compute Joukowski Jacobian  */
      jake1 = 1.0 - Math.cos(2.0*thrad)/(radius*radius) ;
      jake2 = Math.sin(2.0*thrad)/(radius*radius) ;
      jakesq = jake1*jake1 + jake2*jake2 ;
      if (Math.abs(jakesq) <= .01) jakesq = .01 ;  /* protection */
      vsq = usq / jakesq ;
          /* vel is velocity ratio - pres is coefficient  (p-p0)/q0   */
      if (foil > 0) {
           vel = Math.sqrt(vsq) ;
           pres = 1.0 - vsq ;
      }
      if (foil == 0) {
           vel = Math.sqrt(usq) ;
           pres = 1.0 - usq ;
      }
      return ;
    }

    public void getProbe () { /* all of the information needed for the probe */
      double prxg;
      int index;
                       /* get variables in the generating plane */
      if (Math.abs(ypval) < .01) ypval = .05 ;
      solve.getPoints (xpval,ypval) ;

      solve.getVel(lrg,lthg) ;
      loadProbe() ;

      pxg = lxgt ;
      pyg = lygt ;
      prg = lrgt ;
      pthg = lthgt ;
      pxm = lxmt ;
      pym = lymt ;
                                    /* smoke */
      if (pboflag == 3 ) {
        prxg = xpval ;
        for (index =1; index <=nptc; ++ index) {
          solve.getPoints (prxg,ypval) ;
          xg[19][index] = lxgt ;
          yg[19][index] = lygt ;
          rg[19][index] = lrgt ;
          thg[19][index] = lthgt ;
          xm[19][index] = lxmt ;
          ym[19][index] = lymt ;
          solve.getVel(lrg,lthg) ;
          prxg = prxg + vxdir*deltb ;
        }
      }
      return ;
    }

    public void computePitch ()  { // compute variables needed for pitch 
      double force,accel ;      /* MODS  2 Apr 99   whole routine */
      double xc,yc,del ;
      int ic ;

      force = side/fconv ;
      accel = force / mball * 32.2 ;
      if (Math.abs(accel) >= .0001 ) {
            omega = accel / (vfsd / vconv) ;
            radcrv = (vfsd*vfsd)/(vconv*vconv) / accel ;
            ptchflag = 1 ;
            yoff =12.0*Math.abs(
              relsy + lpit * Math.tan(angr*convdr) + (radcrv/Math.abs(radcrv))*
              (Math.abs(radcrv)-Math.sqrt(radcrv*radcrv-lpit*lpit)) );
      }
      else {
            omega  = 1.0 ;
            radcrv = 10000. ;
            ptchflag = 2 ;
            yoff = 12.0*Math.abs(relsy + lpit * Math.tan(angr*convdr))  ;
      }

      del = 85.0 / (vfsd / vconv) / .05 ;
      if (del > 99.) del = 99. ;
      npt = (int) del ;
      del = .05 ;
      xc = radcrv * Math.sin(-convdr*angr) ;
      yc = radcrv *(1.0 - Math.cos(-convdr*angr)) ;
      for (ic=1; ic <=npt; ++ic) {
         if (ptchflag == 1) {             /* curve ball */
              pltx[ic] = -xc-radcrv*Math.sin(omega*del*(ic-1) + convdr*angr); 
              plty[ic] = -yc + relsy +
                   radcrv*(1.0 - Math.cos(omega*del*(ic-1) + convdr*angr)) ;
         }
         if (ptchflag == 2) {             /* fast ball */
              pltx[ic] = -((vfsd/vconv)*del*(ic-1))*Math.cos(convdr*angr) ;
              plty[ic] = relsy +((vfsd/vconv)*del*(ic-1))*Math.sin(convdr*angr);
          }
      }
      return ;
    }
  }
 
     class M extends Panel {
        Ball outerparent ;
        Mdl mdl ;
        Mdr mdr ;
   
        M (Ball target) { 
          outerparent = target ;
          setLayout(new GridLayout(1,2,5,5)) ;
 
          mdl = new Mdl(outerparent) ;
          mdr = new Mdr(outerparent) ;

          add(mdr) ;  
          add(mdl) ;  
     }

     class Mdl extends Panel {
        Ball outerparent ;
        Mdlu mdlu ;
        Mdll mdll ;

        Mdl (Ball target) { 
          outerparent = target ;
          setLayout(new GridLayout(2,1,5,5)) ;
 
          mdll = new Mdll(outerparent) ;
          mdlu = new Mdlu(outerparent) ;

          add(mdlu) ;  
          add(mdll) ;  
        }

        class Mdlu extends Panel {
           Ball outerparent ;
           Choice placeh,weatch ;

           Mdlu (Ball target) { 
             outerparent = target ;
             setLayout(new GridLayout(2,2,0,0)) ;
    
             placeh = new Choice() ;
             placeh.addItem("Cleveland") ;
             placeh.addItem("Denver") ;
             placeh.addItem("New York") ;
             placeh.addItem("Chicago") ;
             placeh.addItem("Mt.Everest") ;
             placeh.addItem("Mars");
             placeh.select(0) ;
 
             weatch = new Choice() ;
             weatch.addItem("Average Day") ;
             weatch.addItem("Hot Day") ;
             weatch.addItem("Cold Day") ;
             weatch.addItem("Specify") ;
             weatch.select(0) ;

             add(new Label("Stadium Location :", Label.RIGHT)) ;
             add(placeh) ;

             add(new Label("Weather Conditions:", Label.RIGHT)) ;
             add(weatch) ;
           }

           public boolean action(Event evt, Object arg) {
              if(evt.target instanceof Choice) {
                  this.handleProb(arg) ;
                  return true ;
              }
              else return false ;
           }

           public void handleProb(Object obj) {
              int i2,i1 ;
    
              planet  = 0 ;
              i1  = weatch.getSelectedIndex() ;
              i2  = placeh.getSelectedIndex() ;
              switch (i2) {
                 case 0:{                            /* Cleveland */
                        place = 0 ;
                        alt = 700.0 * lconv ;
                        break ;
                 }
                 case 1:{                            /* Denver */
                        place = 0 ;
                        alt = 5280.0 * lconv ;
                        break ;
                 }
                 case 2:{                            /* New York */
                        place = 0 ;
                        alt = 0. * lconv ;
                        break ;
                 }
                 case 3:{                            /* Chicago */
                        place = 0 ;
                        alt = 586. * lconv ;
                        break ;
                 }
                 case 4:{                            /* Mt. Everest */
                        place = 1 ;
                        alt = 29028. * lconv ;
                        break ;
                 }
                 case 5:{                            /* Mars */
                        planet = 1 ;
                        alt = 0.0 * lconv ;
                        break ;
                 }
              }

              switch (i1) {
                 case 0:{                            /* Average Day */
                        conflag = 1 ;
                        wthopt = 0 ;
                        break ;
                 }
                 case 1:{                            /* Hot Day */
                        conflag = 1 ;
                        wthopt = 1 ;
                        break ;
                 }
                 case 2:{                            /* Cold Day */
                        conflag = 1 ;
                        wthopt = 2 ;
                        break ;
                 }
                 case 3:{                            /* Set Conditions */
                        conflag = 2 ;
                        m.mdl.mdll.o1.setBackground(Color.white) ;
                        m.mdl.mdll.o1.setForeground(Color.black) ;
                        m.mdl.mdll.o2.setBackground(Color.white) ;
                        m.mdl.mdll.o2.setForeground(Color.black) ;
                        m.mdl.mdll.o3.setBackground(Color.white) ;
                        m.mdl.mdll.o3.setForeground(Color.black) ;
                        break ;
                 }
              }

              computeFlow() ;
           }
        }

        class Mdll extends Panel {
           Ball outerparent ;
           TextField o1,o2,o3,o4 ;
           Label l1,l2,l3,l4 ;
           Label lu1,lu2,lu3,lu4 ;

           Mdll (Ball target) { 
             outerparent = target ;
             setLayout(new GridLayout(2,6,0,0)) ;
 
             l1 = new Label("Press", Label.CENTER) ;
             lu1 = new Label("in Hg", Label.CENTER) ;
             o1 = new TextField("29.172",5) ;
             o1.setBackground(Color.black) ;
             o1.setForeground(Color.yellow) ;
             l2 = new Label("Temp", Label.CENTER) ;
             lu2 = new Label("F", Label.CENTER) ;
             o2 = new TextField("56",5) ;
             o2.setBackground(Color.black) ;
             o2.setForeground(Color.yellow) ;
             l3 = new Label("Humid", Label.CENTER) ;
             lu3 = new Label("%-Rel", Label.CENTER) ;
             o3 = new TextField("0.0",5) ;
             o3.setBackground(Color.black) ;
             o3.setForeground(Color.yellow) ;
             l4 = new Label("Alt", Label.CENTER) ;
             lu4 = new Label("ft ", Label.CENTER) ;
             o4 = new TextField("700.0",5) ;
             o4.setBackground(Color.black) ;
             o4.setForeground(Color.yellow) ;
    
             add(l1) ;
             add(o1) ;
             add(lu1) ;
             add(l3) ;
             add(o3) ;
             add(lu3) ;
    
             add(l2) ;
             add(o2) ;
             add(lu2) ;
             add(l4) ;
             add(o4) ;
             add(lu4) ;
   
           }

           public boolean handleEvent(Event evt) {
             if(evt.id == Event.ACTION_EVENT) {
                this.handleText(evt) ;
                return true ;
             }
             else return false ;
           }
 
           public void handleText(Event evt) {
              Double V1,V2,V3 ;
              double v1,v2,v3 ;
              float fl1 ;

              V1 = Double.valueOf(o1.getText()) ;
              v1 = V1.doubleValue() ;
              V2 = Double.valueOf(o2.getText()) ;
              v2 = V2.doubleValue() ;
              V3 = Double.valueOf(o3.getText()) ;
              v3 = V3.doubleValue() ;
 
              if (conflag == 2) {
                 if(v1 < 5.0) {
                    v1 = 5.0 ;
                    fl1 = (float) v1 ;
                    o1.setText(String.valueOf(fl1)) ;
                 }
                 if(v1 > 50.0) {
                    v1 = 50.0 ;
                    fl1 = (float) v1 ;
                    o1.setText(String.valueOf(fl1)) ;
                 }
                 presm = v1 ;
                 temf = v2 ;
                 if(v3 < 1.0) {
                    v3 = 0.0 ;
                    fl1 = (float) v3 ;
                    o3.setText(String.valueOf(fl1)) ;
                 }
                 if(v3 > 100.0) {
                    v3 = 100.0 ;
                    fl1 = (float) v3 ;
                    o3.setText(String.valueOf(fl1)) ;
                 }
                 rlhum = v3 /100. ;
                 computeFlow() ;
              }
           }
        }
     }

     class Mdr extends Panel {
        Ball outerparent ;
        Button ptchbt,stupbt ;
        Checkbox ch1,ch2 ;
        CheckboxGroup cbg ;
        Label lu4,la1,la2,la3,la4,la5,la6 ;
        Label l1,l2 ;
        TextField f1,f2;
        Choice ballch ;

        Mdr (Ball target) { 
          int i1 ;

          outerparent = target ;
          setLayout(new GridLayout(5,3,5,5)) ;
 
          la1 = new Label("CurveBall Student ", Label.CENTER) ;
          la1.setForeground(Color.red) ;
          la3 = new Label("Student ", Label.LEFT) ;
          la3.setForeground(Color.red) ;
          la2 = new Label("Version 1.5c", Label.CENTER) ;
          la2.setForeground(Color.red) ;
          la6 = new Label("1.5c", Label.LEFT) ;
          la6.setForeground(Color.red) ;

          la4 = new Label("Pitch: ", Label.RIGHT) ;
          la4.setForeground(Color.red) ;
          ptchbt = new Button("Throw") ;
          ptchbt.setBackground(Color.red) ;
          ptchbt.setForeground(Color.white) ;

          stupbt = new Button("Set Up") ;
          stupbt.setBackground(Color.blue) ;
          stupbt.setForeground(Color.white) ;

          la5 = new Label("Pitcher ", Label.RIGHT) ;
          la5.setForeground(Color.red) ;

          ballch = new Choice() ;
          ballch.addItem("Baseball") ;
          ballch.addItem("Fast Softball") ;
          ballch.addItem("Slow Softball") ;
          ballch.addItem("Input Ball");
          ballch.select(0) ;

          l1 = new Label("Diameter (ins)", Label.CENTER) ;
          f1 = new TextField("2.87",5) ;
          f1.setBackground(Color.black) ;
          f1.setForeground(Color.yellow) ;

          l2 = new Label("Weight (oz)", Label.CENTER) ;
          f2 = new TextField("5.0",5) ;
          f2.setBackground(Color.black) ;
          f2.setForeground(Color.yellow) ;

          cbg = new CheckboxGroup() ;

          ch1 = new Checkbox("Righty",cbg,true) ;
          ch2 = new Checkbox("Lefty",cbg,false) ;

          add(la1) ;
          add(new Label(" Ball:", Label.RIGHT) ) ;
          add(ballch) ;

          add(la2) ;
          add(l1) ;
          add(f1) ;

          add(new Label(" ", Label.RIGHT) ) ;
          add(l2) ;
          add(f2) ;

          add(la5) ;
          add(ch2) ;
          add(ch1) ;

          add(la4) ;
          add(stupbt) ;
          add(ptchbt) ;
        }
 
        public boolean action(Event evt, Object arg) {
            if(evt.target instanceof Button) {
               String label = (String)arg ;
               if(label.equals("Throw")){
                     nptb = 1 ;
                     pitch = 1 ;
               }
               if(label.equals("Set Up")){
                     nptb = 1 ;
                     pitch = 0 ;
               }
               computeFlow() ;
               return true ;
            }
            if(evt.target instanceof Checkbox) {
               this.handleZoom(evt) ;
               return true ;
            }
            if(evt.target instanceof Choice) {
               this.handleCho(arg) ;
               return true ;
            }
            if(evt.target instanceof TextField) {
               this.handleText(evt) ;
               return true ;
            }

            else return false ;
        }
 
       public void handleText(Event evt) {
          Double V1,V2,V3,V4 ;
          double v1,v2,v3,v4 ;
          float fl1 ;

          if (balflag == 1) {
            V1 = Double.valueOf(f1.getText()) ;
            v1 = V1.doubleValue() ;
            V2 = Double.valueOf(f2.getText()) ;
            v2 = V2.doubleValue() ;
 
            diameter = v1 ;
            if(v1 < .1) {
               diameter = v1 = .1 ;
               fl1 = (float) v1 ;
               f1.setText(String.valueOf(fl1)) ;
            }
            if(v1 > 12.0) {
               diameter = v1 = 12.0 ;
               fl1 = (float) v1 ;
               f1.setText(String.valueOf(fl1)) ;
            }

            weight = v2 ;
            if(v2 < 1.0) {
               weight = v2 = 1.0 ;
               fl1 = (float) v2 ;
               f2.setText(String.valueOf(fl1)) ;
            }
            if(v2 > 80.0) {
               weight = v2 = 80.0 ;
               fl1 = (float) v2 ;
               f2.setText(String.valueOf(fl1)) ;
            }

            rball = diameter / 24.0 ;
            mball = weight / 16.0 ;
 
            computeFlow() ;
          }
          if (balflag == 0) {
              f1.setText(String.valueOf(diameter)) ;
              f2.setText(String.valueOf(weight)) ;
          }
       }
 
        public void handleCho(Object obj) {
            int iball ;

            iball  = ballch.getSelectedIndex() ;

            switch (iball) {
                case 0:{                            // baseball
                    balflag = 0;
                    diameter = 2.87 ;
                    weight = 5.0 ;
                    lpit = 60.5 ;
                    f1.setText(String.valueOf(filter3(diameter))) ;
                    f1.setBackground(Color.black) ;
                    f1.setForeground(Color.yellow) ;
                    f2.setText(String.valueOf(filter3(weight))) ;
                    f2.setBackground(Color.black) ;
                    f2.setForeground(Color.yellow) ;
                    break ;
                }
                case 1:{                            // fast softball
                    balflag = 0;
                    diameter = 3.8 ;
                    weight = 6.25 ;
                    lpit = 43.0 ;         // fast pitch
                    f1.setText(String.valueOf(filter3(diameter))) ;
                    f1.setBackground(Color.black) ;
                    f1.setForeground(Color.yellow) ;
                    f2.setText(String.valueOf(filter3(weight))) ;
                    f2.setBackground(Color.black) ;
                    f2.setForeground(Color.yellow) ;
                    break ;
                }
                case 2:{                            // slow softball
                    balflag = 0;
                    diameter = 3.8 ;
                    weight = 6.25 ;
                    lpit = 50.0 ;        //slow pitch softball
                    f1.setText(String.valueOf(filter3(diameter))) ;
                    f1.setBackground(Color.black) ;
                    f1.setForeground(Color.yellow) ;
                    f2.setText(String.valueOf(filter3(weight))) ;
                    f2.setBackground(Color.black) ;
                    f2.setForeground(Color.yellow) ;
                    break ;
                }
                case 3:{                            // input ball parameters
                    balflag = 1;
                    f1.setText(String.valueOf(filter3(diameter))) ;
                    f1.setBackground(Color.white) ;
                    f1.setForeground(Color.black) ;
                    f2.setText(String.valueOf(filter3(weight))) ;
                    f2.setBackground(Color.white) ;
                    f2.setForeground(Color.black) ;
                    break ;
               }
             }
             rball = diameter / 24.0 ;
             mball = weight / 16.0 ;
             computeFlow() ;
        }

        public void handleZoom(Object obj) {
            int i1,i2 ;
            Double V3 ;
            double v3 ;

            if (ch1.getState()) {    // righty
                 spindr = -1.0 ;
                 relsdr = 1.0 ;
            }
            if (ch2.getState()) {    // lefty
                 spindr = 1.0 ;
                 relsdr = -1.0 ;
            }

            V3 = Double.valueOf(l.lol1.f3.getText()) ;
            v3 = V3.doubleValue() ;
            relsy = v3 * relsdr ;

            computeFlow() ;
        }
     }
  }

  class L extends Panel {
     Ball outerparent ;
     Lor1 lor1 ;
     Lor2 lor2 ;
     Lol1 lol1 ;
     Lol2 lol2 ;

     L (Ball target) { 
          outerparent = target ;
          setLayout(new GridLayout(1,4,5,5)) ;
 
          lol1 = new Lol1(outerparent) ;
          lol2 = new Lol2(outerparent) ;
          lor1 = new Lor1(outerparent) ;
          lor2 = new Lor2(outerparent) ;

          add(lor1) ;  
          add(lor2) ;  
          add(lol1) ;  
          add(lol2) ;  
     }

     class Lol1 extends Panel {
        Ball outerparent ;
        TextField f1,f2,f3,f4 ;
        Label l1,l2,l3,l4,lab1 ;

        Lol1 (Ball target) {

         outerparent = target ;
         setLayout(new GridLayout(5,2,5,5)) ;

         lab1 = new Label("Set Conditions:", Label.CENTER) ;
         lab1.setForeground(Color.red) ;

         l1 = new Label("Speed - mph", Label.CENTER) ;
         f1 = new TextField("100.0",5) ;

         l2 = new Label("Spin - rpm ", Label.CENTER) ;
         f2 = new TextField("0.0",5) ;

         l3 = new Label("Release - ft", Label.CENTER) ;
         f3 = new TextField("0.0",5) ;

         l4 = new Label("Angle - degs", Label.CENTER) ;
         f4 = new TextField("0.0",5) ;

         add(new Label(" ", Label.CENTER)) ;
         add(lab1) ;

         add(l1) ;
         add(f1) ;

         add(l2) ;
         add(f2) ;

         add(l3) ;
         add(f3) ;

         add(l4) ;
         add(f4) ;
       }
 
       public boolean handleEvent(Event evt) {
         if(evt.id == Event.ACTION_EVENT) {
            this.handleText(evt) ;
            return true ;
         }
         else return false ;
       }
 
       public void handleText(Event evt) {
          Double V1,V2,V3,V4 ;
          double v1,v2,v3,v4 ;
          float fl1 ;
          int i1,i2,i3,i4 ;

          V1 = Double.valueOf(f1.getText()) ;
          v1 = V1.doubleValue() ;
          V2 = Double.valueOf(f2.getText()) ;
          v2 = V2.doubleValue() ;
          V3 = Double.valueOf(f3.getText()) ;
          v3 = V3.doubleValue() ;
          V4 = Double.valueOf(f4.getText()) ;
          v4 = V4.doubleValue() ;
 
          vfsd = v1 ;
          if(v1 < vmn1) {
             vfsd = v1 = vmn1 ;
             fl1 = (float) v1 ;
             f1.setText(String.valueOf(fl1)) ;
          }
          if(v1 > vmx1) {
             vfsd = v1 = vmx1 ;
             fl1 = (float) v1 ;
             f1.setText(String.valueOf(fl1)) ;
          }

          spin = v2 ;
          if(v2 < vmn2) {
             spin = v2 = vmn2 ;
             fl1 = (float) v2 ;
             f2.setText(String.valueOf(fl1)) ;
          }
          if(v2 > vmx2) {
             spin = v2 = vmx2 ;
             fl1 = (float) v2 ;
             f2.setText(String.valueOf(fl1)) ;
          }
          spin = spin/60.0 ;

          relsy = v3 ;
          if(v3 < vmn3) {
             relsy = v3 = vmn3 ;
             fl1 = (float) v3 ;
             f3.setText(String.valueOf(fl1)) ;
          }
          if(v3 > vmx3) {
             relsy = v3 = vmx3 ;
             fl1 = (float) v3 ;
             f3.setText(String.valueOf(fl1)) ;
          }
          relsy = relsy / lconv * relsdr ;

          angr = v4 ;
          if(v4 < vmn4) {
             angr = v4 = vmn4 ;
             fl1 = (float) v4 ;
             f4.setText(String.valueOf(fl1)) ;
          }
          if(v4 > vmx4) {
             angr = v4 = vmx4 ;
             fl1 = (float) v4 ;
             f4.setText(String.valueOf(fl1)) ;
          }

          i1 = (int) (((v1 - vmn1)/(vmx1-vmn1))*1000.) ;
          i2 = (int) (((v2 - vmn2)/(vmx2-vmn2))*1000.) ;
          i3 = (int) (((v3 - vmn3)/(vmx3-vmn3))*1000.) ;
          i4 = (int) (((v4 - vmn4)/(vmx4-vmn4))*1000.) ;

          lol2.s1.setValue(i1) ;
          lol2.s2.setValue(i2) ;
          lol2.s3.setValue(i3) ;
          lol2.s4.setValue(i4) ;
 
          computeFlow() ;
       }
     }

     class Lol2 extends Panel {
        Ball outerparent ;
        Scrollbar s1,s2,s3,s4;

        Lol2 (Ball target) {
         int i1,i2,i3,i4 ;

         outerparent = target ;
         setLayout(new GridLayout(5,1,5,5)) ;

         i1 = (int) (((100.0 - vmn1)/(vmx1-vmn1))*1000.) ;
         i2 = (int) (((0.0 - vmn2)/(vmx2-vmn2))*1000.) ;
         i3 = (int) (((0.0 - vmn3)/(vmx3-vmn3))*1000.) ;
         i4 = (int) (((0.0 - vmn4)/(vmx4-vmn4))*1000.) ;

         s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
         s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);
         s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
         s4 = new Scrollbar(Scrollbar.HORIZONTAL,i4,10,0,1000);

         add(new Label(" ", Label.CENTER)) ;
         add(s1) ;
         add(s2) ;
         add(s3) ;
         add(s4) ;
       }
 
       public boolean handleEvent(Event evt) {
            if(evt.id == Event.ACTION_EVENT) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_ABSOLUTE) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_DOWN) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_UP) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_DOWN) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_UP) {
               this.handleBar(evt) ;
               return true ;
            }
            else return false ;
       }

       public void handleBar(Event evt) {
         int i1,i2,i3,i4 ;
         double v1,v2,v3,v4 ;
         float fl1,fl2,fl3,fl4 ;

// Input for computations
         i1 = s1.getValue() ;
         i2 = s2.getValue() ;
         i3 = s3.getValue() ;
         i4 = s4.getValue() ;

         v1 = i1 * (vmx1 - vmn1)/ 1000. + vmn1 ;
         v2 = i2 * (vmx2 - vmn2)/ 1000. + vmn2 ;
         v3 = i3 * (vmx3 - vmn3)/ 1000. + vmn3 ;
         v4 = i4 * (vmx4 - vmn4)/ 1000. + vmn4 ;
 
         vfsd = v1 ;
         spin  = v2 ;
         spin = spin/60.0 ;
         relsy = v3 / lconv * relsdr ;
         angr = v4 ;

         fl1 = (float) v1 ;
         fl2 = (float) v2 ;
         fl3 = (float) v3 ;
         fl4 = (float) v4 ;
 
         lol1.f1.setText(String.valueOf(fl1)) ;
         lol1.f2.setText(String.valueOf(fl2)) ;
         lol1.f3.setText(String.valueOf(fl3)) ;
         lol1.f4.setText(String.valueOf(fl4)) ;

         computeFlow() ;
       }

     }

     class Lor1 extends Panel {
        Ball outerparent ;
        Choice prbch,untch ;
        Button resbt ;
        TextField out ;
        Label lprb ;

        Lor1 (Ball target) {
         outerparent = target ;
         setLayout(new GridLayout(4,2,2,2)) ;

          lprb = new Label("Probe:", Label.RIGHT) ;
          prbch = new Choice() ;
          prbch.addItem("OFF") ;
          prbch.addItem("Velocity");
          prbch.addItem("Pressure");
          prbch.addItem("Smoke");
          prbch.select(0) ;
 
          resbt = new Button("Reset") ;
          resbt.setBackground(Color.orange) ;
          resbt.setForeground(Color.black) ;
 
          untch = new Choice() ;
          untch.addItem("English") ;
          untch.addItem("Metric");
          untch.select(0) ;

          out = new TextField("12.5",5) ;
          out.setBackground(Color.black) ;
          out.setForeground(Color.yellow) ;
      
          add(lprb) ;
          add(prbch) ;

          add(new Label(" ", Label.RIGHT)) ;
          add(new Label(" ", Label.RIGHT)) ;

          add(new Label("Force ", Label.RIGHT)) ;
          add(out) ;

          add(resbt) ;
          add(new Label(" ", Label.RIGHT)) ;
       }

       public boolean action(Event evt, Object arg) {
          if(evt.target instanceof Button) {
             this.handleRset(evt) ;
             return true ;
          }
          if(evt.target instanceof Choice) {
              this.handleProb(arg) ;
              return true ;
          }
          else return false ;
       }

       public void handleRset(Event evt) {
         solve.setDefaults() ;
         m.mdr.ch1.setState(true) ;
         m.mdr.ch2.setState(false) ;
         m.mdr.ballch.select(0) ;
         m.mdr.f1.setBackground(Color.black) ;
         m.mdr.f1.setForeground(Color.yellow) ;
         m.mdr.f2.setBackground(Color.black) ;
         m.mdr.f2.setForeground(Color.yellow) ;
         m.mdr.f1.setText(String.valueOf(diameter)) ;
         m.mdr.f2.setText(String.valueOf(weight)) ;

         m.mdl.mdlu.placeh.select(0) ;
         m.mdl.mdlu.weatch.select(0) ;
         l.lor1.untch.select(lunits) ;
         l.lor1.prbch.select(0) ;
         m.mdl.mdll.o1.setBackground(Color.black) ;
         m.mdl.mdll.o1.setForeground(Color.yellow) ;
         m.mdl.mdll.o2.setBackground(Color.black) ;
         m.mdl.mdll.o2.setForeground(Color.yellow) ;
         m.mdl.mdll.o3.setBackground(Color.black) ;
         m.mdl.mdll.o3.setForeground(Color.yellow) ;
         m.mdl.mdll.o2.setText("59") ;
         loadProb() ;
       }

       public void handleProb(Object obj) {
          int i2 ;

          lunits  = untch.getSelectedIndex() ;
          pboflag  = prbch.getSelectedIndex() ;
             // **** the lunits check MUST come first
          setUnits () ;
          loadProb () ;
          loadOut() ;
          if(pboflag > 0) loadProbe() ;

          computeFlow() ;
       }
     }

     class Lor2 extends Panel {
        Ball outerparent ;
        Scrollbar s1,s2;
        Lor2a lor2a;

        Lor2 (Ball target) {

         outerparent = target ;
         setLayout(new BorderLayout(5,5)) ;

         s1 = new Scrollbar(Scrollbar.VERTICAL,550,10,0,1000);
         s2 = new Scrollbar(Scrollbar.HORIZONTAL,550,10,0,1000);

         lor2a = new Lor2a(outerparent) ;

         add("West",s1) ;
         add("South",s2) ;
         add("Center",lor2a) ;
       }

       public boolean handleEvent(Event evt) {
            if(evt.id == Event.ACTION_EVENT) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_ABSOLUTE) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_DOWN) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_LINE_UP) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_DOWN) {
               this.handleBar(evt) ;
               return true ;
            }
            if(evt.id == Event.SCROLL_PAGE_UP) {
               this.handleBar(evt) ;
               return true ;
            }
            else return false ;
       }

       public void handleBar(Event evt) {
         int i1,i2 ;

         i1 = s1.getValue() ;
         i2 = s2.getValue() ;

         ypval = 5.0 - i1 * 10.0/ 1000. ;
         xpval = i2 * 20.0/ 1000. -10.0 ;

         computeFlow() ;
       }

       class Lor2a extends Canvas  {
          Ball outerparent ;

          Lor2a (Ball target) {
            setBackground(Color.black) ;
          }

          public void update(Graphics g) {
            l.lor2.lor2a.paint(g) ;
          }
 
          public void paint(Graphics g) {
            int ex,ey,index ;

            off3Gg.setColor(Color.black) ;
            off3Gg.fillRect(0,0,150,100) ;

            if (pboflag == 0) off3Gg.setColor(Color.gray) ;
            if (pboflag > 0) off3Gg.setColor(Color.yellow) ;
            off3Gg.fillArc(20,10,80,80,-23,227) ;
            off3Gg.setColor(Color.black) ;
     // tick marks
            for (index = 1; index <= 4; ++ index) {
                ex = 60 + (int) (50.0 * Math.cos(convdr * (-22.5 + 45.0 * index))) ;
                ey = 50 - (int) (50.0 * Math.sin(convdr * (-22.5 + 45.0 * index))) ;
                off3Gg.drawLine(60,50,ex,ey) ;
            }
            off3Gg.fillArc(25,15,70,70,-25,235) ;

            ex = 60 ;
            ey = 50 ;
            off3Gg.setColor(Color.green) ;
            if (pboflag == 1) {
               off3Gg.drawString(String.valueOf(filter0(pbval)),50,70) ;
               if (lunits == 0) {
                   off3Gg.drawString("mph",50,80) ;
                   off3Gg.setColor(Color.yellow) ;
                   off3Gg.drawString("0",10,80) ;
                   off3Gg.drawString("100",5,30) ;
                   off3Gg.drawString("200",30,10) ;
                   off3Gg.drawString("300",70,10) ;
                   off3Gg.drawString("400",100,30) ;
                   off3Gg.drawString("500",90,80) ;
                   ex = 60 - (int) (30.0 * Math.cos(convdr * 
                       (-22.5 + pbval * 225. / 500.))) ;
                   ey = 50 - (int) (30.0 * Math.sin(convdr * 
                       (-22.5 + pbval * 225. / 500.))) ;
               }
               if (lunits == 1) {
                   off3Gg.drawString("km/hr",50,80) ;
                   off3Gg.setColor(Color.yellow) ;
                   off3Gg.drawString("0",10,80) ;
                   off3Gg.drawString("200",5,30) ;
                   off3Gg.drawString("400",30,10) ;
                   off3Gg.drawString("600",70,10) ;
                   off3Gg.drawString("800",100,30) ;
                   off3Gg.drawString("1000",90,80) ;
                   ex = 60 - (int) (30.0 * Math.cos(convdr * 
                       (-22.5 + pbval * 225. / 1000.))) ;
                   ey = 50 - (int) (30.0 * Math.sin(convdr * 
                       (-22.5 + pbval * 225. / 1000.))) ;
               }
            }
            if (pboflag == 2) {
               off3Gg.drawString(String.valueOf(filter3(pbval)),40,70) ;
               if (lunits == 0) {
                   off3Gg.drawString("psi",50,80) ;
                   off3Gg.setColor(Color.yellow) ;
                   off3Gg.drawString("0",10,80) ;
                   off3Gg.drawString("3",5,30) ;
                   off3Gg.drawString("6",30,10) ;
                   off3Gg.drawString("9",70,10) ;
                   off3Gg.drawString("12",100,30) ;
                   off3Gg.drawString("15",90,80) ;
                   ex = 60 - (int) (30.0 * Math.cos(convdr * 
                       (-22.5 + pbval * 225. / 15.))) ;
                   ey = 50 - (int) (30.0 * Math.sin(convdr * 
                       (-22.5 + pbval * 225. / 15.))) ;
               }
               if (lunits == 1) {
                   off3Gg.drawString("k-Pa",50,80) ;
                   off3Gg.setColor(Color.yellow) ;
                   off3Gg.drawString("0",10,80) ;
                   off3Gg.drawString("20",5,30) ;
                   off3Gg.drawString("40",30,10) ;
                   off3Gg.drawString("60",70,10) ;
                   off3Gg.drawString("80",100,30) ;
                   off3Gg.drawString("100",90,80) ;
                   ex = 60 - (int) (30.0 * Math.cos(convdr * 
                       (-22.5 + pbval * 225. / 100.))) ;
                   ey = 50 - (int) (30.0 * Math.sin(convdr * 
                       (-22.5 + pbval * 225. / 100.))) ;
               }
            }
            if (pboflag == 3) {
               off3Gg.drawString(String.valueOf(filter0(pbval)),50,70) ;
               off3Gg.drawString("",50,80) ;
               ex = 60 ; 
               ey = 50 ; 
            }

            off3Gg.setColor(Color.yellow) ;
            off3Gg.drawLine(60,50,ex,ey) ;

            g.drawImage(offImg3,0,0,this) ;   
          }
       }
     }
  }

  class View1 extends Canvas  
         implements Runnable{
     Ball outerparent ;
     Thread runner ;

     View1 (Ball target) {
         setBackground(Color.black) ;
         runner = null ;
     }

     public Insets insets() {
        return new Insets(0,10,0,10) ;
     }
 
     public void start() {
        if (runner == null) {
           runner = new Thread(this) ;
           runner.start() ;
        }
        antim1 = 0 ;                              /* MODS  21 JUL 99 */
        ancol1 = 1 ;                              /* MODS  27 JUL 99 */
     }

     public void run() {
       int timer ;
 
       timer = 100 ;
       while (true) {
          ++ antim1 ;
          try { Thread.sleep(timer); }
          catch (InterruptedException e) {}
          view1.repaint() ;
          if (antim1 == 3) {
             antim1 = 0;
             ancol1 = - ancol1 ;               /* MODS 27 JUL 99 */
          }
          timer = 135 - (int) (.227 *vfsd/vconv) ;
                                            // make the ball spin
          plthg1[1] = plthg1[1] - spin*spindr*1.489 ;
          if (plthg1[1] < -360.0) {
                plthg1[1] = plthg1[1] + 360.0 ;
          }
          if (plthg1[1] > 360.0) {
                plthg1[1] = plthg1[1] - 360.0 ;
          }
          plthg1[0] = plthg1[1] + 180.0 ;
          if (plthg1[0] > 270.0) {
                plthg1[0] = plthg1[0] - 360.0 ;
          }
                                             // animate path of ball
          nptb = nptb + 1 ;
          if (nptb == npt) nptb = 1 ;
       }
     }

     public void loadPlot() {
       return ;
     }

     public void update(Graphics g) {
        view1.paint(g) ;
     }
 
    public void paint(Graphics g) {
      int i,j,k,n,tbgn,tend ;
      int xlabel,ylabel,ind,inmax,inmin ;
      int exes[] = new int[8] ;
      int whys[] = new int[8] ;
      double offx,scalex,offy,scaley,waste,incy,incx;
      double xtrans,ytrans,xl,yl,length;
      int camx[] = new int[19] ;
      int camy[] = new int[19] ;
      Color col1 ;

      col1 = new Color(0,121,0) ;   // dark green

      xtrans = 375. ;
      ytrans = 30.0 ;

      offx = 0.0 - begx ;
      scalex = 8.0/(endx-begx) ;
      incx = (endx-begx)/(ntikx-1) ;
      offy = 0.0 - begy ;
      scaley = 4.0/(endy-begy) ;
      incy = (endy-begy)/(ntiky-1) ;

         offsGg.setColor(col1) ;
         offsGg.fillRect(0,0,750,300) ;
                                         // pitching rubber
         exes[0] = (int) (fact * scalex*(offx - 0.5)) + xt ;
         whys[0] = (int) (fact * scaley*(-offy + 1.0)) + yt ;
         exes[1] = (int) (fact * scalex*(offx + 0.5)) + xt ;
         whys[1] = (int) (fact * scaley*(-offy + 1.0)) + yt ;
         exes[2] = (int) (fact * scalex*(offx + 0.5)) + xt ;
         whys[2] = (int) (fact * scaley*(-offy - 1.0)) + yt ;
         exes[3] = (int) (fact * scalex*(offx - 0.5)) + xt ;
         whys[3] = (int) (fact * scaley*(-offy - 1.0)) + yt ;
         offsGg.setColor(Color.yellow) ;
         offsGg.fillPolygon(exes,whys,4) ;
                                           // home plate
         exes[0] = (int) (fact* (scalex*(offx - (lpit + 2.5)))) + xt ;
         whys[0] = (int) (fact* (-scaley*(offy - .75))) + yt ;
         exes[1] = (int) (fact* (scalex*(offx - (lpit + .1)))) + xt ;
         whys[1] = (int) (fact* (-scaley*(offy - .75))) + yt ;
         exes[2] = (int) (fact* (scalex*(offx - (lpit+.1)))) + xt ;
         whys[2] = (int) (fact* (-scaley*(offy + .75))) + yt ;
         exes[3] = (int) (fact* (scalex*(offx - (lpit+2.5)))) + xt ;
         whys[3] = (int) (fact* (-scaley*(offy + .75))) + yt ;
         exes[4] = (int) (fact* (scalex*(offx - (lpit+3.5)))) + xt ;
         whys[4] = (int) (fact* (-scaley*(offy + 0.0))) + yt ;
         offsGg.setColor(Color.black) ;
         offsGg.fillPolygon(exes,whys,5) ;

         exes[0] = (int) (fact* (scalex*(offx - (lpit+2.2)))) + xt ;
         whys[0] = (int) (fact* (-scaley*(offy - .6))) + yt ;
         exes[1] = (int) (fact* (scalex*(offx - (lpit + .4)))) + xt ;
         whys[1] = (int) (fact* (-scaley*(offy - .6))) + yt ;
         exes[2] = (int) (fact* (scalex*(offx - (lpit+.4)))) + xt ;
         whys[2] = (int) (fact* (-scaley*(offy + .6))) + yt ;
         exes[3] = (int) (fact* (scalex*(offx - (lpit+2.2)))) + xt ;
         whys[3] = (int) (fact* (-scaley*(offy + .6))) + yt ;
         exes[4] = (int) (fact* (scalex*(offx - (lpit+3.0)))) + xt ;
         whys[4] = (int) (fact* (-scaley*(offy + 0.0))) + yt ;
         offsGg.setColor(Color.white) ;
         offsGg.fillPolygon(exes,whys,5) ;
                                           /* Foul lines */
         exes[0] = (int) (fact* (scalex*(offx - (lpit+2.5)))) + xt ;
         whys[0] = (int) (fact* (-scaley*(offy - .75))) + yt ;
         exes[1] = (int) (fact* (scalex*(offx - (lpit-57.35)))) + xt ;
         whys[1] = (int) (fact* (-scaley*(offy - 25.5))) + yt ;
         offsGg.setColor(Color.white) ;
         offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
         exes[0] = (int) (fact* (scalex*(offx - (lpit+2.5)))) + xt;
         whys[0] = (int) (fact* (-scaley*(offy + .75))) + yt ;
         exes[1] = (int) (fact* (scalex*(offx - (lpit-57.35)))) + xt ;
         whys[1] = (int) (fact* (-scaley*(offy + 25.5))) + yt ;
         offsGg.setColor(Color.white) ;
         offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

         offsGg.fillOval((int) (fact*scalex*(offx+pltx[1])) + xt,
             (int) (fact*(-scaley*(offy+plty[1]))) + yt - 2,5,5);
         exes[0] = (int) (fact* (scalex*(offx+pltx[1]))) + xt ;
         whys[0] = (int) (fact* (-scaley*(offy+plty[1]))) + yt ;
         length = 10.0 * (vfsd / vmaxa) ;
         exes[1] = (int) (fact* (scalex*(offx+pltx[1] - 
                     length * Math.cos(convdr*angr)))) + xt ;
         whys[1] = (int) (fact* (-scaley*(offy+plty[1] +
                     length * Math.sin(convdr*angr)))) + yt ;
         offsGg.setColor(Color.black) ;
         offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
         exes[0] = exes[1] + 
            (int) (10.0*Math.cos(convdr*angr) - 4.0*Math.sin(convdr*angr)) ;
         exes[2] = exes[1] +
            (int) (10.0*Math.cos(convdr*angr) + 4.0*Math.sin(convdr*angr)) ;
         whys[0] = whys[1] +
            (int) (10.0*Math.sin(convdr*angr) + 4.0*Math.cos(convdr*angr)) ;
         whys[2] = whys[1] +
            (int) (10.0*Math.sin(convdr*angr) - 4.0*Math.cos(convdr*angr)) ;
         offsGg.fillPolygon(exes,whys,3) ;
         offsGg.setColor(Color.white) ;
         offsGg.fillOval((int) (fact*scalex*(offx+pltx[1] )) + xt ,
             (int) (fact*(-scaley*(offy+plty[1]))) + yt - 5,10,10);
         exes[0] = (int) (fact* (scalex*(offx+pltx[1] + 
                     0.5 * Math.cos(convdr*(plthg1[1] + 180.))))) + xt + 5 ;
         whys[0] = (int) (fact* (-scaley*(offy+plty[1] +
                     0.5 * Math.sin(convdr*(plthg1[1] + 180.))))) + yt ;
         exes[1] = (int) (fact* (scalex*(offx+pltx[1] + 
                     0.5 * Math.cos(convdr*plthg1[1])))) + xt + 5 ;
         whys[1] = (int) (fact* (-scaley*(offy+plty[1] +
                     0.5 * Math.sin(convdr*plthg1[1])))) + yt ;
         offsGg.setColor(Color.red) ;
         offsGg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                                         // animated pitch
         if (pitch == 1) {
            offsGg.setColor(Color.white) ;
            for (i=2; i<=nptb; ++i) {                /* plot trajectory */
               offsGg.fillOval((int) (fact*scalex*(offx+pltx[i])) + xt,
                  (int) (fact*(-scaley*(offy+plty[i]))) + yt - 2,5,5);
            }
         }

        offsGg.setColor(Color.white) ;
        offsGg.drawString("Top View",600,10) ;

   // outcome

       if (pitch == 1) {
         if (yoff < 9.0) {
            offsGg.setColor(Color.red) ;
            offsGg.fillRect(10,10,90,30) ;
            offsGg.setColor(Color.white) ;
            offsGg.drawString("STRIKE !",30,30) ;
         }
         else {
            offsGg.setColor(Color.blue) ;
            offsGg.fillRect(10,10,90,30) ;
            offsGg.setColor(Color.white) ;
            offsGg.drawString("Ball",40,30) ;
         }
         offsGg.setColor(Color.white) ;
         offsGg.drawString(String.valueOf(filter0(yoff)),10,80) ;
         offsGg.drawString("inches from Center",35,80) ;
       }

       g.drawImage(offscreenImg,0,0,this) ;   
     }
   }

  class View2 extends Canvas  
         implements Runnable{
     Ball outerparent ;
     Thread runner ;

     View2 (Ball target) {
         setBackground(Color.black) ;
         runner = null ;
     }

     public Insets insets() {
        return new Insets(0,10,0,10) ;
     }
 
     public void start() {
        if (runner == null) {
           runner = new Thread(this) ;
           runner.start() ;
        }
        antim2 = 0 ;                              /* MODS  21 JUL 99 */
        ancol2 = 1 ;                              /* MODS  27 JUL 99 */
     }

     public void run() {
       int timer ;
 
       timer = 100 ;
       while (true) {
          ++ antim2 ;
          try { Thread.sleep(timer); }
          catch (InterruptedException e) {}
          view2.repaint() ;
          if (antim2 == 3) {
             antim2 = 0;
             ancol2 = - ancol2 ;               /* MODS 27 JUL 99 */
          }
          timer = 135 - (int) (.227 *vfsd/vconv) ;
                                            // make the ball spin
          plthg2[1] = plthg2[1] - spin*spindr*1.489 ;
          if (plthg2[1] < -360.0) {
                plthg2[1] = plthg2[1] + 360.0 ;
          }
          if (plthg2[1] > 360.0) {
                plthg2[1] = plthg2[1] - 360.0 ;
          }
          pltxg[1] = pltrg[1] * Math.cos(convdr*plthg2[1]) ;
          pltyg[1] = pltrg[1] * Math.sin(convdr*plthg2[1]) ;
          plthg2[0] = plthg2[1] + 180.0 ;
          if (plthg2[0] > 270.0) {
                plthg2[0] = plthg2[0] - 360.0 ;
          }
          pltxg[0] = pltrg[1] * Math.cos(convdr*plthg2[0]) ;
          pltyg[0] = pltrg[1] * Math.sin(convdr*plthg2[0]) ;
       }
     }

     public void loadPlot() {
       return ;
     }

     public void update(Graphics g) {
        view2.paint(g) ;
     }
 
    public void paint(Graphics g) {
      int i,j,k,n,tbgn,tend ;
      int xlabel,ylabel,ind,inmax,inmin ;
      int exes[] = new int[8] ;
      int whys[] = new int[8] ;
      double offx,scalex,offy,scaley,waste,incy,incx;
      double xtrans,ytrans,xl,yl,length;
      int camx[] = new int[19] ;
      int camy[] = new int[19] ;
      Color col ;

      xtrans = 375. ;
      ytrans = 30.0 ;

      offx = 0.0 - begx ;
      scalex = 8.0/(endx-begx) ;
      incx = (endx-begx)/(ntikx-1) ;
      offy = 0.0 - begy ;
      scaley = 4.0/(endy-begy) ;
      incy = (endy-begy)/(ntiky-1) ;

      col = new Color(0,0,0) ;
      if(planet == 0) col = Color.white ;
      if(planet == 1) col = Color.yellow ;

      if (vfsd > .01) {  /* plot flowfield */
       off4Gg.setColor(Color.black) ;
       off4Gg.fillRect(0,0,750,300) ;
       if (pitch == 1) {
          for (j=1; j<=nln2-1; ++j) {           /* lower half */
             exes[1] = (int) (fact2*xg[j][1]) + xt2 ;
             whys[1] = (int) (fact2*(-yg[j][1])) + yt2 ;
             for (i=2 ; i<= nptc; ++i) {
                exes[0] = exes[1] ;
                whys[0] = whys[1] ;
                exes[1] = (int) (fact2*xg[j][i]) + xt2 ;
                whys[1] = (int) (fact2*(-yg[j][i])) + yt2 ;
                if ((i-antim2)/3*3 == (i-antim2) ) {
                  if (ancol2 == -1) {          /* MODS  27 JUL 99 */
                    if((i-antim2)/6*6 == (i-antim2))off4Gg.setColor(col);
                    if((i-antim2)/6*6 != (i-antim2))off4Gg.setColor(Color.red);
                  }
                  if (ancol2 == 1) {          /* MODS  27 JUL 99 */
                    if((i-antim2)/6*6 == (i-antim2))off4Gg.setColor(Color.red);
                    if((i-antim2)/6*6 != (i-antim2))off4Gg.setColor(col);
                  }
                  off4Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                }
             }
          }
          for (j=nln2+1; j<=nlnc; ++j) {          /* upper half */
             exes[1] = (int) (fact2*xg[j][1]) + xt2 ;
             whys[1] = (int) (fact2*(-yg[j][1])) + yt2 ;
                for (i=2 ; i<= nptc; ++i) {
                   exes[0] = exes[1] ;
                whys[0] = whys[1] ;
                exes[1] = (int) (fact2*xg[j][i]) + xt2 ;
                whys[1] = (int) (fact2*(-yg[j][i])) + yt2 ;
                if ( ((i-antim2)/3*3 == (i-antim2)) ) {
                  if (ancol2 == -1) {          /* MODS  27 JUL 99 */
                    if((i-antim2)/6*6 == (i-antim2))off4Gg.setColor(col);
                    if((i-antim2)/6*6 != (i-antim2))off4Gg.setColor(Color.red);
                  }
                  if (ancol2 == 1) {          /* MODS  27 JUL 99 */
                    if((i-antim2)/6*6 == (i-antim2))off4Gg.setColor(Color.red);
                    if((i-antim2)/6*6 != (i-antim2))off4Gg.setColor(col);
                  }
                  off4Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                }
             }
          }
          off4Gg.setColor(Color.white) ; /* stagnation */
          exes[1] = (int) (fact2*xg[nln2][1]) + xt2 ;
          whys[1] = (int) (fact2*(-yg[nln2][1])) + yt2 ;
          for (i=2 ; i<= npt2-1; ++i) {
                exes[0] = exes[1] ;
                whys[0] = whys[1] ;
                exes[1] = (int) (fact2*xg[nln2][i]) + xt2 ;
                whys[1] = (int) (fact2*(-yg[nln2][i])) + yt2 ;
                off4Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          }
          exes[1] = (int) (fact2*xg[nln2][npt2+1]) + xt2 ;
          whys[1] = (int) (fact2*(-yg[nln2][npt2+1])) + yt2 ;
          for (i=npt2+2 ; i<= nptc; ++i) {
              exes[0] = exes[1] ;
              whys[0] = whys[1] ;
              exes[1] = (int) (fact2*xg[nln2][i]) + xt2 ;
              whys[1] = (int) (fact2*(-yg[nln2][i])) + yt2 ;
              off4Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
           }
                                               /*  probe location */
           if (pboflag > 0) {
             off4Gg.setColor(Color.red) ; 
             off4Gg.fillOval((int) (fact2*pxg) + xt2,
                  (int) (fact2*(-pyg)) + yt2 - 2,5,5);
             off4Gg.setColor(Color.white) ; 
             exes[0] = (int) (fact2*(pxg + .1)) +xt2 ;
             whys[0] = (int) (fact2*(-pyg)) + yt2 ;
             exes[1] = (int) (fact2*(pxg + .5)) +xt2 ;
             whys[1] = (int) (fact2*(-pyg)) + yt2 ;
             exes[2] = (int) (fact2*(pxg + .5)) +xt2 ;
             if (pyg > 0.0 ) {
                   whys[2] = (int) (fact2*(-pyg -50.)) +yt2 ;
             }
             else {
                   whys[2] = (int) (fact2*(-pyg +50.)) +yt2 ;
             }
             off4Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             off4Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
             if (pboflag == 3) {    /* smoke trail  MODS  21 JUL 99 */
               off4Gg.setColor(Color.white) ; 
               exes[1] = (int) (fact2*xg[19][1]) + xt2 ;
               whys[1] = (int) (fact2*(-yg[19][1])) + yt2 ;
               for (i=2 ; i<= nptc; ++i) {
                  exes[0] = exes[1] ;
                  whys[0] = whys[1] ;
                  exes[1] = (int) (fact2*xg[19][i]) + xt2 ;
                  whys[1] = (int) (fact2*(-yg[19][i])) + yt2 ;
                  off4Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
               }
             }
           }
         }
    // draw the ball
        off4Gg.setColor(Color.white) ;
        exes[1] = (int) (fact2*(xg[0][npt2])) + xt2 ;
        whys[1] = (int) (fact2*(-yg[0][npt2])) + yt2 ;
        exes[2] = (int) (fact2*(xg[0][npt2])) + xt2 ;
        whys[2] = (int) (fact2*(-yg[0][npt2])) + yt2 ;
        for (i=1 ; i<= npt2-1; ++i) {
           exes[0] = exes[1] ;
           whys[0] = whys[1] ;
           exes[1] = (int) (fact2*(xg[0][npt2-i])) + xt2 ;
           whys[1] = (int) (fact2*(-yg[0][npt2-i])) + yt2 ;
           exes[3] = exes[2] ;
           whys[3] = whys[2] ;
           exes[2] = (int) (fact2*(xg[0][npt2+i])) + xt2 ;
           whys[2] = (int) (fact2*(-yg[0][npt2+i])) + yt2 ;
           camx[i] = (exes[1] + exes[2]) / 2 ;
           camy[i] = (whys[1] + whys[2]) / 2 ;
           off4Gg.fillPolygon(exes,whys,4) ;
        }
        off4Gg.setColor(Color.cyan) ;
        off4Gg.drawString("Ball Held Fixed",600,50) ;
        off4Gg.drawString(" in Wind Tunnel",600,62) ;
        off4Gg.drawString("Flow of Air",10,50) ;
        off4Gg.drawLine(20,67,50,67) ;
        off4Gg.drawString("Wind Tunnel View",580,10) ;
        exes[0] = 50 ;  exes[1] = 50; exes[2] = 60 ;
        whys[0] = 72 ;  whys[1] = 62; whys[2] = 67 ;
        off4Gg.fillPolygon(exes,whys,3) ;
    // rotation
        off4Gg.setColor(Color.red) ; 
        tbgn = (int) plthg2[1] + 90 + 30 ;
        off4Gg.drawArc((int)(fact2*(pltxg[1]-.5))+xt2,(int)(fact2*(-pltyg[1]-.5))+yt2,
           (int)(fact2*1.0),(int)(fact2*1.0),tbgn,130) ;
        tbgn = (int) plthg2[0] + 90 + 30  ;
        off4Gg.drawArc((int)(fact2*(pltxg[0]-.5))+xt2,(int)(fact2*(-pltyg[0]-.5))+yt2,
           (int)(fact2*1.0),(int)(fact2*1.0),tbgn,130) ;
 
       g.drawImage(offImg4,0,0,this) ;   
     }
   }
 }
}
