/*
                           KiteModeler
   
                           A Java Applet
               to design and evaluate a variety of kites

                           Version 1.3a 

                         Written by Tom Benson
                       NASA Glenn Research Center

>                              NOTICE
>This software is in the Public Domain.  It may be freely copied and used in
>non-commercial products, assuming proper credit to the author is given.  IT
>MAY NOT BE RESOLD.  If you want to use the software for commercial
>products, contact the author.
>No copyright is claimed in the United States under Title 17, U. S. Code.
>This software is provided "as is" without any warranty of any kind, either
>express, implied, or statutory, including, but not limited to, any warranty
>that the software will conform to specifications, any implied warranties of
>merchantability, fitness for a particular purpose, and freedom from
>infringement, and any warranty that the documentation will conform to the
>program, or any warranty that the software will be error free.
>In no event shall NASA be liable for any damages, including, but not
>limited to direct, indirect, special or consequential damages, arising out
>of, resulting from, or in any way connected with this software, whether or
>not based on warranty, contract, tort or otherwise, whether or not injury
>was sustained by persons or property or otherwise, and whether or not loss
>was sustained from, or arose out of the results of, or use of, the software
>or services provided hereunder.
 
  New test -
              * create panels - look like RocketModeler - begun 15Jan02
              * add default geometries - reset button
              * add string tension output
                add large output box - save geometry (like FoilSim)

                                           TJB  29 Jul 02

*/

import java.awt.*;
import java.lang.Math ;

public class Kite extends java.applet.Applet {
 
   static double convdr = 3.1415926/180. ;
   static double pid2 = 3.1415926/2.0 ;
   static double pi = 3.1415926 ;

   static double h1,h2,w1,w2,lkite ;
   static double h1d,h2d,w1d,w2d ;
   static double hmn,hmx,wmn,wmx ;
   static double hmnd,hmxd,wmnd,wmxd ;
   static double area,lengstk,cg,cp,ar;
   static double weight,lift,drag;
   static double wtarea,wtlngs,wtlngl,wttail;
   static double lbrid,lknot,ltail,xbr,ybr,kbase ;
   static double lbridd,lknotd,ltaild ;
   static double lbrmn,lbrmx,ltlmn,ltlmx ;
   static double lbrmnd,lbrmxd,ltlmnd,ltlmxd ;
   static double anga,lpiv,wind,lline ;
   static double windd,llined ;
   static double wndmn,wndmx,llnmn,llnmx ;
   static double wndmnd,wndmxd,llnmnd,llnmxd ;
   static double density,cl,cd ;
   static double angl,angk,alpha,torq,tension;
   static double almn,almx ;
   static double lconv,wconv,hconv ;
   static double[] tor  = new double[405] ; 
   static double con1,con2,xkite,ykite;
   static double[] xline = new double[21] ; 
   static double[] yline = new double[21] ; 

   static int surtyp,stktyp,lintyp,taltyp;
   static int iter,unstab ;

   static int ktype,angmod,units,fold,ground;
   static double beta;

       /*  plot & probe data */
   static double fact,vscale,fscale;
   static int xt,yt,sldloc,vldloc,fldloc,viewflg,pick;
 
   Solver solve ;
   S s ;
   P p ;
   CardLayout layin,layout ;
   Image offImg1 ;
   Graphics off1Gg ;
   Image offImg2 ;
   Graphics off2Gg ;

   public void init() {
     int i;
     Kite a = new Kite() ;
     solve = new Solver() ;

     offImg1 = createImage(this.size().width,
                      this.size().height) ;
     off1Gg = offImg1.getGraphics() ;
     offImg2 = createImage(this.size().width,
                      this.size().height) ;
     off2Gg = offImg2.getGraphics() ;

     setLayout(new GridLayout(2,1,0,5)) ;

     solve.setDefaults () ;
 
     s = new S(this) ;
     p = new P() ;

     add(s) ;
     add(p) ;

     loadInput() ;

     compute () ;
     s.view.start() ;
  }
 
  public Insets insets() {
     return new Insets(10,10,10,10) ;
  }

  public void compute() { 
     double wt1,wt2,wt3,wtail;
     double cosa,sina,tana,thet;
     double el,bet2;
     double alold,alnew,tnew,told;
     double slope,tzero,cosl,sinl ;
     double a1,a2,a3,c1 ;
     double delx ;
     int isave,iflag ;

 // compute geometry for various kites

     wtail = ltail * wttail ;

    if (ktype == 0) { // diamond
       lkite = h1 + h2 ;
       area = .5 * lkite * w1 ;
       wt1 = area * wtarea;
       lengstk = lkite + w1 ;
       wt2 = lkite *  wtlngs ;
       wt3 = w1 *  wtlngs ;
       weight = wt1 + wt2 + wt3 + wtail ;
       cg = (wt1 * (h1 + 2.0* h2) / 3.0 +
             wt2 * lkite / 2.0 +
             wt3 * h2  +
             wtail * (-ltail/2.0)
            )/ weight ;
       cp = .5 * lkite + h2/3.0 ;
       ar = w1 * w1 / area ;
       kbase = 0.0 ;
    }
    if (ktype == 1) { //delta
       lkite = h1 + h2 ;
       area = .5 * lkite * w1 ;
       wt1 = area * wtarea;
       lengstk = Math.sqrt(4.0*h1*h1 + w1*w1) ;
       wt2 = lengstk *  wtlngs ;
       weight = wt1 + wt2 + wtail ;
       cg = (wt1 * (h1 + 2.0* h2) / 3.0 +
             wt2 * (h2 + h1 /2.0) +
             wtail * (-ltail/2.0)
             ) / weight ; 
       cp = .5 * lkite + h2/3.0 ;
       ar = w1 * w1 / area ;
       kbase = 0.0 ;
     }
     if (ktype == 2) { //sled
       lkite = h1 + h2 ;
       area = .5 * lkite * (w1 + w2) ;
       wt1 = .5 * lkite * (w2 - w1) * wtarea ;
       wt2 = lkite * w1 * wtarea ;
       lengstk = 2.0 * lkite ;
       wt3 = lengstk *  wtlngs ;
       weight = wt1 + wt2 + wt3 + wtail ;
       cg = (wt1 * (h1 + 2.0* h2) / 3.0 +
             wt2 * lkite / 2.0 +
             wt3 * lkite / 2.0 +
             wtail * (-ltail/2.0)
            ) / weight ;
       cp = .85*(wt1 * (.5 * lkite + h2/3.0) +
             wt2 * .75 * lkite) / (wt1 + wt2) ;
       ar = w2 * w2 / area ;
       kbase = 0.0 ;
     }
     if (ktype == 3) { //box
       lkite = 2.0*h1 + h2 ;
       area = 4.0 * h1 * w1 ;
       lengstk = 4.0 * lkite + 4.0 * w1 ;
       weight = area * wtarea + lengstk * wtlngs + wtail ;
       cg = ((h1 + .5 * h2) * (weight - wtail) +
              wtail * (-ltail/2.0)
            ) / weight ;
       cp = h1 + .5 * h2 + .25*h1 ;
       ar = w1 / h1 ;
       kbase = w1 / 2.0 ;
     }
     if (ktype == 4) { //winged box
       lkite = 2.0*h1 + h2 ;
       area = 4.0* h1 * w1 + .5 * lkite * (w2 - w1) ;
       wt1 = .5 * lkite * (w2 - w1) * wtarea ;
       lengstk = 4.0 * lkite + 4.0 * w1 + w2 ;
       wt2 = (4.0* h1 * w1) * wtarea + (4.0 * lkite + 4.0 * w1)*wtlngs ;
       wt3 = w2 *  wtlngs ;
       weight = wt1 + wt2 + wt3 + wtail ;
       cg = (wt1 * (h1 + 2.0* h2) / 3.0 +
             wt2 * (h1 + .5 * h2) +
             wt3 * (h1 + h2) +
             wtail * (-ltail/2.0)
            ) / weight ;
       cp = (wt1 * (.75 * lkite - (.75*lkite - (h1+h2))/3.0) +
            ((4.0* h1 * w1) * wtarea)*( h1 + .5 * h2 + .25*h1))/
             (wt1 + ((4.0* h1 * w1) * wtarea)) ;
       ar = w2 / h1 ;
       kbase = w1 / 2.0 ;
     }
    if (ktype == 5) { // twin-trapezoid
       el = w2 / 2.0 - w1 / 2.0 ;
       lkite = Math.sqrt(el * el + h1 * h1) ;
       area = h1 * (h2 - w1 - el) ;
       wt1 = (h1 * h2) * wtarea;
       bet2 = Math.atan(el / h1) / convdr ;
       beta = 90.0 - bet2 ;
       lengstk = (h2 - w1) * Math.sin(convdr*beta) ;
       wt2 = lengstk *  wtlngs ;
       weight = wt1 + wt2 + wtail ;
       cg = (wt1 * lkite / 2.0 +
             wt2 * (lkite - (h2-w1)*Math.cos(convdr*beta)) +
             wtail * (-ltail/2.0)
            )/ weight ;
 // look at Cp carefully
       cp = .75 * lkite - (h2-w1)*Math.cos(convdr*beta)/2.0 ;
       ar = lengstk * lengstk / area ;
       kbase = 0.0 ;
    }
        
 // compute the bridle geometry - bridle point

     if (ktype == 2) { // special for sled
         if (lbrid < w2) lbrid = w2 + .01 ;
         lknot = lbrid ;

         ybr = h2 ;
         tana = lbrid /(w2 - w1) ;
         cosa = w1 / Math.sqrt(lbrid*lbrid + (w2-w1)*(w2-w1)) ;
         thet = 3.1415926 - Math.atan(tana) - Math.acos(cosa) ;
         xbr = .5*(w2 - w1) * Math.sin(thet) ; 
     }
     else {
         if (ktype == 5) {  // special for twintrap
            if (lknot > lkite) lknot = lkite - .1 ;
            ybr = lknot ;
            xbr = .4 ;
         }
         else {
            if (lbrid < lkite) lbrid = lkite + .01 ;
            if (lknot > lbrid) lknot = lbrid ;

            cosa = (lknot*lknot + lkite*lkite - (lbrid-lknot)*(lbrid-lknot))/
               (2.0 * lknot * lkite) ;
            anga = Math.acos(cosa) ;
            sina = Math.sin(anga) ;
            xbr = lknot * sina ;
            ybr = lknot * cosa ;
            xbr = xbr + kbase;
        }
     }

  // determine trim angle

     if (angmod > 0) { // angle input mode
        iter = 0;
        unstab = 0 ;
        s.c.o.up.o9.setForeground(Color.yellow) ;
        s.c.o.up.o10.setForeground(Color.yellow) ;
        alnew = angk =  alpha * convdr ;
        cosl = Math.cos(alnew) ;
        sinl = Math.sin(alnew) ;

        cl = 2.0 * pi * alnew / (1.0 + 2.0 * alnew / ar) ;
        cd = 1.28 * sinl + (cl*cl)/(.7 * pi * ar) ;

        lift = .5 * cl * density * wind * wind * area / 144.0 * 16.0 ;
        drag = .5 * cd * density * wind * wind * area / 144.0 * 16.0 ;
   
        torq = - lift * cosl * (ybr - cp) 
               - lift * sinl * xbr
               + drag * cosl * xbr
               - drag * sinl * (ybr - cp) 
               + weight * cosl * (ybr - cg)
               + weight * sinl * xbr ;
        if (lift > weight) {
            angl = Math.atan(drag / (lift - weight)) ;
        }
        else {
            angl = 0.0 ;
        }
     }
 
     if (angmod == 0) { // compute angle

    // load array for range of angles
        s.c.o.up.o9.setForeground(Color.green) ;
        s.c.o.up.o10.setForeground(Color.green) ;
        for(iter = 0; iter <=400; ++iter){
           alnew = iter* .125 * convdr ;
           cosl = Math.cos(alnew) ;
           sinl = Math.sin(alnew) ;

           cl = 2.0 * pi * alnew / (1.0 + 2.0 * alnew / ar) ;
           cd = 1.28 * sinl + (cl*cl)/(.7 * pi * ar) ;

           lift = .5 * cl * density * wind * wind * area / 144.0 * 16.0 ;
           drag = .5 * cd * density * wind * wind * area / 144.0 * 16.0 ;
   
           tor[iter] = - lift * cosl * (ybr - cp) 
                       - lift * sinl * xbr
                       + drag * cosl * xbr
                       - drag * sinl * (ybr - cp) 
                       + weight * cosl * (ybr - cg)
                       + weight * sinl * xbr ;
        }

   // check for stability -- find where torque sign changes 
        unstab = 1 ;
        iflag = 0 ;
        isave = 0 ;
        for(iter = 0; iter <=400; ++iter){
           if(tor[iter] < 0.0) {
              unstab = 0 ;
              if (iflag == 0) {
                 iflag = 1 ;
                 isave = iter ;
              }
              if (isave == 0) {
                 unstab = 1 ;
                 iflag = 0 ;
              }
           }
        }
    // load answer
        if (unstab == 0) {
           alpha = isave * .125 ;
           angk = alpha * convdr ;
           cosl = Math.cos(angk) ;
           sinl = Math.sin(angk) ;

           cl = 2.0 * pi * angk / (1.0 + 2.0 * angk / ar) ;
           cd = 1.28 * sinl + (cl*cl)/(.7 * pi * ar) ;

           lift = .5 * cl * density * wind * wind * area / 144.0 * 16.0 ;
           drag = .5 * cd * density * wind * wind * area / 144.0 * 16.0 ;
   
           torq = - lift * cosl * (ybr - cp) 
                  - lift * sinl * xbr
                  + drag * cosl * xbr
                  - drag * sinl * (ybr - cp) 
                  + weight * cosl * (ybr - cg)
                  + weight * sinl * xbr ;
           if (lift > weight) {
               angl = Math.atan(drag / (lift - weight)) ;
           }
           else {
               angl = 0.0 ;
           }
        }
     }

  // compute the kite altitude

     ground = 0 ;
     a1 = (lift - weight - lline*wtlngl);
     c1 = drag / wtlngl ;
     if (a1 <= 0.0 ) {
         a1 = 0.0 ;
         ground = 1;
     }
     con1 = isinh(a1/drag) ;
     con2 = -c1 * cosh(con1) ;

     a2 = (lift - weight) / drag ;
     if (a2 <= 0.0 ) a2 = 0.0 ;
     a3 = isinh(a2) ;
     xkite = c1 * (a3 - con1) ;
     if (c1 > 0.0) {
       ykite = con2 + c1 * cosh((xkite/c1) + con1) ;
     }
     else {
       ykite = 0.0 ;
     }

  // compute the line shape
 
     delx = xkite / 20.0 ;
     for(iter = 0; iter <=20; ++iter){
        xline[iter] = iter * delx ;
        if (c1 > 0.0) {
           yline[iter] = con2 + c1 * cosh((xline[iter]/c1) + con1) ;
        }
        else {
          ykite = 0.0 ;
        }
     }

  // compute the line tension

     tension = Math.sqrt(a1 * a1 + drag * drag) ;

     loadOut() ;
  }

  public int filter0(double inumbr) {
        //  output only to .
       int number ;
       int intermed ;
 
       number = (int) (inumbr);
       return number ;
  }

  public float filter1(double inumbr) {
     //  output only to .1
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 10.) ;
       number = (float) (intermed / 10. );
       return number ;
  }
 
  public float filter3(double inumbr) {
     //  output only to .001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 1000.) ;
       number = (float) (intermed / 1000. );
       return number ;
  }
 
  public float filter5(double inumbr) {
     //  output only to .00001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 100000.) ;
       number = (float) (intermed / 100000. );
       return number ;
  }
 
  public double cosh(double inumbr) {
     //  hyperbolic cosine
       double number ;
 
       number = 1.0 
              + Math.pow(inumbr,2)/2.0 
              + Math.pow(inumbr,4)/24.0
              + Math.pow(inumbr,6)/720.0 
              + Math.pow(inumbr,8)/40320. ;
       return number ;
  }
 
  public double isinh(double inumbr) {
     //  inverse hyperbolic sine
       double number ;
 
       if (inumbr * inumbr <= 1.0) {
          number = inumbr 
                 - Math.pow(inumbr,3)/6.0 
                 + 3.0  * Math.pow(inumbr,5)/40.0
                 - 15.0 * Math.pow(inumbr,7)/336. 
                 + 21.0 * Math.pow(inumbr,9)/3456. ; 
       }
       else {
          number = Math.log(2.0*inumbr) 
                 + 1.0   / (4.0    * Math.pow(inumbr,2)) 
                 - 3.0   / (32.0   * Math.pow(inumbr,4)) 
                 + 15.0  / (288.0  * Math.pow(inumbr,6))
                 - 105.0 / (3072.0 * Math.pow(inumbr,8)) ;
       }
       return number ;
  }
 
  public void loadInput() {   // load the input panels
       int i1,i2,i3,i4,i5,i6 ;
       double v1,v2,v3,v4,v5,v6 ;
       float fl1,fl2,fl3,fl4,fl5,fl6 ;

       h1d = h1 * lconv ;
       h2d = h2 * lconv ;
       w1d = w1 * lconv ;
       w2d = w2 * lconv ;

       fl1 = (float) h1d ;
       fl2 = (float) h2d ;
       fl3 = (float) w1d ;
       fl4 = (float) w2d ;

       hmnd = hmn * lconv ;
       hmxd = hmx * lconv ;
       wmnd = wmn * lconv ;
       wmxd = wmx * lconv ;

       i1 = (int) (((h1d - hmnd)/(hmxd-hmnd))*1000.) ;
       i2 = (int) (((h2d - hmnd)/(hmxd-hmnd))*1000.) ;
       i3 = (int) (((w1d - wmnd)/(wmxd-wmnd))*1000.) ;
       i4 = (int) (((w2d - wmnd)/(wmxd-wmnd))*1000.) ;

       s.c.i.inshape.l.f1.setText(String.valueOf(fl1)) ;
       s.c.i.inshape.l.f2.setText(String.valueOf(fl2)) ;
       s.c.i.inshape.l.f3.setText(String.valueOf(fl3)) ;
       s.c.i.inshape.l.f4.setText(String.valueOf(fl4)) ;

       s.c.i.inshape.r.s1.setValue(i1) ;
       s.c.i.inshape.r.s2.setValue(i2) ;
       s.c.i.inshape.r.s3.setValue(i3) ;
       s.c.i.inshape.r.s4.setValue(i4) ;

       if (ktype < 5) {
         if (lbrid < lkite) lbrid = lkite + .01 ;
         if (lknot > lbrid) lknot = lbrid ;
       }

       lbridd = lbrid * lconv ;
       lknotd = lknot * lconv ;
       ltaild = ltail * lconv ;

       fl1 = (float) lbridd ;
       fl2 = (float) lknotd ;
       fl3 = (float) alpha ;
       fl4 = (float) ltaild ;

       lbrmnd = lbrmn * lconv ;
       lbrmxd = lbrmx * lconv ;
       ltlmnd = ltlmn * lconv ;
       ltlmxd = ltlmx * lconv ;

       i1 = (int) (((lbridd - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
       i2 = (int) (((lknotd - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
       i3 = (int) (((alpha - almn)/(almx-almn))*1000.) ;
       i4 = (int) (((ltaild - ltlmnd)/(ltlmxd-ltlmnd))*1000.) ;

       s.c.i.intrim.l.f1.setText(String.valueOf(fl1)) ;
       s.c.i.intrim.l.f2.setText(String.valueOf(fl2)) ;
       s.c.i.intrim.l.f3.setText(String.valueOf(fl3)) ;
       s.c.i.intrim.l.f4.setText(String.valueOf(fl4)) ;

       s.c.i.intrim.r.s1.setValue(i1) ;
       s.c.i.intrim.r.s2.setValue(i2) ;
       s.c.i.intrim.r.s3.setValue(i3) ;
       s.c.i.intrim.r.s4.setValue(i4) ;

       windd = wind * hconv ;
       llined = lline * hconv ;

       fl1 = (float) windd ;
       fl3 = (float) llined ;

       wndmnd = wndmn * hconv ;
       wndmxd = wndmx * hconv ;
       llnmnd = llnmn * hconv ;
       llnmxd = llnmx * hconv ;

       i1 = (int) (((windd - wndmnd)/(wndmxd-wndmnd))*1000.) ;
       i3 = (int) (((llined - llnmnd)/(llnmxd-llnmnd))*1000.) ;

       s.c.i.inflight.l.f1.setText(String.valueOf(fl1)) ;
       s.c.i.inflight.l.f3.setText(String.valueOf(fl3)) ;
 
       s.c.i.inflight.r.s1.setValue(i1) ;
       s.c.i.inflight.r.s3.setValue(i3) ;

       fl1 = (float) (wtarea * wconv / lconv / lconv) ;
       fl2 = (float) (wtlngs * wconv / lconv)  ;
       fl3 = (float) (wtlngl * wconv / hconv)  ;
       fl4 = (float) (wttail * wconv / lconv)  ;

       s.c.i.inmate.f1.setText(String.valueOf(fl1)) ;
       s.c.i.inmate.f2.setText(String.valueOf(fl2)) ;
       s.c.i.inmate.f3.setText(String.valueOf(fl3)) ;
       s.c.i.inmate.f4.setText(String.valueOf(fl4)) ;

       return ;
  }

  public void loadOut() {   // output routine
     int i3 ;
     double v3 ;
     float fl3 ;
 
     s.c.o.up.o3.setText(String.valueOf(filter3(weight * wconv))) ;
     s.c.o.up.o7.setText(String.valueOf(filter3(lift * wconv))) ;
     s.c.o.up.o8.setText(String.valueOf(filter3(drag * wconv))) ;
     s.c.o.up.o9.setText(String.valueOf(filter3(torq))) ;
     s.c.o.up.o10.setText(String.valueOf(filter3(alpha))) ;

     s.c.o.dn.odes.o1.setText(String.valueOf(filter3(area *lconv * lconv))) ;
     s.c.o.dn.odes.o2.setText(String.valueOf(filter3(lengstk *lconv))) ;
     s.c.o.dn.odes.o4.setText(String.valueOf(filter3(cg *lconv))) ;
     s.c.o.dn.odes.o5.setText(String.valueOf(filter3(cp *lconv))) ;
     s.c.o.dn.odes.o6.setText(String.valueOf(filter3(ar))) ;
     s.c.o.dn.odes.o16.setText(String.valueOf(filter0(ykite * hconv))) ;

     s.c.o.dn.oflt.o12.setText(String.valueOf(filter0(xkite * hconv))) ;
     s.c.o.dn.oflt.o13.setText(String.valueOf(filter0(ykite * hconv))) ;
     s.c.o.dn.oflt.o14.setText(String.valueOf(filter3(wind * hconv))) ;
     s.c.o.dn.oflt.o15.setText(String.valueOf(filter3(cg * lconv))) ;
     s.c.o.dn.oflt.o17.setText(String.valueOf(filter3(tension * wconv))) ;

     return ;
  }
 
  public void printData() {
     switch(ktype) {
        case 0: {
           p.prnt.appendText( "\n\n Diamond Kite -->" ) ;
           break ;
        }
        case 1: {
           p.prnt.appendText( "\n\n Delta Kite -->" ) ;
           break ;
        }
        case 2: {
           p.prnt.appendText( "\n\n Sled Kite -->" ) ;
           break ;
        }
        case 3: {
           p.prnt.appendText( "\n\n Box Kite -->" ) ;
           break ;
        }
        case 4: {
           p.prnt.appendText( "\n\n Winged Box -->" ) ;
           break ;
        }
        case 5: {
           p.prnt.appendText( "\n\n Twin-Trap -->" ) ;
           break ;
        }
     }
     switch(surtyp) {    // surface  material
        case 0: {
           p.prnt.appendText( " Plastic Surface," ) ;
           break ;
        }
        case 1: {
           p.prnt.appendText( " Tissue Surface," ) ;
           break ;
        }
        case 2: {
           p.prnt.appendText( " Rip Stop Surface," ) ;
           break ;
        }
        case 3: {
           p.prnt.appendText( " Paper Surface," ) ;
           break ;
        }
        case 4: {
           p.prnt.appendText( " Silk Span Surface," ) ;
           break ;
        }
        case 5: {
           p.prnt.appendText( " Specified Surface," ) ;
           break ;
        }
     }
     switch(stktyp) {         // frame material
        case 0: {
           p.prnt.appendText( " 1/4 Balsa Frame," ) ;
           break ;
        }
        case 1: {
           p.prnt.appendText( " 1/8 Blasa Frame," ) ;
           break ;
        }
        case 2: {
           p.prnt.appendText( " 1/4 Birch Frame," ) ;
           break ;
        }
        case 3: {
           p.prnt.appendText( " 3/8 Plastic Frame," ) ;
           break ;
        }
        case 4: {
           p.prnt.appendText( " Skewer Frame," ) ;
           break ;
        }
        case 5: {
           p.prnt.appendText( " Specified Frame," ) ;
           break ;
        }
     }
     switch(taltyp) {   /// tail material
        case 0: {
           p.prnt.appendText( " 1 inch Plastic Tail," ) ;
           break ;
        }
        case 1: {
           p.prnt.appendText( " 3 inch Plastic Tail," ) ;
           break ;
        }
        case 2: {
           p.prnt.appendText( " 1 inch Nylon Tail," ) ;
           break ;
        }
        case 3: {
           p.prnt.appendText( " Specified Tail," ) ;
           break ;
        }
     }
     switch(lintyp) {   /// line material
        case 0: {
           p.prnt.appendText( " Nylon Line" ) ;
           break ;
        }
        case 1: {
           p.prnt.appendText( " Cotton Line" ) ;
           break ;
        }
        case 2: {
           p.prnt.appendText( " Specified Line" ) ;
           break ;
        }
     }
     p.prnt.appendText( "\n H1 = " + filter1(h1d) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;
     p.prnt.appendText( " H2 = " + filter1(h2d) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;
     p.prnt.appendText( " W1 = " + filter1(w1d) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;
     p.prnt.appendText( " W2 = " + filter1(w2d) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;
     p.prnt.appendText( " Tail = " + filter1(ltaild) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;

     p.prnt.appendText( "\n Surface = " + filter1(area*lconv*lconv) ) ;
     if (units == 1) p.prnt.appendText( " sq in," ) ;
     if (units == -1) p.prnt.appendText( "sq  cm," ) ;
     p.prnt.appendText( " Frame = " + filter1(lengstk*lconv) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;
     p.prnt.appendText( " AR = " + filter3(ar) ) ;
     p.prnt.appendText( "," ) ;
     p.prnt.appendText( " Cg = " + filter3(cg*lconv) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;
     p.prnt.appendText( " Cp = " + filter3(cp*lconv) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;

     p.prnt.appendText( "\n Bridle = " + filter1(lbridd) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;
     p.prnt.appendText( " Knot = " + filter1(lknotd) ) ;
     if (units == 1) p.prnt.appendText( " in," ) ;
     if (units == -1) p.prnt.appendText( " cm," ) ;
     p.prnt.appendText( " Trim Angle = " + filter3(alpha) ) ;
     p.prnt.appendText( " degrees," ) ;
     p.prnt.appendText( " Torque = " + filter3(torq*wconv*lconv) ) ;
     if (units == 1) p.prnt.appendText( " oz-in," ) ;
     if (units == -1) p.prnt.appendText( " gm-cm," ) ;

     p.prnt.appendText( " \n Weight = " + filter3(weight*wconv) ) ;
     if (units == 1) p.prnt.appendText( " oz," ) ;
     if (units == -1) p.prnt.appendText( " gm," ) ;
     p.prnt.appendText( " Lift = " + filter3(lift*wconv) ) ;
     if (units == 1) p.prnt.appendText( " oz," ) ;
     if (units == -1) p.prnt.appendText( " gm," ) ;
     p.prnt.appendText( " Drag = " + filter3(drag*wconv) ) ;
     if (units == 1) p.prnt.appendText( " oz," ) ;
     if (units == -1) p.prnt.appendText( " gm," ) ;
     p.prnt.appendText( " Tension = " + filter3(tension*wconv) ) ;
     if (units == 1) p.prnt.appendText( " oz," ) ;
     if (units == -1) p.prnt.appendText( " gm," ) ;

     p.prnt.appendText( " \n Wind = " + filter1(wind*hconv) ) ;
     if (units == 1) p.prnt.appendText( " ft/sec," ) ;
     if (units == -1) p.prnt.appendText( " m/sec," ) ;
     p.prnt.appendText( " Line = " + filter0(llined) ) ;
     if (units == 1) p.prnt.appendText( " ft," ) ;
     if (units == -1) p.prnt.appendText( " m," ) ;
     p.prnt.appendText( " Far = " + filter0(xkite*hconv) ) ;
     if (units == 1) p.prnt.appendText( " ft," ) ;
     if (units == -1) p.prnt.appendText( " m," ) ;
     p.prnt.appendText( " Height = " + filter0(ykite*hconv) ) ;
     if (units == 1) p.prnt.appendText( " ft," ) ;
     if (units == -1) p.prnt.appendText( " m," ) ;
  }

  class Solver {
 
     Solver () {
     }

     public void setDefaults() {

        ktype = 0;

        h1d = h1 = 5.0 ;
        h2d = h2 = 10.0 ;
        w1d = w1 = 10.0 ;
        w2d = w2 = 20.0 ;
        lkite = 15. ;
        kbase = 0.0 ;
        area = 75.0;
        lengstk = 25.0 ;
        hmnd = hmn = .5 ;  hmxd = hmx = 100. ;
        wmnd = wmn = .5 ;  wmxd = wmx = 100. ;
        surtyp = 0; wtarea = .0004752 ; 
        stktyp = 0; wtlngs = .003952; 
        lintyp = 0; wtlngl=.004;
        taltyp = 0; wttail=.0004752;

        lbridd = lbrid = 15.5 ;
        lknotd = lknot = 12. ;
        angmod = 0 ;
        alpha = 5.0 ;
        ltaild = ltail = 12.0 ;
        llined = lline = 100. ;
        windd = wind = 10.0 ;
        lbrmnd = lbrmn = 1.0 ; lbrmxd = lbrmx = 70. ; 
        ltlmnd = ltlmn = 0.0 ; ltlmxd = ltlmx = 60. ;
        llnmnd = llnmn = 0.0 ; llnmxd = llnmx = 1000. ;
        wndmnd = wndmn = 0.5 ; wndmxd = wndmx = 25. ;
        almn = 0.0 ; almx = 45.0 ;
        density = .00237 ;

        fact = 10.0;
        vscale = 10.0 ;
        fscale = .25 ;
        xt = 150; yt =225 ;
        sldloc= 70;
        vldloc= 150;
        fldloc= 150;
        viewflg = 0 ;
        pick = -1 ;

        units = 1 ;
        lconv = 1.0 ;
        wconv = 1.0 ;
        hconv = 1.0 ;

        fold = -1 ;
        ground = 0 ;
        return ;
     }
  }

  class S extends Panel {
     Kite outerparent ;
     Viewer view ;
     C c ;

     S (Kite target) { 
         outerparent = target ;
         setLayout(new GridLayout(1,2,5,5)) ;

         view  = new Viewer(outerparent) ;
         c  = new C(outerparent) ;

         add(view) ;
         add(c) ;
     }

     class C extends Panel {
        Kite outerparent ;
        I i ;
        O o ;

        C (Kite target) { 
          outerparent = target ;
          setLayout(new GridLayout(2,1,5,5)) ;

          i = new I(outerparent) ;
          o = new O(outerparent) ;

          add(o) ;
          add(i) ;
        }

        class I extends Panel {
           Kite outerparent ;
           Inshape inshape;
           Inmate inmate ;
           Inflight inflight ;
           Intrim intrim ;
   
           I (Kite target) { 
             outerparent = target ;
             layin = new CardLayout() ;
             setLayout(layin) ;
    
             inshape = new Inshape(outerparent) ;
             inmate = new Inmate(outerparent) ;
             inflight = new Inflight(outerparent) ;
             intrim = new Intrim(outerparent) ;
   
             add ("first", inshape) ;
             add ("second", inmate) ;
             add ("third", inflight) ;
             add ("fourth", intrim) ;
           }
 
           class Inshape extends Panel {
              Kite outerparent ;
              L l ;
              R r ;
   
              Inshape (Kite target) {
   
                 outerparent = target ;
                 setLayout(new GridLayout(1,2,5,5)) ;
   
                 l = new L(outerparent) ;
                 r = new R(outerparent) ;
   
                 add(l) ;
                 add(r) ;
              }
   
              class L extends Panel {
                 Kite outerparent ;
                 TextField f1,f2,f3,f4 ;
                 Button bt1,bt2 ;
                 Label l1,l2,l3,l4 ;
   
                 L (Kite target) {
   
                  outerparent = target ;
                  setLayout(new GridLayout(5,2,2,10)) ;
 
                  bt1 = new Button("Shape") ;
                  bt1.setBackground(Color.blue) ;
                  bt1.setForeground(Color.white) ;
   
                  bt2 = new Button("Material") ;
                  bt2.setBackground(Color.white) ;
                  bt2.setForeground(Color.blue) ;
   
                  l1 = new Label("H1 - in", Label.CENTER) ;
                  f1 = new TextField("5.0",5) ;
   
                  l2 = new Label("H2 - in", Label.CENTER) ;
                  f2 = new TextField("10.0",5) ;
   
                  l3 = new Label("W1 - in", Label.CENTER) ;
                  f3 = new TextField("10.0",5) ;
   
                  l4 = new Label("W2 - in", Label.CENTER) ;
                  f4 = new TextField("20.0",5) ;
   
                  add(bt1) ;
                  add(bt2) ;
   
                  add(l1) ;
                  add(f1) ;
   
                  add(l2) ;
                  add(f2) ;
   
                  add(l3) ;
                  add(f3) ;
   
                  add(l4) ;
                  add(f4) ;
                }
    
                public boolean action(Event evt, Object arg) {
                    if(evt.target instanceof Button) {
                       this.handleDispla(arg) ;
                       return true ;
                    }
                    if(evt.id == Event.ACTION_EVENT) {
                       this.handleText(evt) ;
                       return true ;
                    }
                    else return false ;
                }

                public void handleText(Event evt) {
                  Double V1,V2,V3,V4 ;
                  double v1,v2,v3,v4 ;
                  float fl1 ;
                  int i1,i2,i3,i4 ;
   
                  V1 = Double.valueOf(f1.getText()) ;
                  v1 = V1.doubleValue() ;
                  V2 = Double.valueOf(f2.getText()) ;
                  v2 = V2.doubleValue() ;
                  V3 = Double.valueOf(f3.getText()) ;
                  v3 = V3.doubleValue() ;
                  V4 = Double.valueOf(f4.getText()) ;
                  v4 = V4.doubleValue() ;
   
                  h1d = v1 ;
                  if(h1d < hmnd) {
                    h1d = v1 = hmnd ;
                    fl1 = (float) v1 ;
                    f1.setText(String.valueOf(fl1)) ;
                  }
                  if(h1d > hmxd) {
                    h1d = v1 = hmxd ;
                    fl1 = (float) v1 ;
                    f1.setText(String.valueOf(fl1)) ;
                  }
                  h1 = h1d / lconv ;
 
                  h2d = v2 ;
                  if(h2d < hmnd) {
                    h2d = v2 = hmnd ;
                    fl1 = (float) v2 ;
                    f2.setText(String.valueOf(fl1)) ;
                  }
                  if(h2d > hmxd) {
                    h2d = v2 = hmxd ;
                    fl1 = (float) v2 ;
                    f2.setText(String.valueOf(fl1)) ;
                  }
                  h2 = h2d / lconv ;
 
                  w1d = v3 ;
                  if (ktype == 5) {
                     if(w1d > h2d) {
                       w1d = v3 = h2d - .1 ;
                       fl1 = (float) v3 ;
                       f3.setText(String.valueOf(fl1)) ;
                     }
                  }
                  if(w1d < wmnd) {
                    w1d = v3 = wmnd ;
                    fl1 = (float) v3 ;
                    f3.setText(String.valueOf(fl1)) ;
                  }
                  if(w1d > wmxd) {
                    w1d = v3 = wmxd ;
                    fl1 = (float) v3 ;
                    f3.setText(String.valueOf(fl1)) ;
                  }
                  if(w1d > w2d) {
                    w2d = v4 = w1d +.2 ;
                    fl1 = (float) v4 ;
                    f4.setText(String.valueOf(fl1)) ;
                  }
                  w1 = w1d / lconv ;
  
                  w2d = v4 ;
                  if (ktype == 5) {
                     if(w2d > h2d) {
                       w2d = v3 = h2d - .1 ;
                       fl1 = (float) v3 ;
                       f3.setText(String.valueOf(fl1)) ;
                     }
                  }
                  if(w2d < wmnd) {
                    w2d = v4 = wmnd ;
                    fl1 = (float) v4 ;
                    f4.setText(String.valueOf(fl1)) ;
                  }
                  if(w2d > wmxd) {
                    w2d = v4 = wmxd ;
                    fl1 = (float) v4 ;
                    f4.setText(String.valueOf(fl1)) ;
                  }
                  w2 = w2d / lconv ;
   
                  i1 = (int) (((v1 - hmnd)/(hmxd-hmnd))*1000.) ;
                  i2 = (int) (((v2 - hmnd)/(hmxd-hmnd))*1000.) ;
                  i3 = (int) (((v3 - wmnd)/(wmxd-wmnd))*1000.) ;
                  i4 = (int) (((v4 - wmnd)/(wmxd-wmnd))*1000.) ;
   
                  r.s1.setValue(i1) ;
                  r.s2.setValue(i2) ;
                  r.s3.setValue(i3) ;
                  r.s4.setValue(i4) ;
   
                  compute();
                } // TextHandler

                public void handleDispla(Object arg) {
                  String label = (String)arg ;
 
                  if (label.equals("Shape")) {
                     layin.show(i, "first")  ;
                  }

                  if (label.equals("Material")) {
                     layin.show(i, "second")  ;
                  }
                }  // Button handler
              }  // Inleft
 
              class R extends Panel {
                 Kite outerparent ;
                 Scrollbar s1,s2,s3,s4;
                 Choice ktch;
   
                 R (Kite target) {
                  int i1,i2,i3,i4 ;
   
                  outerparent = target ;
                  setLayout(new GridLayout(5,1,2,10)) ;
   
                  ktch = new Choice() ;
                  ktch.addItem("Diamond Kite") ;
                  ktch.addItem("Delta Kite");
                  ktch.addItem("Sled Kite");
                  ktch.addItem("Box Kite");
                  ktch.addItem("Winged Box Kite");
                  ktch.addItem("Twin-Trap Kite");
                  ktch.setBackground(Color.white) ;
                  ktch.setForeground(Color.red) ;
                  ktch.select(0) ;

                  i1 = (int) (((5.0 - hmn)/(hmx-hmn))*1000.) ;
                  i2 = (int) (((10.0 - hmn)/(hmx-hmn))*1000.) ;
                  i3 = (int) (((10.0 - wmn)/(wmx-wmn))*1000.) ;
                  i4 = (int) (((20.0 - wmn)/(wmx-wmn))*1000.) ;
   
                  s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                  s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);
                  s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
                  s4 = new Scrollbar(Scrollbar.HORIZONTAL,i4,10,0,1000);
   
                  add(ktch) ;
                  add(s1) ;
                  add(s2) ;
                  add(s3) ;
                  add(s4) ;
                }
      
                public boolean handleEvent(Event evt) {
                     if(evt.id == Event.ACTION_EVENT) {
                        this.handleCho() ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_ABSOLUTE) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_LINE_DOWN) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_LINE_UP) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_PAGE_DOWN) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_PAGE_UP) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     else return false ;
                }

                public void handleBar(Event evt) {
                   int i1,i2,i3,i4 ;
                   double v1,v2,v3,v4 ;
                   float fl1,fl2,fl3,fl4 ;
            // Input    for computations
                   i1 = s1.getValue() ;
                   i2 = s2.getValue() ;
                   i3 = s3.getValue() ;
                   i4 = s4.getValue() ;
   
                   if (ktype == 5) {
                     if (i3 > i2) {
                        i3 = i2 - 1 ;
                        s3.setValue(i3) ;
                     }
                     if (i4 > i2) {
                        i4 = i2 - 1 ;
                        s4.setValue(i3) ;
                     }
                   }

                   if (i3 > i4) {
                     i4 = i3 + 1 ;
                     s4.setValue(i4) ;
                   }

                   h1d = v1 = i1 * (hmxd - hmnd)/ 1000. + hmnd ;
                   h1 = h1d / lconv ;
                   h2d = v2 = i2 * (hmxd - hmnd)/ 1000. + hmnd ;
                   h2 = h2d / lconv ;
                   w1d = v3 = i3 * (wmxd - wmnd)/ 1000. + wmnd ;
                   w1 = w1d / lconv ;
                   w2d = v4 = i4 * (wmxd - wmnd)/ 1000. + wmnd ;
                   w2 = w2d / lconv ;
   
                   fl1 = (float) v1 ;
                   fl2 = (float) v2 ;
                   fl3 = (float) v3 ;
                   fl4 = (float) v4 ;
   
                   l.f1.setText(String.valueOf(fl1)) ;
                   l.f2.setText(String.valueOf(fl2)) ;
                   l.f3.setText(String.valueOf(fl3)) ;
                   l.f4.setText(String.valueOf(fl4)) ;
                   
                   compute() ;
                } // handle bar
 
                public void handleCho() {
                   int i1,i2 ;
                   double v1,v2 ;
                   double wdisplay ;
                   float fl1,fl2 ;
   
                   ktype = ktch.getSelectedIndex() ;

                   if (ktype == 0) {//default for diamond
                      h1 = .5 * w1 ;
                      h2 =  w1 ;
                      lbrid = 1.55 * w1 ;
                      lknot = 1.2 * w1 ;
                   }
                   if (ktype == 1) {//default for delta
                      h1 = .5 * w1 ;
                      h2 = .15 * w1 ;
                      lbrid = .85 * w1 ;
                      lknot = .5 * w1 ;
                   }
                   if (ktype == 2) {//default for sled
                      h1 = .5 * w1 ;
                      h2 =  w1 ;
                      w2 =  2.0 * w1 ;
                      lbrid = 1.5 * w1 ;
                      lknot = 1.5 * w1 ;
                   }
                   if (ktype == 3) {//default for box
                      h1 = .5 * w1 ;
                      h2 =  w1 ;
                      lbrid = 2.25 * w1 ;
                      lknot = 1.5 * w1 ;
                   }
                   if (ktype == 4) {//default for winged box
                      h1 = .5 * w1 ;
                      h2 =  w1 ;
                      w2 =  2.0 * w1 ;
                      lbrid = 2.25 * w1 ;
                      lknot = 1.5 * w1 ;
                   }
                   if (ktype == 5) {//defaults for twin trap
                      h1 = 8.5 ;
                      h2 = 11.0 ;
                      w1 = 2.75 ;
                      w2 = 7.0 ;

                      wtarea = .0012832 ;
                      wdisplay = wtarea * wconv / lconv / lconv ;
                      fl1 = (float) wdisplay ;
                      inmate.f1.setText(String.valueOf(fl1)) ;
                      inmate.matsur.select(3) ;

                      wtlngs = .00090 ;
                      wdisplay = wtlngs * wconv / lconv ;
                      fl1 = (float) wdisplay ;
                      inmate.f2.setText(String.valueOf(fl1)) ;
                      inmate.matstk.select(4) ;
                   }
 
                   w1d = w1 * lconv ;
                   w2d = w2 * lconv ;
                   h1d = h1 * lconv ;
                   h2d = h2 * lconv ;
                   lbridd = lbrid * lconv ;
                   lknotd = lknot * lconv ;

                   fl1 = (float) h1d ;
                   i1 = (int) (((h1d - hmnd)/(hmxd-hmnd))*1000.) ;
                   l.f1.setText(String.valueOf(fl1)) ;
                   s1.setValue(i1) ;

                   fl1 = (float) h2d ;
                   i1 = (int) (((h2d - hmnd)/(hmxd-hmnd))*1000.) ;
                   l.f2.setText(String.valueOf(fl1)) ;
                   s2.setValue(i1) ;

                   fl1 = (float) w1d ;
                   i1 = (int) (((w1d - wmnd)/(wmxd-wmnd))*1000.) ;
                   l.f3.setText(String.valueOf(fl1)) ;
                   s3.setValue(i1) ;

                   fl1 = (float) w2d ;
                   i1 = (int) (((w2d - wmnd)/(wmxd-wmnd))*1000.) ;
                   l.f4.setText(String.valueOf(fl1)) ;
                   s4.setValue(i1) ;

                   fl1 = (float) lbridd ;
                   i1 = (int) (((lbridd - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
                   intrim.l.f1.setText(String.valueOf(fl1)) ;
                   intrim.r.s1.setValue(i1) ;

                   fl1 = (float) lknotd ;
                   i1 = (int) (((lknotd - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
                   intrim.l.f2.setText(String.valueOf(fl1)) ;
                   intrim.r.s2.setValue(i1) ;

                   compute() ;
                } // handleCho
              }  // Inright
           }  // Inshape

           class Inmate extends Panel {
              Kite outerparent ;
              Label l1,l2,l3,l4 ;
              Choice matsur,matstk,matlin,matail;
              TextField f1,f2,f3,f4 ;
              Button bt1,bt2;
   
              Inmate (Kite target) {
                 outerparent = target ;
                 setLayout(new GridLayout(5,3,5,5)) ;
    
                 bt1 = new Button("Shape") ;
                 bt1.setBackground(Color.white) ;
                 bt1.setForeground(Color.blue) ;
   
                 bt2 = new Button("Material") ;
                 bt2.setBackground(Color.blue) ;
                 bt2.setForeground(Color.white) ;

                 l1 = new Label("Surface-oz/sq in", Label.CENTER) ;
                 matsur = new Choice() ;
                 matsur.addItem("Plastic") ;
                 matsur.addItem("Tissue Paper");
                 matsur.addItem("Rip Stop");
                 matsur.addItem("Paper");
                 matsur.addItem("Silk Span");
                 matsur.addItem("Cellophane");
                 matsur.addItem("<-- Specify");
                 matsur.setBackground(Color.white) ;
                 matsur.setForeground(Color.red) ;
                 matsur.select(0) ;
                 f1 = new TextField(".0004752",5) ;
                 f1.setBackground(Color.black) ;
                 f1.setForeground(Color.green) ;

                 l2 = new Label("Frame-oz/in", Label.CENTER) ;
                 matstk = new Choice() ;
                 matstk.addItem("1/4 Balsa") ;
                 matstk.addItem("1/8 Balsa");
                 matstk.addItem("1/4 Birch");
                 matstk.addItem("3/8 Plastic Tube");
                 matstk.addItem("Skewer Stick");
                 matstk.addItem("<-- Specify");
                 matstk.setBackground(Color.white) ;
                 matstk.setForeground(Color.red) ;
                 matstk.select(0) ;
                 f2 = new TextField(".003952",5) ;
                 f2.setBackground(Color.black) ;
                 f2.setForeground(Color.green) ;

                 l3 = new Label("Line-oz/ft", Label.CENTER) ;
                 matlin = new Choice() ;
                 matlin.addItem("Nylon") ;
                 matlin.addItem("Cotton");
                 matlin.addItem("<-- Specify");
                 matlin.setBackground(Color.white) ;
                 matlin.setForeground(Color.red) ;
                 matlin.select(0) ;
                 f3 = new TextField(".004",5) ;
                 f3.setBackground(Color.black) ;
                 f3.setForeground(Color.green) ;

                 l4 = new Label("Tail-oz/in", Label.CENTER) ;
                 matail = new Choice() ;
                 matail.addItem("1 in Plastic") ;
                 matail.addItem("3 in Plastic") ;
                 matail.addItem("1 in Nylon");
                 matail.addItem("<-- Specify");
                 matail.setBackground(Color.white) ;
                 matail.setForeground(Color.red) ;
                 matail.select(0) ;
                 f4 = new TextField(".004752",5) ;
                 f4.setBackground(Color.black) ;
                 f4.setForeground(Color.green) ;

                 add(bt1) ;
                 add(bt2) ;
                 add(new Label("Density", Label.LEFT)) ;

                 add(l1) ;
                 add(f1) ;
                 add(matsur) ;

                 add(l2) ;
                 add(f2) ;
                 add(matstk) ;

                 add(l4) ;
                 add(f4) ;
                 add(matail) ;

                 add(l3) ;
                 add(f3) ;
                 add(matlin) ;
              }

              public boolean action(Event evt, Object arg) {
                 if(evt.target instanceof Button) {
                    this.handleDispla(arg) ;
                    return true ;
                 }
                 if(evt.id == Event.ACTION_EVENT) {
                    this.handleText(evt) ;
                    return true ;
                 }
                 else return false ;
              }

              public void handleText(Event evt) {
                 Double V1,V2,V3,V4 ;
                 double v1,v2,v3,v4 ;
                 double wdisplay ;
                 float fl1 ;
  
                 surtyp = matsur.getSelectedIndex() ;
                 stktyp = matstk.getSelectedIndex() ;
                 lintyp = matlin.getSelectedIndex() ;
                 taltyp = matail.getSelectedIndex() ;

                 switch (surtyp) { // surface weight per area
                     case 0: {    // plastic 
                         wtarea = .0004752 ;
                         wdisplay = wtarea * wconv / lconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f1.setText(String.valueOf(fl1)) ;
                         f1.setBackground(Color.black) ;
                         f1.setForeground(Color.green) ;
                         break;
                     }
                     case 1: {   // tissue
                         wtarea = .0004576 ;
                         wdisplay = wtarea * wconv / lconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f1.setText(String.valueOf(fl1)) ;
                         f1.setBackground(Color.black) ;
                         f1.setForeground(Color.green) ;
                         break ;
                     }
                     case 2: {   // rip stop
                         wtarea = .0009552 ;
                         wdisplay = wtarea * wconv / lconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f1.setText(String.valueOf(fl1)) ;
                         f1.setBackground(Color.black) ;
                         f1.setForeground(Color.green) ;
                         break ;
                     }
                     case 3: {   // paper
                         wtarea = .0012832 ;
                         wdisplay = wtarea * wconv / lconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f1.setText(String.valueOf(fl1)) ;
                         f1.setBackground(Color.black) ;
                         f1.setForeground(Color.green) ;
                         break ;
                     }
                     case 4: {   // silk span
                         wtarea = .000384 ;
                         wdisplay = wtarea * wconv / lconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f1.setText(String.valueOf(fl1)) ;
                         f1.setBackground(Color.black) ;
                         f1.setForeground(Color.green) ;
                         break ;
                     }
                     case 5: {   // celophane
                         wtarea = .000656 ;
                         wdisplay = wtarea * wconv / lconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f1.setText(String.valueOf(fl1)) ;
                         f1.setBackground(Color.black) ;
                         f1.setForeground(Color.green) ;
                         break ;
                     }
                     case 6: {   // specify
                         V1 = Double.valueOf(f1.getText()) ;
                         wdisplay = V1.doubleValue() ;
                         wtarea = wdisplay * lconv * lconv / wconv ;
                         f1.setBackground(Color.white) ;
                         f1.setForeground(Color.black) ;
                         break ;
                     }
                 }

                 switch (stktyp) { // stick weight per length
                     case 0: {    // quarter balsa
                         wtlngs = .003952 ;
                         wdisplay = wtlngs * wconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f2.setText(String.valueOf(fl1)) ;
                         f2.setBackground(Color.black) ;
                         f2.setForeground(Color.green) ;
                         break;
                     }
                     case 1: {   // eighth balsa
                         wtlngs = .001968 ;
                         wdisplay = wtlngs * wconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f2.setText(String.valueOf(fl1)) ;
                         f2.setBackground(Color.black) ;
                         f2.setForeground(Color.green) ;
                         break ;
                     }
                     case 2: {   //  quarter birch
                         wtlngs = .0216 ;
                         wdisplay = wtlngs * wconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f2.setText(String.valueOf(fl1)) ;
                         f2.setBackground(Color.black) ;
                         f2.setForeground(Color.green) ;
                         break ;
                     }
                     case 3: {   // plastic tube
                         wtlngs = .02096 ;
                         wdisplay = wtlngs * wconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f2.setText(String.valueOf(fl1)) ;
                         f2.setBackground(Color.black) ;
                         f2.setForeground(Color.green) ;
                         break ;
                     }
                     case 4: {   // skewer stick
                         wtlngs = .00325 ;
                         wdisplay = wtlngs * wconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f2.setText(String.valueOf(fl1)) ;
                         f2.setBackground(Color.black) ;
                         f2.setForeground(Color.green) ;
                         break ;
                     }
                     case 5: {   // specify
                         V2 = Double.valueOf(f2.getText()) ;
                         wdisplay = V2.doubleValue() ;
                         wtlngs = wdisplay * lconv / wconv ;
                         f2.setBackground(Color.white) ;
                         f2.setForeground(Color.black) ;
                         break ;
                      }
                 }

                 switch (lintyp) { // line weight per foot
                     case 0: {    // nylon
                         wtlngl = .004 ;
                         wdisplay = wtlngl * wconv / hconv ;
                         fl1 = (float) wdisplay ;
                         f3.setText(String.valueOf(fl1)) ;
                         f3.setBackground(Color.black) ;
                         f3.setForeground(Color.green) ;
                         break;
                     }
                     case 1: {   // cotton
                         wtlngl = .002 ;
                         wdisplay = wtlngl * wconv / hconv ;
                         fl1 = (float) wdisplay ;
                         f3.setText(String.valueOf(fl1)) ;
                         f3.setBackground(Color.black) ;
                         f3.setForeground(Color.green) ;
                         break ;
                     }
                     case 2: {   // specify
                         V3 = Double.valueOf(f3.getText()) ;
                         wdisplay = V3.doubleValue() ;
                         wtlngl = wdisplay * hconv / wconv ;
                         f3.setBackground(Color.white) ;
                         f3.setForeground(Color.black) ;
                         break ;
                     }
                 }

                 switch (taltyp) { // tail weight per length
                     case 0: {    // 1 inch plastic
                         wttail = .0004752 ;
                         wdisplay = wttail * wconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f4.setText(String.valueOf(fl1)) ;
                         f4.setBackground(Color.black) ;
                         f4.setForeground(Color.green) ;
                         break;
                     }
                     case 1: {    // 3 inch plastic
                         wttail = .0014256 ;
                         wdisplay = wttail * wconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f4.setText(String.valueOf(fl1)) ;
                         f4.setBackground(Color.black) ;
                         f4.setForeground(Color.green) ;
                         break;
                     }
                     case 2: {   // 1 inch nylon
                         wttail = .000522 ;
                         wdisplay = wttail * wconv / lconv ;
                         fl1 = (float) wdisplay ;
                         f4.setText(String.valueOf(fl1)) ;
                         f4.setBackground(Color.black) ;
                         f4.setForeground(Color.green) ;
                         break ;
                     }
                     case 3: {   // specify
                         V4 = Double.valueOf(f4.getText()) ;
                         wdisplay = V4.doubleValue() ;
                         wttail = wdisplay * lconv / wconv ;
                         f4.setBackground(Color.white) ;
                         f4.setForeground(Color.black) ;
                         break ;
                     }
                 }

                 compute();
              } // TextHandler

              public void handleDispla(Object arg) {
                String label = (String)arg ;

                if (label.equals("Shape")) {
                   layin.show(i, "first")  ;
                }

                if (label.equals("Material")) {
                   layin.show(i, "second")  ;
                }
              } // Display handler
           }  // Inmate

           class Inflight extends Panel {
              Kite outerparent ;
              L l ;
              R r ;
   
              Inflight (Kite target) {
                 outerparent = target ;
                 setLayout(new GridLayout(1,2,5,5)) ;
   
                 l = new L(outerparent) ;
                 r = new R(outerparent) ;
   
                 add(l) ;
                 add(r) ;
              }
   
              class L extends Panel {
                 Kite outerparent ;
                 TextField f1,f2,f3 ;
                 Label l1,l2,l3 ;
   
                 L (Kite target) {
                  outerparent = target ;
                  setLayout(new GridLayout(5,2,2,10)) ;
 
                  l1 = new Label("Wind-ft/sec", Label.CENTER) ;
                  f1 = new TextField("20.0",5) ;
   
                  l3 = new Label("Line - ft", Label.CENTER) ;
                  f3 = new TextField("100.0",5) ;
   
                  add(new Label(" ", Label.CENTER)) ;
                  add(new Label(" ", Label.CENTER)) ;

                  add(new Label("Input:", Label.CENTER)) ;
                  add(new Label(" ", Label.CENTER)) ;

                  add(l1) ;
                  add(f1) ;
   
                  add(l3) ;
                  add(f3) ;
   
                  add(new Label(" ", Label.CENTER)) ;
                  add(new Label(" ", Label.CENTER)) ;
                }
    
                public boolean action(Event evt, Object arg) {
                    if(evt.id == Event.ACTION_EVENT) {
                       this.handleText(evt) ;
                       return true ;
                    }
                    else return false ;
                }

                public void handleText(Event evt) {
                  Double V1,V2,V3,V4,V5 ;
                  double v1,v2,v3,v4,v5 ;
                  float fl1 ;
                  int i1,i2,i3,i4,i5 ;
   
                  V1 = Double.valueOf(f1.getText()) ;
                  v1 = V1.doubleValue() ;
                  V3 = Double.valueOf(f3.getText()) ;
                  v3 = V3.doubleValue() ;
   
                  windd = v1 ;
                  if(windd < wndmnd) {
                    windd = v1 = wndmnd ;
                    fl1 = (float) v1 ;
                    f1.setText(String.valueOf(fl1)) ;
                  }
                  if(windd > wndmxd) {
                    windd = v1 = wndmxd ;
                    fl1 = (float) v1 ;
                    f1.setText(String.valueOf(fl1)) ;
                  }
                  wind = windd / hconv ;
   
                  llined = v3 ;
                  if(llined < llnmnd) {
                    llined = v3 = llnmnd ;
                    fl1 = (float) v3 ;
                    f3.setText(String.valueOf(fl1)) ;
                  }
                  if(llined > llnmxd) {
                    llined = v3 = llnmxd ;
                    fl1 = (float) v3 ;
                    f3.setText(String.valueOf(fl1)) ;
                  }
                  lline = llined / hconv ;
   
                  i1 = (int) (((v1 - wndmnd)/(wndmxd-wndmnd))*1000.) ;
                  i3 = (int) (((v3 - llnmnd)/(llnmxd-llnmnd))*1000.) ;
   
                  r.s1.setValue(i1) ;
                  r.s3.setValue(i3) ;

                  compute();
                } // TextHandler
              }  // Inleft
 
              class R extends Panel {
                 Kite outerparent ;
                 Scrollbar s1,s3;
   
                 R (Kite target) {
                  int i1,i3 ;
   
                  outerparent = target ;
                  setLayout(new GridLayout(5,1,2,10)) ;
   
                  i1 = (int) (((20.0 - wndmn)/(wndmx-wndmn))*1000.) ;
                  i3 = (int) (((100.0 - llnmn)/(llnmx-llnmn))*1000.) ;
   
                  s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                  s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
   
                  add(new Label(" ", Label.CENTER)) ;
                  add(new Label(" ", Label.CENTER)) ;
                  add(s1) ;
                  add(s3) ;
                  add(new Label(" ", Label.CENTER)) ;
                }
      
                public boolean handleEvent(Event evt) {
                     if(evt.id == Event.SCROLL_ABSOLUTE) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_LINE_DOWN) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_LINE_UP) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_PAGE_DOWN) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_PAGE_UP) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     else return false ;
                }

                public void handleBar(Event evt) {
                   int i1,i2,i3 ;
                   double v1,v2,v3 ;
                   float fl1,fl2,fl3 ;
            // Input    for computations
                   i1 = s1.getValue() ;
                   i3 = s3.getValue() ;
   
                   windd = v1 = i1 * (wndmxd - wndmnd)/ 1000. + wndmnd ;
                   wind = windd / hconv ;
                   llined = v3 = i3 * (llnmxd - llnmnd)/ 1000. + llnmnd ;
                   lline = llined / hconv;
 
                   fl1 = (float) v1 ;
                   fl3 = (float) v3 ;
   
                   l.f1.setText(String.valueOf(fl1)) ;
                   l.f3.setText(String.valueOf(fl3)) ;
                   
                   compute() ;
                } // handle bar
              }  // Inright
           } // Inflight

           class Intrim extends Panel {
              Kite outerparent ;
              L l ;
              R r ;
   
              Intrim (Kite target) {
                 outerparent = target ;
                 setLayout(new GridLayout(1,2,5,5)) ;
   
                 l = new L(outerparent) ;
                 r = new R(outerparent) ;
   
                 add(l) ;
                 add(r) ;
              }
   
              class L extends Panel {
                 Kite outerparent ;
                 TextField f1,f2,f3,f4 ;
                 Label l1,l2,l3,l4 ;
   
                 L (Kite target) {
                  outerparent = target ;
                  setLayout(new GridLayout(5,2,2,10)) ;
 
                  l1 = new Label("B-Bridle-in", Label.CENTER) ;
                  f1 = new TextField("17.0",5) ;
   
                  l2 = new Label("K-Knot-in", Label.CENTER) ;
                  f2 = new TextField("12.5",5) ;
   
                  l3 = new Label("Angle", Label.CENTER) ;
                  f3 = new TextField("5.0",5) ;
   
                  l4 = new Label("T-Tail-in", Label.CENTER) ;
                  f4 = new TextField("6.0",5) ;
   
                  add(new Label(" ", Label.CENTER)) ;
                  add(new Label(" ", Label.CENTER)) ;

                  add(l3) ;
                  add(f3) ;

                  add(l1) ;
                  add(f1) ;
   
                  add(l2) ;
                  add(f2) ;

                  add(l4) ;
                  add(f4) ;

                }
    
                public boolean action(Event evt, Object arg) {
                    if(evt.id == Event.ACTION_EVENT) {
                       this.handleText(evt) ;
                       return true ;
                    }
                    else return false ;
                }

                public void handleText(Event evt) {
                  Double V1,V2,V3,V4 ;
                  double v1,v2,v3,v4 ;
                  float fl1 ;
                  int i1,i2,i3,i4;
   
                  V1 = Double.valueOf(f1.getText()) ;
                  v1 = V1.doubleValue() ;
                  V2 = Double.valueOf(f2.getText()) ;
                  v2 = V2.doubleValue() ;
                  V4 = Double.valueOf(f4.getText()) ;
                  v4 = V4.doubleValue() ;
   
                  if (ktype < 5) {
                    lbridd = v1 ;
                    if (lbridd < lkite*lconv) {
                      lbridd = v1 = lkite*lconv + .01 ;
                      fl1 = (float) v1 ;
                      f1.setText(String.valueOf(fl1)) ;
                    }
                    if(lbridd < lbrmnd) {
                      lbridd = v1 = lbrmnd ;
                      fl1 = (float) v1 ;
                      f1.setText(String.valueOf(fl1)) ;
                    }
                    if(lbridd > lbrmxd) {
                      lbridd = v1 = lbrmxd ;
                      fl1 = (float) v1 ;
                      f1.setText(String.valueOf(fl1)) ;
                    }
                    lbrid = lbridd / lconv ;
                  }
 
                  lknotd = v2 ;
                  if (ktype < 5) {
                     if(lknotd > lbridd) {
                       lknotd = v2 = lbridd ;
                       fl1 = (float) v2 ;
                       f2.setText(String.valueOf(fl1)) ;
                     }
                  }
                  else {
                     if(lknotd > lkite*lconv) {
                       lknotd = v2 = lkite*lconv - .1 ;
                       fl1 = (float) v2 ;
                       f2.setText(String.valueOf(fl1)) ;
                     }
                  }
                  if(lknotd < lbrmnd) {
                    lknotd = v2 = lbrmnd ;
                    fl1 = (float) v2 ;
                    f2.setText(String.valueOf(fl1)) ;
                  }
                  if(lknotd > lbrmxd) {
                    lknotd = v2 = lbrmxd ;
                    fl1 = (float) v2 ;
                    f2.setText(String.valueOf(fl1)) ;
                  }
                  lknot = lknotd / lconv ;
 
                  ltaild = v4 ;
                  if(ltaild < ltlmnd) {
                    ltaild = v4 = ltlmnd ;
                    fl1 = (float) v4 ;
                    f4.setText(String.valueOf(fl1)) ;
                  }
                  if(ltaild > ltlmxd) {
                    ltaild = v4 = ltlmxd ;
                    fl1 = (float) v4 ;
                    f4.setText(String.valueOf(fl1)) ;
                  }
                  ltail = ltaild / lconv ;
   
                  i1 = (int) (((v1 - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
                  i2 = (int) (((v2 - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
                  i4 = (int) (((v4 - ltlmnd)/(ltlmxd-ltlmnd))*1000.) ;
   
                  r.s1.setValue(i1) ;
                  r.s2.setValue(i2) ;
                  r.s4.setValue(i4) ;

                  if (angmod > 0) {
                    V3 = Double.valueOf(f3.getText()) ;
                    v3 = V3.doubleValue() ;
                    alpha = v3 ;
                    if(alpha < almn) {
                      alpha = v3 = almn ;
                      fl1 = (float) v3 ;
                      f3.setText(String.valueOf(fl1)) ;
                    }
                    if(alpha > almx) {
                      alpha = v3 = almx ;
                      fl1 = (float) v3 ;
                      f3.setText(String.valueOf(fl1)) ;
                    }
     
                    i3 = (int) (((v3 - almn)/(almx-almn))*1000.) ;
                    r.s3.setValue(i3) ;
                  }
   
                  compute();
                } // TextHandler
              }  // Inleft
 
              class R extends Panel {
                 Kite outerparent ;
                 Scrollbar s1,s2,s3,s4;
                 Choice amode;
   
                 R (Kite target) {
                  int i1,i2,i3,i4;
   
                  outerparent = target ;
                  setLayout(new GridLayout(5,1,2,10)) ;
   
                  i1 = (int) (((20.0 - lbrmn)/(lbrmx-lbrmn))*1000.) ;
                  i2 = (int) (((10.0 - lbrmn)/(lbrmx-lbrmn))*1000.) ;
                  i3 = (int) (((0.0 - almn)/(almx-almn))*1000.) ;
                  i4 = (int) (((6.0 - ltlmn)/(ltlmx-ltlmn))*1000.) ;

                  amode = new Choice() ;
                  amode.addItem("Compute Trim Angle") ;
                  amode.addItem("Set Trim Angle ");
                  amode.setBackground(Color.white) ;
                  amode.setForeground(Color.red) ;
                  amode.select(0) ;

                  s1 = new Scrollbar(Scrollbar.HORIZONTAL,i1,10,0,1000);
                  s2 = new Scrollbar(Scrollbar.HORIZONTAL,i2,10,0,1000);
                  s3 = new Scrollbar(Scrollbar.HORIZONTAL,i3,10,0,1000);
                  s4 = new Scrollbar(Scrollbar.HORIZONTAL,i4,10,0,1000);
   
                  add(amode) ;
                  add(s3) ;
                  add(s1) ;
                  add(s2) ;
                  add(s4) ;
                }
      
                public boolean handleEvent(Event evt) {
                     if(evt.id == Event.ACTION_EVENT) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_ABSOLUTE) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_LINE_DOWN) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_LINE_UP) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_PAGE_DOWN) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     if(evt.id == Event.SCROLL_PAGE_UP) {
                        this.handleBar(evt) ;
                        return true ;
                     }
                     else return false ;
                }

                public void handleBar(Event evt) {
                   int i1,i2,i3,i4 ;
                   double v1,v2,v3,v4 ;
                   float fl1,fl2,fl3,fl4 ;
            // Input    for computations
                   i1 = s1.getValue() ;
                   i2 = s2.getValue() ;
                   i4 = s4.getValue() ;
                   angmod = amode.getSelectedIndex() ;

                   lbridd = v1 = i1 * (lbrmxd - lbrmnd)/ 1000. + lbrmnd ;
                   lbrid = lbridd / lconv ;
                   lknotd = v2 = i2 * (lbrmxd - lbrmnd)/ 1000. + lbrmnd ;
                   lknot = lknotd / lconv ;
                   ltaild = v4 = i4 * (ltlmxd - ltlmnd)/ 1000. + ltlmnd ;
                   ltail = ltaild / lconv ;
 
                   if (ktype < 5) {
                      if (lbridd < lkite*lconv) {
                        lbridd = v1 = lkite*lconv + .01 ;
                        i1 = (int) (((lbridd - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
                        s1.setValue(i1) ;
                      }
                      if (lknotd > lbridd) {
                        lknotd = v2 = lbridd ;
                        i2 = (int) (((lknotd - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
                        s2.setValue(i2) ;
                      }
                   }
                   else {
                      if (lknotd > lkite*lconv) {
                        lknotd = v2 = lkite*lconv - .1 ;
                        i2 = (int) (((lknotd - lbrmnd)/(lbrmxd-lbrmnd))*1000.) ;
                        s2.setValue(i2) ;
                      }
                   }
   
                   fl1 = (float) v1 ;
                   fl2 = (float) v2 ;
                   fl4 = (float) v4 ;
   
                   l.f1.setText(String.valueOf(fl1)) ;
                   l.f2.setText(String.valueOf(fl2)) ;
                   l.f4.setText(String.valueOf(fl4)) ;
                   
                   if (angmod > 0) {
                      i3 = s3.getValue() ;
                      alpha = v3 = i3 * (almx - almn)/ 1000. + almn ;
                      fl3 = (float) v3 ;
                      l.f3.setText(String.valueOf(fl3)) ;
                   }

                   compute() ;
                } // handle bar
              }  // Inright
           } // Intrim
        }  // Inppnl 

        class O extends Panel {
           Kite outerparent ;
           Up up ;
           Dn dn ;
   
           O (Kite target) { 
             outerparent = target ;
             setLayout(new GridLayout(2,1,2,2)) ;
    
             up = new Up(outerparent) ;
             dn = new Dn(outerparent) ;
   
             add(up) ;
             add(dn) ;
           }

           class Up extends Panel {
              Kite outerparent ;
              Label lo3,lo7,lo8,lo9,lo10 ;
              TextField o3,o7,o8,o9,o10 ;
              Button bt3,bt4 ;
    
              Up (Kite target) { 
                outerparent = target ;
                setLayout(new GridLayout(3,4,2,2)) ;
    
                bt3 = new Button("Design") ;
                bt3.setBackground(Color.blue) ;
                bt3.setForeground(Color.white) ;
   
                bt4 = new Button("Trim") ;
                bt4.setBackground(Color.white) ;
                bt4.setForeground(Color.blue) ;
   
                lo3 = new Label("Weight -oz", Label.CENTER) ;
                o3 = new TextField("25.0",5) ;
                o3.setBackground(Color.black) ;
                o3.setForeground(Color.green) ;

                lo7 = new Label("Lift -oz", Label.CENTER) ;
                o7 = new TextField("25.0",5) ;
                o7.setBackground(Color.black) ;
                o7.setForeground(Color.green) ;
   
                lo8 = new Label("Drag -oz", Label.CENTER) ;
                o8 = new TextField("25.0",5) ;
                o8.setBackground(Color.black) ;
                o8.setForeground(Color.green) ;

                lo9 = new Label("Torque", Label.CENTER) ;
                o9 = new TextField("25.0",5) ;
                o9.setBackground(Color.black) ;
                o9.setForeground(Color.green) ;

                lo10 = new Label("Angle", Label.CENTER) ;
                o10 = new TextField("5.0",5) ;
                o10.setBackground(Color.black) ;
                o10.setForeground(Color.green) ;

                add(bt3) ;
                add(bt4) ;
                add(lo3) ;
                add(o3) ;

                add(lo7) ;
                add(o7) ;
                add(lo8) ;
                add(o8) ;

                add(lo10) ;
                add(o10) ;
                add(lo9) ;
                add(o9) ;
             }

             public boolean action(Event evt, Object arg) {
               String label = (String)arg ;
   
               if(evt.target instanceof Button) {
                  if (label.equals("Design")) {
                     bt3.setBackground(Color.blue) ;
                     bt3.setForeground(Color.white) ;
                     bt4.setBackground(Color.white) ;
                     bt4.setForeground(Color.blue) ;
                     dn.oflt.bt5.setBackground(Color.white) ;
                     dn.oflt.bt5.setForeground(Color.blue) ;
                     layin.show(i, "first")  ;
                     layout.show(dn, "fsto")  ;
                     if (pick == -1) viewflg = 0 ;
                  }
                  if (label.equals("Trim")) {
                     bt3.setBackground(Color.white) ;
                     bt3.setForeground(Color.blue) ;
                     bt4.setBackground(Color.blue) ;
                     bt4.setForeground(Color.white) ;
                     dn.oflt.bt5.setBackground(Color.white) ;
                     dn.oflt.bt5.setForeground(Color.blue) ;
                     layin.show(i, "fourth")  ;
                     layout.show(dn, "seco")  ;
                     if (pick == -1) viewflg = 1 ;
                  }
                  loadInput() ;
                  compute() ;
                  return true ;
               }
               else return false ;
             } // Handler
           }  // end Up

           class Dn extends Panel {
              Kite outerparent ;
              Odes odes;
              Oflt oflt;
    
              Dn (Kite target) { 
                outerparent = target ;
                layout = new CardLayout() ;
                setLayout(layout) ;
    
                odes = new Odes(outerparent) ;
                oflt = new Oflt(outerparent) ;

                add ("fsto", odes) ;
                add ("seco", oflt) ;
              }

              class Odes extends Panel {
                 Kite outerparent ;
                 Label lo1,lo2,lo4,lo5,lo6,lo16 ;
                 TextField o1,o2,o4,o5,o6,o16 ;

                 Odes (Kite target) {
                   outerparent = target ;
                   setLayout(new GridLayout(3,4,2,2)) ;
    
                   lo1 = new Label("Surface-sqin", Label.CENTER) ;
                   o1 = new TextField("75.0",5) ;
                   o1.setBackground(Color.black) ;
                   o1.setForeground(Color.green) ;

                   lo2 = new Label("Frame-in", Label.CENTER) ;
                   o2 = new TextField("25.0",5) ;
                   o2.setBackground(Color.black) ;
                   o2.setForeground(Color.green) ;

                   lo4 = new Label("Cg -in", Label.CENTER) ;
                   o4 = new TextField("25.0",5) ;
                   o4.setBackground(Color.black) ;
                   o4.setForeground(Color.green) ;
   
                   lo5 = new Label("Cp -in", Label.CENTER) ;
                   o5 = new TextField("25.0",5) ;
                   o5.setBackground(Color.black) ;
                   o5.setForeground(Color.green) ;
      
                   lo6 = new Label("Aspect Ratio", Label.CENTER) ;
                   o6 = new TextField("25.0",5) ;
                   o6.setBackground(Color.black) ;
                   o6.setForeground(Color.green) ;

                   lo16 = new Label("Y-Height-ft", Label.CENTER) ;
                   o16 = new TextField("75.0",5) ;
                   o16.setBackground(Color.black) ;
                   o16.setForeground(Color.green) ;

                   add(lo4) ;
                   add(o4) ;
                   add(lo16) ;
                   add(o16) ;

                   add(lo5) ;
                   add(o5) ;
                   add(lo6) ;
                   add(o6) ;

                   add(lo1) ;
                   add(o1) ;
                   add(lo2) ;
                   add(o2) ;
                }
              }  //  end Odes

              class Oflt extends Panel {
                 Kite outerparent ;
                 Label lo12,lo13,lo14,lo15,lo17 ;
                 TextField o12,o13,o14,o15,o17 ;
                 Button bt5 ;

                 Oflt (Kite target) {
                   outerparent = target ;
                   setLayout(new GridLayout(3,4,2,2)) ;
    
                   bt5 = new Button("Fly") ;
                   bt5.setBackground(Color.white) ;
                   bt5.setForeground(Color.blue) ;
   
                   lo12 = new Label("X-Far-ft", Label.CENTER) ;
                   o12 = new TextField("75.0",5) ;
                   o12.setBackground(Color.black) ;
                   o12.setForeground(Color.green) ;

                   lo13 = new Label("Y-Height-ft", Label.CENTER) ;
                   o13 = new TextField("75.0",5) ;
                   o13.setBackground(Color.black) ;
                   o13.setForeground(Color.green) ;

                   lo14 = new Label("Wind ft/sec", Label.CENTER) ;
                   o14 = new TextField("10.0",5) ;
                   o14.setBackground(Color.black) ;
                   o14.setForeground(Color.green) ;

                   lo15 = new Label("Cg -in", Label.CENTER) ;
                   o15 = new TextField("25.0",5) ;
                   o15.setBackground(Color.black) ;
                   o15.setForeground(Color.green) ;
   
                   lo17 = new Label("Tension-oz", Label.CENTER) ;
                   o17 = new TextField("25.0",5) ;
                   o17.setBackground(Color.black) ;
                   o17.setForeground(Color.green) ;
   
                   add(lo15) ;
                   add(o15) ;
                   add(lo13) ;
                   add(o13) ;

                   add(lo14) ;
                   add(o14) ;
                   add(lo12) ;
                   add(o12) ;

                   add(bt5) ;
                   add(new Label(" ", Label.CENTER)) ;
                   add(lo17) ;
                   add(o17) ;
                }

                public boolean action(Event evt, Object arg) {
                  String label = (String)arg ;
   
                  if(evt.target instanceof Button) {
                     if (label.equals("Fly")) {
                        layin.show(i, "third")  ;
                        bt5.setBackground(Color.blue) ;
                        bt5.setForeground(Color.white) ;
                        up.bt4.setBackground(Color.white) ;
                        up.bt4.setForeground(Color.blue) ;
                        if (pick == -1) viewflg = 2 ;
                     }
                     loadInput() ;
                     compute() ;
                     return true ;
                  }
                  else return false ;
                } // Handler
              }  //  end Oflt
           }  //  end Dn
        }  // Outpnl 
     } // Conppnl

     class Viewer extends Canvas  
            implements Runnable{
        Kite outerparent ;
        Thread runner ;
        Point locate,anchor;
   
        Viewer (Kite target) {
            setBackground(Color.blue) ;
            runner = null ;
        } 

        public Insets insets() {
           return new Insets(0,10,0,10) ;
        }
 
        public boolean mouseDown(Event evt, int x, int y) {
           anchor = new Point(x,y) ;
           return true;
        }

        public boolean mouseUp(Event evt, int x, int y) {
           handleb(x,y) ;
           return true;
        }

        public boolean mouseDrag(Event evt, int x, int y) {
           handle(x,y) ;
           return true;
        }

        public void handle(int x, int y) {  // slider widgets
         // determine location
           if (y >= 45 && y <= 275) {
             if (x <= 21 ) {   // zoom widget
               sldloc = y ;
               if (sldloc < 55) sldloc = 55;
               if (sldloc > 275) sldloc = 275;
               fact = 20.0 * 10.0/(sldloc-50);
             }
           }
           if (viewflg == 1) {  // force scale
             if (x >= 95 && x <= 225) {
               if (y >= 330 ) {   // scale widget
                 vldloc = x ;
                 if (vldloc < 95) vldloc = 95;
                 if (vldloc > 225) vldloc = 225;
                 vscale = 1.0 + (vldloc-95)*.3 ;
               }
             }
           }
           if (viewflg == 2) {  // field scale
             if (x >= 95 && x <= 225) {
               if (y >= 330 ) {   // scale widget
                 fldloc = x ;
                 if (fldloc < 95) fldloc = 95;
                 if (fldloc > 225) fldloc = 225;
                 fscale = .04 + (fldloc-95)*.007 ;
               }
             }
           }
        }

        public void handleb(int x, int y) { // view buttons
          float fl3 ;
          int i3 ;

          if (y <= 15) {
             if (x >= 5 && x <= 60) {   // select units
                  units = -units ;
                  setUnits() ;
             }
             if (x >= 65 && x <= 140) {   // select view
                  pick = -pick ;
             }
             if (x >= 150 && x <= 199) {   // front view
                  if (pick == 1) viewflg = 0 ;
             }
             if (x >= 200 && x <= 240) {   // side view
                  if (pick == 1) viewflg = 1 ;
             }
             if (x >= 241 && x <= 289 ) {   // field view
                  if (pick == 1) viewflg = 2 ;
             }
          }
          if (y >= 330 && y <= 347) {
             if (x >= 2 && x <= 47) {  // save button
                printData() ;
             } 
             if (x >= 240 && x <= 285) {
                if (ktype <= 4) {  // Reset
                  w1 = 10.0 ;
                  w1d = w1*lconv ;

                  fl3 = (float) w1d ;
                  i3 = (int) (((w1d - wmnd)/(wmxd-wmnd))*1000.) ;

                  s.c.i.inshape.l.f3.setText(String.valueOf(fl3)) ;
                  s.c.i.inshape.r.s3.setValue(i3) ;
                  s.c.i.inshape.r.handleCho() ;
                }
                if (ktype == 5 && viewflg == 0) {
                  fold = - fold ;
                }
             } 
          }

          s.view.repaint() ;
        }

        public void setUnits() { // units switch
    // change labels
           if (units == -1) {  // metric units
              lconv = 2.54 ;
              wconv = 28.35 ;
              hconv = .3048 ;
              s.c.o.up.lo3.setText("Weight -gm") ;
              s.c.o.up.lo7.setText("Lift -gm") ;
              s.c.o.up.lo8.setText("Drag -gm") ;
              s.c.o.dn.odes.lo4.setText("Cg -cm") ;
              s.c.o.dn.odes.lo5.setText("Cp -cm") ;
              s.c.o.dn.odes.lo1.setText("Surface-cm2") ;
              s.c.o.dn.odes.lo16.setText("Y-Height-m") ;
              s.c.o.dn.odes.lo2.setText("Frame-cm") ;
              s.c.o.dn.oflt.lo14.setText("Wind m/sec") ;
              s.c.o.dn.oflt.lo15.setText("Cg -cm") ;
              s.c.o.dn.oflt.lo17.setText("Tension-gm") ;
              s.c.o.dn.oflt.lo13.setText("Y-Height-m") ;
              s.c.o.dn.oflt.lo12.setText("X-Far-m") ;
              s.c.i.inshape.l.l1.setText("H1 - cm") ;
              s.c.i.inshape.l.l2.setText("H2 - cm") ;
              s.c.i.inshape.l.l3.setText("W1 - cm") ;
              s.c.i.inshape.l.l4.setText("W2 - cm") ;
              s.c.i.intrim.l.l1.setText("B-Bridle-cm") ;
              s.c.i.intrim.l.l2.setText("K-Knot-cm") ;
              s.c.i.intrim.l.l4.setText("T-Tail-cm") ;
              s.c.i.inflight.l.l3.setText("Line - m") ;
              s.c.i.inflight.l.l1.setText("Wind-m/sec") ;
              s.c.i.inmate.l1.setText("Surface-gm/cm2") ;
              s.c.i.inmate.l2.setText("Frame-gm/cm") ;
              s.c.i.inmate.l4.setText("Tail-gm/cm") ;
              s.c.i.inmate.l3.setText("Line-gm/m") ;
           }
           if (units == 1) {  // english units
              lconv = 1.0 ;
              wconv = 1.0 ;
              hconv = 1.0 ;
              s.c.o.up.lo3.setText("Weight -oz") ;
              s.c.o.up.lo7.setText("Lift -oz") ;
              s.c.o.up.lo8.setText("Drag -oz") ;
              s.c.o.dn.odes.lo4.setText("Cg -in") ;
              s.c.o.dn.odes.lo5.setText("Cp -in") ;
              s.c.o.dn.odes.lo1.setText("Surface-in2") ;
              s.c.o.dn.odes.lo16.setText("Y-Height-ft") ;
              s.c.o.dn.odes.lo2.setText("Frame-in") ;
              s.c.o.dn.oflt.lo14.setText("Wind ft/sec") ;
              s.c.o.dn.oflt.lo15.setText("Cg -in") ;
              s.c.o.dn.oflt.lo17.setText("Tension-oz") ;
              s.c.o.dn.oflt.lo13.setText("Y-Height-ft") ;
              s.c.o.dn.oflt.lo12.setText("X-Far-ft") ;
              s.c.i.inshape.l.l1.setText("H1 - in") ;
              s.c.i.inshape.l.l2.setText("H2 - in") ;
              s.c.i.inshape.l.l3.setText("W1 - in") ;
              s.c.i.inshape.l.l4.setText("W2 - in") ;
              s.c.i.intrim.l.l1.setText("B-Bridle-in") ;
              s.c.i.intrim.l.l2.setText("K-Knot-in") ;
              s.c.i.intrim.l.l4.setText("T-Tail-in") ;
              s.c.i.inflight.l.l3.setText("Line - ft") ;
              s.c.i.inflight.l.l1.setText("Wind-ft/sec") ;
              s.c.i.inmate.l1.setText("Surface-oz/in2") ;
              s.c.i.inmate.l2.setText("Frame-oz/in") ;
              s.c.i.inmate.l4.setText("Tail-oz/in") ;
              s.c.i.inmate.l3.setText("Line-oz/ft") ;
           }
           loadInput() ;
           compute() ;
        }

        public void start() {
           if (runner == null) {
              runner = new Thread(this) ;
              runner.start() ;
           }
        }

        public void run() {
          int timer ;
    
          timer = 100 ;
          while (true) {
             try { Thread.sleep(timer); }
             catch (InterruptedException e) {}
             s.view.repaint() ;
          }
        }

        public void update(Graphics g) {
           s.view.paint(g) ;
        }
 
        public void paint(Graphics g) {
           int i,j,k,n ;
           int exes[] = new int[9] ;
           int whys[] = new int[9] ;
           int ylabel,ylabel2,xlabel,xlabel2;
           double ripple;
           Color col ;

           col = new Color(0,0,1) ;
           off1Gg.setColor(Color.blue) ;
           off1Gg.fillRect(0,0,300,500) ;
 // FRONT VIEW
           if (viewflg == 0) {
              xt = 150; yt =225 ;
              if (ktype == 0) {  // diamond
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(0.0)) + xt ;
                 whys[0] = (int) (fact*(-h1 - h2)) + yt ;
                 exes[1] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h2)) + yt ;
                 exes[2] = (int) (fact*(0.0)) + xt ;
                 whys[2] = (int) (fact*(0.0)) + yt ;
                 exes[3] = (int) (fact*(w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(-h2)) + yt ;
                 exes[4] = (int) (fact*(0.0)) + xt ;
                 whys[4] = (int) (fact*(-h1 - h2)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[0],whys[0],exes[2],whys[2]) ;
                 off1Gg.drawLine(exes[1],whys[1],exes[3],whys[3]) ;
               //Labels
                 off1Gg.setColor(Color.yellow) ;
                 off1Gg.drawString("W1",xt-5,337) ;
                 off1Gg.drawLine(exes[1],327,exes[1],337) ;
                 off1Gg.drawLine(exes[1],332,xt-20,332) ;
                 off1Gg.drawLine(exes[3],332,xt+20,332) ;
                 off1Gg.drawLine(exes[3],327,exes[3],337) ;
                 off1Gg.setColor(Color.blue) ;
                 off1Gg.fillRect(270,0,30,300) ;
                 off1Gg.setColor(Color.yellow) ;
                 ylabel = (int) (fact*(-h2/2.0)) + yt ;
                 off1Gg.drawString("H2",270,ylabel) ;
                 off1Gg.drawLine(270,yt,280,yt) ;
                 off1Gg.drawLine(275,yt,275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[1]) ;
                 off1Gg.drawLine(270,whys[1],280,whys[1]) ;
                 ylabel = (int) (fact*(-h1/2.0 -h2)) + yt ;
                 off1Gg.drawString("H1",270,ylabel) ;
                 off1Gg.drawLine(275,whys[1],275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[0]) ;
                 off1Gg.drawLine(270,whys[0],280,whys[0]) ;
              }
              if (ktype == 1) {  // delta
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(0.0)) + xt ;
                 whys[0] = (int) (fact*(-h1 - h2)) + yt ;
                 exes[1] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h2)) + yt ;
                 exes[2] = (int) (fact*(0.0)) + xt ;
                 whys[2] = (int) (fact*(0.0)) + yt ;
                 exes[3] = (int) (fact*(w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(-h2)) + yt ;
                 exes[4] = (int) (fact*(0.0)) + xt ;
                 whys[4] = (int) (fact*(-h1 - h2)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[0],whys[0]+2,exes[1]+2,whys[1]) ;
                 off1Gg.drawLine(exes[0],whys[0]+2,exes[3]-2,whys[3]) ;
               //Labels
                 off1Gg.setColor(Color.yellow) ;
                 off1Gg.drawString("W1",xt-5,337) ;
                 off1Gg.drawLine(exes[1],327,exes[1],337) ;
                 off1Gg.drawLine(exes[1],332,xt-20,332) ;
                 off1Gg.drawLine(exes[3],332,xt+20,332) ;
                 off1Gg.drawLine(exes[3],327,exes[3],337) ;
                 off1Gg.setColor(Color.blue) ;
                 off1Gg.fillRect(270,0,30,300) ;
                 off1Gg.setColor(Color.yellow) ;
                 ylabel = (int) (fact*(-h2/2.0)) + yt ;
                 off1Gg.drawString("H2",270,ylabel) ;
                 off1Gg.drawLine(270,yt,280,yt) ;
                 off1Gg.drawLine(275,yt,275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[1]) ;
                 off1Gg.drawLine(270,whys[1],280,whys[1]) ;
                 ylabel = (int) (fact*(-h1/2.0 -h2)) + yt ;
                 off1Gg.drawString("H1",270,ylabel) ;
                 off1Gg.drawLine(275,whys[1],275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[0]) ;
                 off1Gg.drawLine(270,whys[0],280,whys[0]) ;
              }
              if (ktype == 2) {  // sled
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[0] = (int) (fact*(-h1 - h2)) + yt ;
                 exes[1] = (int) (fact*(-w2/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h2)) + yt ;
                 exes[2] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[2] = (int) (fact*(0.0)) + yt ;
                 exes[3] = (int) (fact*(w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(0.0)) + yt ;
                 exes[4] = (int) (fact*(w2/2.0)) + xt ;
                 whys[4] = (int) (fact*(-h2)) + yt ;
                 exes[5] = (int) (fact*(w1/2.0)) + xt ;
                 whys[5] = (int) (fact*(-h1 - h2)) + yt ;
                 exes[6] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[6] = (int) (fact*(-h1 - h2)) + yt ;
                 off1Gg.fillPolygon(exes,whys,7) ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[0],whys[0],exes[2],whys[2]) ;
                 off1Gg.drawLine(exes[3],whys[3],exes[5],whys[5]) ;
               //Labels
                 off1Gg.setColor(Color.yellow) ;
                 off1Gg.drawString("W1",xt-5,337) ;
                 off1Gg.drawLine(exes[0],327,exes[0],337) ;
                 off1Gg.drawLine(exes[0],332,xt-20,332) ;
                 off1Gg.drawLine(exes[5],332,xt+20,332) ;
                 off1Gg.drawLine(exes[5],327,exes[5],337) ;

                 off1Gg.drawString("W2",xt-5,350) ;
                 off1Gg.drawLine(exes[1],340,exes[1],355) ;
                 off1Gg.drawLine(exes[1],345,xt-20,345) ;
                 off1Gg.drawLine(exes[4],345,xt+20,345) ;
                 off1Gg.drawLine(exes[4],340,exes[4],355) ;

                 off1Gg.setColor(Color.blue) ;
                 off1Gg.fillRect(270,0,30,300) ;
                 off1Gg.setColor(Color.yellow) ;
                 ylabel = (int) (fact*(-h2/2.0)) + yt ;
                 off1Gg.drawString("H2",270,ylabel) ;
                 off1Gg.drawLine(270,yt,280,yt) ;
                 off1Gg.drawLine(275,yt,275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[4]) ;
                 off1Gg.drawLine(270,whys[4],280,whys[4]) ;
                 ylabel = (int) (fact*(-h1/2.0 -h2)) + yt ;
                 off1Gg.drawString("H1",270,ylabel) ;
                 off1Gg.drawLine(275,whys[4],275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[0]) ;
                 off1Gg.drawLine(270,whys[0],280,whys[0]) ;
              }
              if (ktype == 3) {  // box
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(w1/2.0)) + xt ;
                 whys[0] = (int) (fact*(-h1)) + yt ;
                 exes[1] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h1)) + yt ;
                 exes[2] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[2] = (int) (fact*(0.0)) + yt ;
                 exes[3] = (int) (fact*(w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(0.0)) + yt ;
                 exes[4] = (int) (fact*(w1/2.0)) + xt ;
                 whys[4] = (int) (fact*(-h1)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;

                 exes[0] = (int) (fact*(w1/2.0)) + xt ;
                 whys[0] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 exes[1] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 exes[2] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[2] = (int) (fact*(-h1 -h2)) + yt ;
                 exes[3] = (int) (fact*(w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(-h1 -h2)) + yt ;
                 exes[4] = (int) (fact*(w1/2.0)) + xt ;
                 whys[4] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
   
               //sticks
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[1],yt,exes[1],whys[1]) ;
                 off1Gg.drawLine(xt,yt,xt,whys[1]) ;
                 off1Gg.drawLine(exes[4],yt,exes[4],whys[4]) ;
               //Labels
                 off1Gg.setColor(Color.yellow) ;
                 off1Gg.drawString("W1",xt-5,337) ;
                 off1Gg.drawLine(exes[1],327,exes[1],337) ;
                 off1Gg.drawLine(exes[1],332,xt-20,332) ;
                 off1Gg.drawLine(exes[3],332,xt+20,332) ;
                 off1Gg.drawLine(exes[3],327,exes[3],337) ;
              //   off1Gg.drawString("W2",xt-5,350) ;
                 ylabel = (int) (fact*(-h1/2.0)) + yt ;
                 ylabel2 = (int) (fact*(-h1)) + yt ;
                 off1Gg.setColor(Color.blue) ;
                 off1Gg.fillRect(270,0,30,300) ;
                 off1Gg.setColor(Color.yellow) ;
                 off1Gg.drawString("H1",270,ylabel) ;
                 off1Gg.drawLine(270,yt,280,yt) ;
                 off1Gg.drawLine(275,yt,275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,ylabel2) ;
                 off1Gg.drawLine(270,ylabel2,280,ylabel2) ;
                 ylabel = (int) (fact*(-h1 -h2/2.0)) + yt ;
                 off1Gg.drawString("H2",270,ylabel) ;
                 off1Gg.drawLine(275,ylabel2,275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[3]) ;
                 off1Gg.drawLine(270,whys[3],280,whys[3]) ;
                 ylabel = (int) (fact*(-h1/2.0 -h2 -h1)) + yt ;
                 off1Gg.drawString("H1",270,ylabel) ;
                 off1Gg.drawLine(275,whys[3],275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[0]) ;
                 off1Gg.drawLine(270,whys[0],280,whys[0]) ;
              }
              if (ktype == 4) {  // winged box
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[0] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 exes[1] = (int) (fact*(-w2/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h1 -h2)) + yt ;
                 exes[2] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[2] = (int) (fact*(0.0)) + yt ;
                 exes[3] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;

                 exes[0] = (int) (fact*(w1/2.0)) + xt ;
                 whys[0] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 exes[1] = (int) (fact*(w2/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h1 -h2)) + yt ;
                 exes[2] = (int) (fact*(w1/2.0)) + xt ;
                 whys[2] = (int) (fact*(0.0)) + yt ;
                 exes[3] = (int) (fact*(w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;

                 exes[0] = (int) (fact*(w1/2.0)) + xt ;
                 whys[0] = (int) (fact*(-h1)) + yt ;
                 exes[1] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h1)) + yt ;
                 exes[2] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[2] = (int) (fact*(0.0)) + yt ;
                 exes[3] = (int) (fact*(w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(0.0)) + yt ;
                 exes[4] = (int) (fact*(w1/2.0)) + xt ;
                 whys[4] = (int) (fact*(-h1)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
   
                 exes[0] = (int) (fact*(w1/2.0)) + xt ;
                 whys[0] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 exes[1] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[1] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 exes[2] = (int) (fact*(-w1/2.0)) + xt ;
                 whys[2] = (int) (fact*(-h1 -h2)) + yt ;
                 exes[3] = (int) (fact*(w1/2.0)) + xt ;
                 whys[3] = (int) (fact*(-h1 -h2)) + yt ;
                 exes[4] = (int) (fact*(w1/2.0)) + xt ;
                 whys[4] = (int) (fact*(-h1 -h2 -h1)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;

               //sticks
                 xlabel = (int) (fact*(-w2/2.0)) + xt ;
                 xlabel2 = (int) (fact*(w2/2.0)) + xt ;
                 ylabel2 = (int) (fact*(-h1 -h2)) + yt ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[1],yt,exes[1],whys[1]) ;
                 off1Gg.drawLine(xt,yt,xt,whys[1]) ;
                 off1Gg.drawLine(exes[4],yt,exes[4],whys[4]) ;
                 off1Gg.drawLine(xlabel,ylabel2,xlabel2,ylabel2) ;
            //Labels
                 off1Gg.setColor(Color.yellow) ;
                 off1Gg.drawString("W1",xt-5,337) ;
                 off1Gg.drawLine(exes[1],327,exes[1],337) ;
                 off1Gg.drawLine(exes[1],332,xt-20,332) ;
                 off1Gg.drawLine(exes[3],332,xt+20,332) ;
                 off1Gg.drawLine(exes[3],327,exes[3],337) ;

                 off1Gg.drawString("W2",xt-5,350) ;
                 off1Gg.drawLine(xlabel,340,xlabel,355) ;
                 off1Gg.drawLine(xlabel,345,xt-20,345) ;
                 off1Gg.drawLine(xlabel2,345,xt+20,345) ;
                 off1Gg.drawLine(xlabel2,340,xlabel2,355) ;

                 off1Gg.setColor(Color.blue) ;
                 off1Gg.fillRect(270,0,30,300) ;
                 off1Gg.setColor(Color.yellow) ;
                 ylabel = (int) (fact*(-h1/2.0)) + yt ;
                 ylabel2 = (int) (fact*(-h1)) + yt ;
                 off1Gg.drawString("H1",270,ylabel) ;
                 off1Gg.drawLine(270,yt,280,yt) ;
                 off1Gg.drawLine(275,yt,275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,ylabel2) ;
                 off1Gg.drawLine(270,ylabel2,280,ylabel2) ;
                 ylabel = (int) (fact*(-h1 -h2/2.0)) + yt ;
                 off1Gg.drawString("H2",270,ylabel) ;
                 off1Gg.drawLine(275,ylabel2,275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[3]) ;
                 off1Gg.drawLine(270,whys[3],280,whys[3]) ;
                 ylabel = (int) (fact*(-h1/2.0 -h2 -h1)) + yt ;
                 off1Gg.drawString("H1",270,ylabel) ;
                 off1Gg.drawLine(275,whys[3],275,ylabel+12) ;
                 off1Gg.drawLine(275,ylabel-12,275,whys[0]) ;
                 off1Gg.drawLine(270,whys[0],280,whys[0]) ;
              }
              if (ktype == 5) {  // twin-trapezoid
                 if (fold == -1) {
                    off1Gg.setColor(Color.white) ;
                    exes[0] = (int) (fact*(-h2/2.0)) + xt ;
                    whys[0] = (int) (fact*(-h1)) + yt ;
                    exes[1] = (int) (fact*(h2/2.0)) + xt ;
                    whys[1] = (int) (fact*(-h1)) + yt ;
                    exes[2] = (int) (fact*(h2/2.0)) + xt ;
                    whys[2] = (int) (fact*(0.0)) + yt ;
                    exes[3] = (int) (fact*(-h2/2.0)) + xt ;
                    whys[3] = (int) (fact*(0.0)) + yt ;
                    exes[4] = (int) (fact*(-h2/2.0)) + xt ;
                    whys[4] = (int) (fact*(-h1)) + yt ;
                    off1Gg.fillPolygon(exes,whys,4) ;
                    off1Gg.setColor(Color.lightGray) ;
                    exes[0] = (int) (fact*(-w2/2.0)) + xt ;
                    whys[0] = (int) (fact*(0.0)) + yt ;
                    exes[1] = (int) (fact*(w2/2.0)) + xt ;
                    whys[1] = (int) (fact*(0.0)) + yt ;
                    exes[2] = (int) (fact*(w1/2.0)) + xt ;
                    whys[2] = (int) (fact*(-h1)) + yt ;
                    exes[3] = (int) (fact*(-w1/2.0)) + xt ;
                    whys[3] = (int) (fact*(-h1)) + yt ;
                    exes[4] = (int) (fact*(-w2/2.0)) + xt ;
                    whys[4] = (int) (fact*(0.0)) + yt ;
                    off1Gg.fillPolygon(exes,whys,4) ;
                    off1Gg.setColor(Color.black) ;
                    off1Gg.drawLine(xt,whys[0],xt,whys[2]) ;
                  //Labels
                    off1Gg.setColor(Color.yellow) ;
                    off1Gg.drawString("W2",xt-5,337) ;
                    off1Gg.drawLine(exes[0],327,exes[0],337) ;
                    off1Gg.drawLine(exes[0],332,xt-20,332) ;
                    off1Gg.drawLine(exes[1],332,xt+20,332) ;
                    off1Gg.drawLine(exes[1],327,exes[1],337) ;
   
                    xlabel = (int) (fact*(-h2/2.0)) + xt ;
                    xlabel2 = (int) (fact*(h2/2.0)) + xt ;
                    off1Gg.drawString("H2",xt-5,350) ;
                    off1Gg.drawLine(xlabel,340,xlabel,355) ;
                    off1Gg.drawLine(xlabel,345,xt-20,345) ;
                    off1Gg.drawLine(xlabel2,345,xt+20,345) ;
                    off1Gg.drawLine(xlabel2,340,xlabel2,355) ;

                    xlabel = (int) (fact*(-w1/2.0)) + xt ;
                    xlabel2 = (int) (fact*(w1/2.0)) + xt ;
                    ylabel = (int) (fact*(-h1)) - 10 + yt ;
                    off1Gg.drawString("W1",xt-5,ylabel) ;
                    off1Gg.drawLine(xlabel,ylabel-5,xlabel,ylabel+5) ;
                    off1Gg.drawLine(xlabel,ylabel,xt-20,ylabel) ;
                    off1Gg.drawLine(xlabel2,ylabel,xt+20,ylabel) ;
                    off1Gg.drawLine(xlabel2,ylabel-5,xlabel2,ylabel+5) ;
   
                    off1Gg.setColor(Color.blue) ;
                    off1Gg.fillRect(270,0,30,300) ;
                    off1Gg.setColor(Color.yellow) ;
                    ylabel = (int) (fact*(-h1/2.0)) + yt ;
                    off1Gg.drawString("H1",270,ylabel) ;
                    off1Gg.drawLine(270,whys[2],280,whys[2]) ;
                    off1Gg.drawLine(275,whys[2],275,ylabel-12) ;
                    off1Gg.drawLine(275,ylabel+12,275,whys[0]) ;
                    off1Gg.drawLine(270,whys[0],280,whys[0]) ;
                 }
                 if (fold == 1) {
                    off1Gg.setColor(Color.white) ;
                    exes[0] = (int) (fact*(0.0)) + xt ;
                    whys[0] = (int) (fact*(-lkite)) + yt ;
                    exes[1] = (int) (fact*(lengstk/2.0)) + xt ;
                    whys[1] = (int) (fact*(-lkite + ((h2 - w1)/2.0) * Math.cos(convdr*beta))) + yt ;
                    exes[2] = exes[1] - (int) (fact*(h1 * Math.cos(convdr*beta))) ;
                    whys[2] = whys[1] + (int) (fact*(h1 * Math.sin(convdr*beta))) ;
                    exes[3] = (int) (fact*(0.0)) + xt ;
                    whys[3] = (int) (fact*(0.0)) + yt ;
                    exes[5] = (int) (fact*(-lengstk/2.0)) + xt ;
                    whys[5] = whys[1] ;
                    exes[4] = exes[5] + (int) (fact*(h1 * Math.cos(convdr*beta))) ;
                    whys[4] = whys[2] ;
                    exes[6] = (int) (fact*(0.0)) + xt ;
                    whys[6] = (int) (fact*(-lkite)) + yt ;
                    off1Gg.fillPolygon(exes,whys,6) ;
                    off1Gg.setColor(Color.black) ;
                    off1Gg.drawLine(xt,whys[0],xt,whys[3]) ;
                 }
   
                 off1Gg.setColor(Color.yellow) ;
                 if (fold == 1) {
                    off1Gg.drawString("Unfold",245,345) ;
                 }
                 else {
                    off1Gg.drawString("Fold",250,345) ;
                 }
              }
       // draw tail
              exes[0] = (int) (fact*(0.0)) + xt ;
              whys[0] = (int) (fact*(0.0)) + yt ;
              exes[1] = (int) (fact*(0.0)) + xt ;
              whys[1] = (int) (fact*(0.0 + ltail)) + yt ;
              off1Gg.setColor(Color.green) ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              if (ktype < 5) {
         // cg location
                 off1Gg.setColor(Color.red) ;
                 exes[0] = (int) (fact*(-w1/2.0)) -10 + xt ;
                 whys[0] = (int) (fact*(-cg)) + yt ;
                 off1Gg.fillOval(xt -3,whys[0]- 3,6,6);
                 off1Gg.drawString("Cg",exes[0],whys[0]) ;
         // cp location
                 off1Gg.setColor(Color.green) ;
                 whys[0] = (int) (fact*(-cp)) + yt ;
                 off1Gg.fillOval(xt -3,whys[0]- 3,6,6);
                 off1Gg.drawString("Cp",exes[0],whys[0]) ;
              }
              if (ktype == 5 && fold == 1) {
         // cg location
                 off1Gg.setColor(Color.red) ;
                 exes[0] = (int) (fact*(-w1/2.0)) -10 + xt ;
                 whys[0] = (int) (fact*(-cg)) + yt ;
                 off1Gg.fillOval(xt -3,whys[0]- 3,6,6);
                 off1Gg.drawString("Cg",exes[0],whys[0]) ;
         // cp location
                 off1Gg.setColor(Color.green) ;
                 whys[0] = (int) (fact*(-cp)) + yt ;
                 off1Gg.fillOval(xt -3,whys[0]- 3,6,6);
                 off1Gg.drawString("Cp",exes[0],whys[0]) ;
              }
           }
  // SIDE VIEW
           if (viewflg == 1) { 
              xt = 200; yt =225 ;
              if (ktype == 0) {  // diamond
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(0.0)) + xt ;
                 whys[0] = (int) (fact*(0.0)) + yt ;
                 exes[1] = (int) (fact*(-lkite * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-lkite * Math.sin(angk))) + yt ;
                 exes[2] = (int) (fact*(
                    -h2 * Math.cos(angk) + 2.0*Math.sin(angk))) + xt ;
                 whys[2] = (int) (fact*(
                    -h2 * Math.sin(angk) - 2.0 * Math.cos(angk))) + yt ;
                 exes[3] = (int) (fact*(0.0)) + xt ;
                 whys[3] = (int) (fact*(0.0)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
              }
              if (ktype == 1) {  // delta
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(0.0)) + xt ;
                 whys[0] = (int) (fact*(0.0)) + yt ;
                 exes[1] = (int) (fact*(-lkite * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-lkite * Math.sin(angk))) + yt ;
                 exes[2] = (int) (fact*(
                    -h2 * Math.cos(angk) + 2.0*Math.sin(angk))) + xt ;
                 whys[2] = (int) (fact*(
                    -h2 * Math.sin(angk) - 2.0 * Math.cos(angk))) + yt ;
                 exes[3] = (int) (fact*(0.0)) + xt ;
                 whys[3] = (int) (fact*(0.0)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[1],whys[1]+2,exes[2]+2,whys[2]) ;
              }
              if (ktype == 2) {  // sled
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(0.0)) + xt ;
                 whys[0] = (int) (fact*(0.0)) + yt ;
                 exes[1] = (int) (fact*(-lkite * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-lkite * Math.sin(angk))) + yt ;
                 exes[2] = (int) (fact*(
                    -h2 * Math.cos(angk) - xbr*Math.sin(angk))) + xt ;
                 whys[2] = (int) (fact*(
                    -h2 * Math.sin(angk) + xbr * Math.cos(angk))) + yt ;
                 exes[3] = (int) (fact*(0.0)) + xt ;
                 whys[3] = (int) (fact*(0.0)) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
              }
              if (ktype == 3) {  // box 
                 off1Gg.setColor(Color.white) ;
    // lower panel
                 exes[0] = (int) (fact*(-kbase * Math.sin(angk))) + xt ;
                 whys[0] = (int) (fact*(kbase * Math.cos(angk))) + yt ;
                 exes[1] = (int) (fact*(-kbase * Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(kbase * Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 exes[3] = (int) (fact*(+kbase*Math.sin(angk))) + xt ;
                 whys[3] = (int) (fact*(-kbase*Math.cos(angk))) + yt ;
                 exes[2] = (int) (fact*(+kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[2] = (int) (fact*(-kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
    // upper panel
                 exes[0] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                       - kbase*Math.sin(angk))) + xt ;
                 whys[0] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                       + kbase*Math.cos(angk))) + yt ;
                 exes[1] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        - kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        + kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 exes[3] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        + kbase*Math.sin(angk))) + xt ;
                 whys[3] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        - kbase*Math.cos(angk))) + yt ;
                 exes[2] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        + kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[2] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        - kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
          //sticks
                 exes[0] = (int) (fact*(-kbase*Math.sin(angk))) + xt ;
                 whys[0] = (int) (fact*(kbase*Math.cos(angk))) + yt ;
                 exes[1] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        - kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        + kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 exes[0] = (int) (fact*(0.0)) + xt ;
                 whys[0] = (int) (fact*(0.0)) + yt ;
                 exes[1] = (int) (fact*(-lkite*Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-lkite*Math.sin(angk))) + yt ;
                 off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 exes[3] = (int) (fact*(+kbase*Math.sin(angk))) + xt ;
                 whys[3] = (int) (fact*(-kbase*Math.cos(angk))) + yt ;
                 exes[2] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        + kbase*Math.sin(angk)
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[2] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        - kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 off1Gg.drawLine(exes[2],whys[2],exes[3],whys[3]) ;
              }
              if (ktype == 4) {  // winged box
                 off1Gg.setColor(Color.white) ;
    // lower panel
                 exes[0] = (int) (fact*(-kbase * Math.sin(angk))) + xt ;
                 whys[0] = (int) (fact*(kbase * Math.cos(angk))) + yt ;
                 exes[1] = (int) (fact*(-kbase * Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(kbase * Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 exes[3] = (int) (fact*(+kbase*Math.sin(angk))) + xt ;
                 whys[3] = (int) (fact*(-kbase*Math.cos(angk))) + yt ;
                 exes[2] = (int) (fact*(+kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[2] = (int) (fact*(-kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
    // upper panel
                 exes[0] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                       - kbase*Math.sin(angk))) + xt ;
                 whys[0] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                       + kbase*Math.cos(angk))) + yt ;
                 exes[1] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        - kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        + kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 exes[3] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        + kbase*Math.sin(angk))) + xt ;
                 whys[3] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        - kbase*Math.cos(angk))) + yt ;
                 exes[2] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        + kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[2] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        - kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 off1Gg.fillPolygon(exes,whys,4) ;
          //sticks
                 exes[0] = (int) (fact*(-kbase*Math.sin(angk))) + xt ;
                 whys[0] = (int) (fact*(kbase*Math.cos(angk))) + yt ;
                 exes[1] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        - kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        + kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 exes[0] = (int) (fact*(0.0)) + xt ;
                 whys[0] = (int) (fact*(0.0)) + yt ;
                 exes[1] = (int) (fact*(-lkite*Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-lkite*Math.sin(angk))) + yt ;
                 off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 exes[3] = (int) (fact*(+kbase*Math.sin(angk))) + xt ;
                 whys[3] = (int) (fact*(-kbase*Math.cos(angk))) + yt ;
                 exes[2] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        + kbase*Math.sin(angk)
                                        - h1 * Math.cos(angk))) + xt ;
                 whys[2] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        - kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                 off1Gg.drawLine(exes[2],whys[2],exes[3],whys[3]) ;
              }
              if (ktype == 5) {  // twin-trapezoid
                 off1Gg.setColor(Color.white) ;
                 exes[0] = (int) (fact*(0.0)) + xt ;
                 whys[0] = (int) (fact*(0.0)) + yt ;
                 exes[1] = (int) (fact*(-lkite * Math.cos(angk))) + xt ;
                 whys[1] = (int) (fact*(-lkite * Math.sin(angk))) + yt ;
                 exes[2] = exes[1] - (int) (fact*((w1/2.0) * Math.sin(angk))) ;
                 whys[2] = whys[1] + (int) (fact*((w1/2.0) * Math.cos(angk))) ;
                 exes[3] = exes[0] - (int) (fact*((w2/2.0) * Math.sin(angk))) ;
                 whys[3] = whys[0] + (int) (fact*((w2/2.0) * Math.cos(angk))) ;
                 exes[4] = (int) (fact*(0.0)) + xt ;
                 whys[4] = (int) (fact*(0.0)) + yt ;
                 off1Gg.setColor(Color.lightGray) ;
                 off1Gg.fillPolygon(exes,whys,4) ;
              }
      // draw tail
              exes[0] = (int) (fact*(0.0)) + xt ;
              whys[0] = (int) (fact*(0.0)) + yt ;
              off1Gg.setColor(Color.cyan) ;
              for (i=1; i<=8; ++ i) {
                ripple = (1 - ((i/2) * 2 / i)) * 1. ;
                if (i == 3) ripple = - ripple ;
                if (i == 7) ripple = - ripple ;
                exes[i] = (int) (fact*(.125*i*ltail * Math.cos(angk))) + xt ;
                whys[i] = (int) (fact*(.125*i*ltail * Math.sin(angk) + ripple)) + yt ;
                off1Gg.drawLine(exes[i],whys[i],exes[i-1],whys[i-1]) ;
              }
              off1Gg.drawString("T",exes[3]-5,whys[3]+15) ;
     // draw bridle
              exes[4] = (int) (fact*(-300. * Math.sin(angl))) + xt ;
              whys[4] = (int) (fact*(300. * Math.cos(angl))) + yt ;
              off1Gg.setColor(Color.white) ;
              switch (ktype) { 
                 case 0: {    // diamond 
                    exes[0] = (int) (fact*(0.0)) + xt ;
                    whys[0] = (int) (fact*(0.0)) + yt ;
                    exes[1] = (int) (fact*(-lknot * Math.cos(angk - anga))) + xt ;
                    whys[1] = (int) (fact*(-lknot * Math.sin(angk - anga))) + yt ;
                    exes[2] = (int) (fact*(-lkite * Math.cos(angk))) + xt ;
                    whys[2] = (int) (fact*(-lkite * Math.sin(angk))) + yt ;
                    off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
                    off1Gg.drawString("B",exes[1]-20,whys[1]) ;
                    off1Gg.fillOval(exes[1]-3,whys[1]-3,6,6) ;
                    off1Gg.setColor(Color.magenta) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[4],whys[4]) ;
                    off1Gg.setColor(Color.yellow) ;
                    off1Gg.drawLine(exes[0],whys[0]+10,exes[1],whys[1]+10) ;
                    off1Gg.drawString("K",(exes[0]+exes[1])/2,((whys[0]+whys[1])/2)+30) ;
                    off1Gg.drawLine(exes[0],whys[0]+5,exes[0],whys[0]+15) ;
                    off1Gg.drawLine(exes[1],whys[1]+5,exes[1],whys[1]+15) ;
                    break;
                 }
                 case 1: {    // delta
                    exes[0] = (int) (fact*(0.0)) + xt ;
                    whys[0] = (int) (fact*(0.0)) + yt ;
                    exes[1] = (int) (fact*(-lknot * Math.cos(angk - anga))) + xt ;
                    whys[1] = (int) (fact*(-lknot * Math.sin(angk - anga))) + yt ;
                    exes[2] = (int) (fact*(-lkite * Math.cos(angk))) + xt ;
                    whys[2] = (int) (fact*(-lkite * Math.sin(angk))) + yt ;
                    exes[3] = (int) (fact*(0.0)) + xt ;
                    whys[3] = (int) (fact*(0.0)) + yt ;
                    off1Gg.setColor(Color.magenta) ;
                    off1Gg.fillPolygon(exes,whys,4) ;
                    off1Gg.setColor(Color.white) ;
                    off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
                    off1Gg.drawString("B",exes[1]-20,whys[1]) ;
                    off1Gg.fillOval(exes[1]-3,whys[1]-3,6,6) ;
                    off1Gg.setColor(Color.magenta) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[4],whys[4]) ;
                    off1Gg.setColor(Color.yellow) ;
                    off1Gg.drawLine(exes[0],whys[0]+10,exes[1],whys[1]+10) ;
                    off1Gg.drawString("K",(exes[0]+exes[1])/2,((whys[0]+whys[1])/2)+30) ;
                    off1Gg.drawLine(exes[0],whys[0]+5,exes[0],whys[0]+15) ;
                    off1Gg.drawLine(exes[1],whys[1]+5,exes[1],whys[1]+15) ;
                    break;
                 }
                 case 2: {    // sled
                    exes[0] = (int) (fact*(
                       -h2 * Math.cos(angk) - xbr*Math.sin(angk))) + xt ;
                    whys[0] = (int) (fact*(
                       -h2 * Math.sin(angk) + xbr * Math.cos(angk))) + yt ;

                    exes[1] = exes[0] -
                             (int) (fact*(lbrid/2.0 * Math.sin(angl)));
                    whys[1] = whys[0] +
                             (int) (fact*(lbrid/2.0 * Math.cos(angl)));
                    off1Gg.setColor(Color.white) ;
                    off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                    off1Gg.drawString("B/2",exes[1]-10,whys[1]-10) ;
                    off1Gg.fillOval(exes[1]-3,whys[1]-3,6,6) ;
                    off1Gg.setColor(Color.magenta) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[4],whys[4]) ;
                    break;
                 }
                 case 3: {    // box 
                    exes[0] = (int) (fact*(-kbase*Math.sin(angk))) + xt ;
                    whys[0] = (int) (fact*(kbase*Math.cos(angk))) + yt ;
                    exes[1] = (int) (fact*(-kbase*Math.sin(angk)
                                     - lknot * Math.cos(angk - anga))) + xt ;
                    whys[1] = (int) (fact*(kbase*Math.cos(angk)
                                     - lknot * Math.sin(angk - anga))) + yt ;
                    exes[2] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        - kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                    whys[2] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        + kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                    off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
                    off1Gg.drawString("B",exes[1]-20,whys[1]) ;
                    off1Gg.fillOval(exes[1]-3,whys[1]-3,6,6) ;
                    off1Gg.setColor(Color.magenta) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[4],whys[4]) ;
                    off1Gg.setColor(Color.yellow) ;
                    off1Gg.drawLine(exes[0],whys[0]+10,exes[1],whys[1]+10) ;
                    off1Gg.drawString("K",(exes[0]+exes[1])/2,((whys[0]+whys[1])/2)+30) ;
                    off1Gg.drawLine(exes[0],whys[0]+5,exes[0],whys[0]+15) ;
                    off1Gg.drawLine(exes[1],whys[1]+5,exes[1],whys[1]+15) ;
                    break;
                 }
                 case 4: {    // winged box 
                    exes[0] = (int) (fact*(-kbase*Math.sin(angk))) + xt ;
                    whys[0] = (int) (fact*(kbase*Math.cos(angk))) + yt ;
                    exes[1] = (int) (fact*(-kbase*Math.sin(angk)
                                     - lknot * Math.cos(angk - anga))) + xt ;
                    whys[1] = (int) (fact*(kbase*Math.cos(angk)
                                     - lknot * Math.sin(angk - anga))) + yt ;
                    exes[2] = (int) (fact*(-(h1+h2)*Math.cos(angk)
                                        - kbase*Math.sin(angk) 
                                        - h1 * Math.cos(angk))) + xt ;
                    whys[2] = (int) (fact*(-(h1+h2)*Math.sin(angk)
                                        + kbase*Math.cos(angk)
                                        - h1 * Math.sin(angk))) + yt ;
                    off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
                    off1Gg.drawString("B",exes[1]-20,whys[1]) ;
                    off1Gg.fillOval(exes[1]-3,whys[1]-3,6,6) ;
                    off1Gg.setColor(Color.magenta) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[4],whys[4]) ;
                    off1Gg.setColor(Color.yellow) ;
                    off1Gg.drawLine(exes[0],whys[0]+10,exes[1],whys[1]+10) ;
                    off1Gg.drawString("K",(exes[0]+exes[1])/2,((whys[0]+whys[1])/2)+30) ;
                    off1Gg.drawLine(exes[0],whys[0]+5,exes[0],whys[0]+15) ;
                    off1Gg.drawLine(exes[1],whys[1]+5,exes[1],whys[1]+15) ;
                    break;
                 }
                 case 5: {    // twin-trapezoid 
                    exes[0] = (int) (fact*(0.0)) + xt ;
                    whys[0] = (int) (fact*(0.0)) + yt ;
                    exes[1] = (int) (fact*(-lknot * Math.cos(angk - anga))) + xt ;
                    whys[1] = (int) (fact*(-lknot * Math.sin(angk - anga))) + yt ;
                    off1Gg.setColor(Color.magenta) ;
                    off1Gg.fillOval(exes[1]-3,whys[1]-3,6,6) ;
                    off1Gg.drawLine(exes[1],whys[1],exes[4],whys[4]) ;
                    off1Gg.setColor(Color.yellow) ;
                    off1Gg.drawLine(exes[0],whys[0]+10,exes[1],whys[1]+10) ;
                    off1Gg.drawString("K",(exes[0]+exes[1])/2,((whys[0]+whys[1])/2)+30) ;
                    off1Gg.drawLine(exes[0],whys[0]+5,exes[0],whys[0]+15) ;
                    off1Gg.drawLine(exes[1],whys[1]+5,exes[1],whys[1]+15) ;
                    break;
                 }
              }
   // compute force vectors
   // weight - cg location
              exes[0] = (int) (fact*(-cg * Math.cos(angk))) + xt ;
              whys[0] = (int) (fact*(-cg * Math.sin(angk))) + yt ;
              exes[1] = (int) (fact*(-cg * Math.cos(angk))) + xt ;
              whys[1] = (int) (fact*(-(cg * Math.sin(angk) - weight*vscale))) + yt ;
              off1Gg.setColor(Color.red) ;
              off1Gg.fillOval(exes[0]-3,whys[0]-3,6,6);
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              exes[0] = exes[1] - 3 ;
              whys[0] = whys[1] - 5 ;
              exes[2] = exes[1] + 3 ;
              whys[2] = whys[1] - 5 ;
              exes[3] = exes[1] - 3 ;
              whys[3] = whys[1] - 5 ;
              off1Gg.fillPolygon(exes,whys,4) ;
              off1Gg.drawString("Weight",exes[1]+5,whys[1]-5) ;
   // lift - cp location
              exes[0] = (int) (fact*(-cp * Math.cos(angk))) + xt ;
              whys[0] = (int) (fact*(-cp * Math.sin(angk))) + yt ;
              exes[1] = (int) (fact*(-cp * Math.cos(angk))) + xt ;
              whys[1] = (int) (fact*(-(cp * Math.sin(angk) + lift*vscale))) + yt ;
              off1Gg.setColor(Color.green) ;
              off1Gg.fillOval(exes[0]-3,whys[0]-3,6,6);
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              exes[0] = exes[1] - 3 ;
              whys[0] = whys[1] + 5 ;
              exes[2] = exes[1] + 3 ;
              whys[2] = whys[1] + 5 ;
              exes[3] = exes[1] - 3 ;
              whys[3] = whys[1] + 5 ;
              off1Gg.fillPolygon(exes,whys,4) ;
              off1Gg.drawString("Lift",exes[1]+20,whys[1]+10) ;
   // drag - cp location
              exes[0] = (int) (fact*(-cp * Math.cos(angk))) + xt ;
              whys[0] = (int) (fact*(-cp * Math.sin(angk))) + yt ;
              exes[1] = (int) (fact*(-cp * Math.cos(angk) + drag*vscale)) + xt ;
              whys[1] = (int) (fact*(-cp * Math.sin(angk))) + yt ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              exes[0] = exes[1] - 5 ;
              whys[0] = whys[1] - 3 ;
              exes[2] = exes[1] - 5 ;
              whys[2] = whys[1] + 3 ;
              exes[3] = exes[1] - 5 ;
              whys[3] = whys[1] - 3 ;
              off1Gg.fillPolygon(exes,whys,4) ;
              off1Gg.drawString("Drag",exes[1]-30,whys[1]-5) ;
   // wind vector
              exes[0] = 100 ;
              whys[0] = 275;
              exes[1] = 100 + (int) (5.*wind) ;
              whys[1] = 275 ;
              off1Gg.setColor(Color.green) ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              exes[0] = exes[1] - 5 ;
              whys[0] = whys[1] - 3 ;
              exes[2] = exes[1] - 5 ;
              whys[2] = whys[1] + 3 ;
              exes[3] = exes[1] - 5 ;
              whys[3] = whys[1] - 3 ;
              off1Gg.fillPolygon(exes,whys,4) ;
              off1Gg.drawString("Wind",exes[1]-30,whys[1]+15) ;
 /* scale  widget */
              off1Gg.setColor(Color.blue) ;
              off1Gg.fillRect(0,330,250,30) ;
              off1Gg.setColor(Color.yellow) ;
              off1Gg.drawString("Scale",55,345) ;
              off1Gg.drawLine(95,340,225,340) ;
              off1Gg.fillRect(vldloc,330,5,20) ;
           }

 // FIELD VIEW
           if (viewflg == 2) { 
              xt = 50; yt =300 ;
              off1Gg.setColor(Color.green) ;
              off1Gg.fillRect(30,260,280,60) ;
              off1Gg.setColor(Color.white) ;
              exes[1] = (int) (fact*fscale*(xline[0])) + xt ;
              whys[1] = (int) (fact*fscale*(-yline[0])) + yt ;
              for (i=1; i<=20; ++i) {
                 exes[0] = exes[1] ;
                 whys[0] = whys[1] ;
                 exes[1] = (int) (fact*fscale*(xline[i])) + xt ;
                 whys[1] = (int) (fact*fscale*(-yline[i])) + yt ;
                 off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              }

              exes[0] = (int) (fact*fscale*(xline[0])) + xt ;
              whys[0] = (int) (fact*fscale*(-yline[0])) + yt ;
              exes[1] = (int) (fact*fscale*(xline[0] -2.0)) + xt ;
              whys[1] = (int) (fact*fscale*(-yline[0] +1.2)) + yt ;
              off1Gg.setColor(Color.black) ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              exes[1] = (int) (fact*fscale*(xline[0] -3.5)) + xt ;
              whys[1] = (int) (fact*fscale*(-yline[0] + 1.2)) + yt ;
              off1Gg.fillRect(exes[1],whys[1],
                  ((int) (fact*fscale*1.5)),((int) (fact*fscale*3.0)));
              exes[0] = (int) (fact*fscale*(xline[0] -3.5)) + xt ;
              whys[0] = (int) (fact*fscale*(-yline[0] + 4.0)) + yt ;
              exes[1] = (int) (fact*fscale*(xline[0] -3.5)) + xt ;
              whys[1] = (int) (fact*fscale*(-yline[0] + 6.0)) + yt ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              exes[0] = (int) (fact*fscale*(xline[0] -2.2)) + xt ;
              whys[0] = (int) (fact*fscale*(-yline[0] + 4.0)) + yt ;
              exes[1] = (int) (fact*fscale*(xline[0] -2.2)) + xt ;
              whys[1] = (int) (fact*fscale*(-yline[0] + 6.0)) + yt ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              exes[1] = (int) (fact*fscale*(xline[0]-3.0)) + xt ;
              whys[1] = (int) (fact*fscale*(-yline[0])) + yt ;
              off1Gg.fillRect(exes[1],whys[1],
                  ((int) (fact*fscale*1.2)),((int) (fact*fscale*1.0)));

              exes[0] = (int) (fact*fscale*(xline[20])) + xt;
              whys[0] = (int) (fact*fscale*(-yline[20])) + yt - 3 ;
              exes[1] = exes[0] + 3 ;
              whys[1] = whys[0] + 3 ;
              exes[2] = exes[0] ;
              whys[2] = whys[0] + 10 ;
              exes[3] = exes[0] - 3 ;
              whys[3] = whys[0] + 3 ;
              exes[4] = exes[0] ;
              whys[4] = whys[0] ;
              off1Gg.setColor(Color.white) ;
              off1Gg.fillPolygon(exes,whys,5) ;

              exes[0] = (int) (fact*fscale*(xline[0])) + xt ;
              whys[0] = (int) (fact*fscale*(-yline[0])) + yt ;
              exes[1] = (int) (fact*fscale*(xline[20])) + xt ;
              whys[1] = whys[0];
              off1Gg.setColor(Color.red) ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              off1Gg.drawString("X",exes[1]-50,whys[1]+12) ;

              exes[0] = (int) (fact*fscale*(xline[20])) + xt ;
              whys[0] = (int) (fact*fscale*(-yline[0])) + yt ;
              exes[1] = exes[0] ;
              whys[1] = (int) (fact*fscale*(-yline[20])) + yt ;
              off1Gg.setColor(Color.yellow) ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              off1Gg.drawString("Y",exes[1]+10,whys[1]+50) ;
   // wind vector
              exes[0] = 100 ;
              whys[0] = 75;
              exes[1] = 100 + (int) (5.*wind) ;
              whys[1] = 75 ;
              off1Gg.setColor(Color.green) ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              exes[0] = exes[1] - 5 ;
              whys[0] = whys[1] - 3 ;
              exes[2] = exes[1] - 5 ;
              whys[2] = whys[1] + 3 ;
              exes[3] = exes[1] - 5 ;
              whys[3] = whys[1] - 3 ;
              off1Gg.fillPolygon(exes,whys,4) ;
              off1Gg.drawString("Wind",exes[1]-30,whys[1]+15) ;
 /* scale  widget */
              off1Gg.setColor(Color.blue) ;
              off1Gg.fillRect(0,330,250,30) ;
              off1Gg.setColor(Color.yellow) ;
              off1Gg.drawString("Scale",55,345) ;
              off1Gg.drawLine(95,340,225,340) ;
              off1Gg.fillRect(fldloc,330,5,20) ;
 // warning 
              if (ground == 1) {
                 off1Gg.setColor(Color.yellow) ;
                 off1Gg.drawString("Line Touches Ground",50,200) ;
              }
           }

 // unstable design
           if (unstab == 1) {
              off1Gg.setColor(Color.yellow) ;
              off1Gg.fillRect(50,250,200,70) ;
              off1Gg.setColor(Color.red) ;
              off1Gg.fillRect(70,265,160,40) ;
              off1Gg.setColor(Color.white) ;
              off1Gg.drawString("UNSTABLE DESIGN",90,280) ;
              off1Gg.drawString(" OR  TRIM  ",130,295) ;
           }

 // zoom widget
           off1Gg.setColor(Color.blue) ;
           off1Gg.fillRect(0,10,30,325) ;
           off1Gg.setColor(Color.yellow) ;
           off1Gg.drawString("Large",5,42) ;
           off1Gg.drawString("Small",2,295) ;
           off1Gg.drawLine(15,50,15,275) ;
           off1Gg.fillRect(5,sldloc,20,5) ;

 // control buttons
           off1Gg.setColor(Color.blue) ;
           off1Gg.fillRect(0,0,300,20) ;

           off1Gg.setColor(Color.yellow) ;
           off1Gg.drawRect(5,0,55,17) ;
           if (units == 1) {
              off1Gg.drawString("Metric",15,12) ;
           }
           else {
              off1Gg.drawString("English",10,12) ;
           }

           off1Gg.setColor(Color.yellow) ;
           off1Gg.drawRect(65,0,80,17) ;
           if (pick == 1) {
              off1Gg.fillRect(65,0,80,17) ;
              off1Gg.setColor(Color.black) ;
           }
           off1Gg.drawString("Select View->",70,12) ;

           off1Gg.setColor(Color.yellow) ;
           off1Gg.drawRect(150,0,50,17) ;
           if (viewflg == 0) {
              off1Gg.fillRect(150,0,50,17) ;
              off1Gg.setColor(Color.black) ;
           }
           off1Gg.drawString("Front",160,12) ;

           off1Gg.setColor(Color.yellow) ;
           off1Gg.drawRect(200,0,40,17) ;
           if (viewflg == 1) {
              off1Gg.fillRect(200,0,40,17) ;
              off1Gg.setColor(Color.black) ;
           }
           off1Gg.drawString("Side",210,12) ;

           off1Gg.setColor(Color.yellow) ;
           off1Gg.drawRect(240,0,50,17) ;
           if (viewflg == 2) {
              off1Gg.fillRect(240,0,50,17) ;
              off1Gg.setColor(Color.black) ;
           }
           off1Gg.drawString("Field",250,12) ;
      // reset button
           off1Gg.setColor(Color.yellow) ;
           off1Gg.drawRect(240,330,45,17) ;
           if (ktype <= 4) {
              off1Gg.drawString("Reset",245,345) ;
           }
      // save button
           off1Gg.setColor(Color.yellow) ;
           off1Gg.drawRect(2,330,45,17) ;
           off1Gg.drawString("Save",7,345) ;

           g.drawImage(offImg1,0,0,this) ;   

       }  // Paint
    }  // Viewer
  } //end of Slvpnl 

  class P extends Panel {
     TextArea prnt ;

     P () {

        setLayout(new GridLayout(1,1,0,0)) ;

        prnt = new TextArea() ;
        prnt.setEditable(false) ;

        prnt.appendText("KiteModeler 1.3 beta - 1 Oct 02  --  NASA Glenn  Learning Technologies Project\n") ;
        add(prnt) ;
     }

  } // end of P
} // Kite
