/*  
                          AtmosModeler

              Program to solve Standard Atmosphere Equations
            and using input velocity, compute the Mach number
                    and dynamic pressure effects

                     Version 1.2a   - 7 Dec 05

                         Written by Tom Benson
                       NASA Glenn Research Center

>                              NOTICE
>This software is in the Public Domain.  It may be freely copied and used in
>non-commercial products, assuming proper credit to the author is given.  IT
>MAY NOT BE RESOLD.  If you want to use the software for commercial
>products, contact the author.
>No copyright is claimed in the United States under Title 17, U. S. Code.
>This software is provided "as is" without any warranty of any kind, either
>express, implied, or statutory, including, but not limited to, any warranty
>that the software will conform to specifications, any implied warranties of
>merchantability, fitness for a particular purpose, and freedom from
>infringement, and any warranty that the documentation will conform to the
>program, or any warranty that the software will be error free.
>In no event shall NASA be liable for any damages, including, but not
>limited to direct, indirect, special or consequential damages, arising out
>of, resulting from, or in any way connected with this software, whether or
>not based on warranty, contract, tort or otherwise, whether or not injury
>was sustained by persons or property or otherwise, and whether or not loss
>was sustained from, or arose out of the results of, or use of, the software
>or services provided hereunder.

      New Test :
                  * rename some modules
                    add output options


                                                     TJB 7 Dec 05
*/

import java.awt.*;
import java.lang.Math ;

public class Atmos extends java.applet.Applet {

   double gama,alt,temp,press,vel,q0,tt0 ;
   double rgas, rho0, rho, a0, lrat, mach ;
   int lunits,var,planet,ther,prs ;
   int mode, ygrph, sldloc, outmod;

   Viewer view ;
   Inp inp ;
   CardLayout layout ;
   Image plnimg ;
   Image mplnimg ;
   Image offscreenImg ;
   Graphics offsGg ;
   Image offImg2 ;
   Graphics off2Gg ;
   Point locate ;

   public void init() {

     offscreenImg = createImage(this.size().width,
                      this.size().height) ;
     offsGg = offscreenImg.getGraphics() ;
     offImg2 = createImage(this.size().width,
                      this.size().height) ;
     off2Gg = offImg2.getGraphics() ;
 
     plnimg = getImage(getCodeBase(),
      "plane.gif");
     mplnimg = getImage(getCodeBase(),
      "mpln.gif");
     locate = new Point(100,300) ;

     setLayout(new GridLayout(1,2,0,0)) ;

     setDefaults () ;

     view = new Viewer(this) ;

     inp = new Inp(this) ;

     add(view) ;
     add(inp) ;

     computeAtmos() ;
  }
 
  public Insets insets() {
     return new Insets(10,10,10,10) ;
  }

  public void setDefaults() {
     var = 0 ;
     alt = 0.0 ;
     vel = 0.0 ;
     planet = 0 ;
     mode = 0 ;
     outmod = 0 ;
     ygrph = locate.y ;
     sldloc = 300 ;
  }

  public void computeAtmos() {
 
     rgas = 1718. ;                /* ft2/sec2 R */
     gama = 1.4 ;
     if (planet == 0) {    // Earth  standard day
       if (alt <= 36152.) {           // Troposphere
          temp = 518.6 - 3.56 * alt/1000. ;
          press = 2116. * Math.pow(temp/518.6,5.256) ;
       }
       if (alt >= 36152. && alt <= 82345.) {   // Stratosphere
          temp = 389.98 ;
          press = 2116. * .2236 *
             Math.exp((36000.-alt)/(53.35*389.98)) ;
       }
       if (alt >= 82345.) {          
          temp = 389.98 + 1.645 * (alt-82345)/1000. ;
          press = 2116. *.02456 * Math.pow(temp/389.98,-11.388) ;
       }
     }

     if (planet == 1) {   // Mars - curve fit of orbiter data
       rgas = 1149. ;                /* ft2/sec2 R */
       gama = 1.29 ;

       if (alt <= 22960.) {     
          temp = 434.02 - .548 * alt/1000. ;
          press = 14.62 * Math.pow(2.71828,-.00003 * alt) ;
       }
       if (alt > 22960.) {  
          temp = 449.36 - 1.217 * alt/1000. ;
          press = 14.62 * Math.pow(2.71828,-.00003 * alt) ;
       }
     }

     a0 = Math.sqrt(gama*rgas*temp) ;
     rho = press/(rgas*temp) ;
     mach = vel/a0 ;
     q0 = .5 * gama * press * mach * mach ;
     tt0 = temp * (1.0 + .5 * (gama - 1.0) * mach * mach) ;

     rho0 = .00237 ;
     lrat = 100. * rho / rho0 ;

     if (lunits == 0) {
             // convert to English (Farenheit and psi)
         temp = temp - 459.7 ;
         press = press / 144. ;
         a0 = a0 * 60.0 / 88.0 ;
         tt0 = tt0 - 459.7 ;
     }
     if (lunits == 1) {
             // convert to metric (Celius and k-PA)
         temp = temp * .55555 - 273.1 ;
         press = press * .04787 ;
         rho = rho * 515.4 ;
         a0 = a0 * .3048 * 3.6 ;
         q0 = q0 * .04787 ;
         tt0 = tt0 * .55555 - 273.1 ;
     }

     if (var == 0) {
         inp.up.la.setText(" ") ;
         inp.up.lb.setText(" ") ;
         inp.up.lc.setText(" ") ;
         inp.up.o2.setText(String.valueOf(filter0(temp))) ;
         if (lunits == 0) inp.up.l2u.setText("  F  ") ;
         if (lunits == 1) inp.up.l2u.setText("  C  ") ;
     }
     if (var == 1) {
         inp.up.la.setText(" ") ;
         inp.up.lb.setText(" ") ;
         inp.up.lc.setText(" ") ;
         inp.up.o2.setText(String.valueOf(filter3(press))) ;
         if (lunits == 0) inp.up.l2u.setText(" psi  ") ;
         if (lunits == 1) inp.up.l2u.setText(" K-Pa  ") ;
     }
     if (var == 2) {
         inp.up.la.setText(" ") ;
         inp.up.lb.setText(" ") ;
         inp.up.lc.setText(" ") ;
         if (lunits == 0) {
            inp.up.o2.setText(String.valueOf(filter5(rho))) ;
            inp.up.l2u.setText("slugs / cu ft") ;
         }
         if (lunits == 1) {
            inp.up.o2.setText(String.valueOf(filter3(rho))) ;
            inp.up.l2u.setText("kg / cu m") ;
         }
     }
     if (var == 3) {
         inp.up.la.setText(" ") ;
         inp.up.lb.setText(" ") ;
         inp.up.lc.setText(" ") ;
         inp.up.o2.setText(String.valueOf(filter0(a0))) ;
         if (lunits == 0) inp.up.l2u.setText(" mph  ") ;
         if (lunits == 1) inp.up.l2u.setText(" km/hr  ") ;
     }
     if (var == 4) {
         inp.up.la.setText(" ") ;
         inp.up.lb.setText(" ") ;
         inp.up.lc.setText(" ") ;
         inp.up.o2.setText(String.valueOf(filter0(q0))) ;
         if (lunits == 0) inp.up.l2u.setText(" psf  ") ;
         if (lunits == 1) inp.up.l2u.setText(" K-Pa  ") ;
     }
     if (var == 5) {
         if (lrat >= 10.0) 
            inp.up.o2.setText(String.valueOf(filter0(lrat))) ;
         if (lrat < 10.0)
            inp.up.o2.setText(String.valueOf(filter3(lrat))) ;
         inp.up.l2u.setText(" ") ;
         inp.up.la.setText("% of Force on ") ;
         inp.up.lb.setText("Earth, Alt = 0") ;
         inp.up.lc.setText("Same Velocity") ;
     }
     if (var == 6) {
         inp.up.o2.setText(String.valueOf(filter3(mach))) ;
         inp.up.l2u.setText(" ") ;
         inp.up.lb.setText(" ") ;
         inp.up.la.setText(" ") ;
         inp.up.lc.setText(" ") ;
     }
     if (var == 7) {
         inp.up.la.setText(" ") ;
         inp.up.lb.setText(" ") ;
         inp.up.lc.setText(" ") ;
         inp.up.o2.setText(String.valueOf(filter0(tt0))) ;
         if (lunits == 0) inp.up.l2u.setText("  F  ") ;
         if (lunits == 1) inp.up.l2u.setText("  C  ") ;
     }

     if (mode == 0) loadAlt() ;
     loadGage () ;
     loadOut () ;
     view.repaint() ;
  }

  public void loadOut() {
     String outvel,outprs,outp2,outden,outtem,outpc  ;

     outvel = " mph" ;
     if (lunits == 1) outvel = " km/hr" ;
     outprs = " psi" ;
     if (lunits == 1) outprs = " kPa" ;
     outp2 = " psf" ;
     if (lunits == 1) outprs = " kPa" ;
     outtem = " F" ;
     if (lunits == 1) outtem = " C" ;
     outden = " slug/cu ft" ;
     if (lunits == 1) outden = " kg/cu m" ;
     outpc = " %" ;

     inp.up.o4.setText(String.valueOf(filter3(mach))) ;

     inp.dn.dat.o1.setText(String.valueOf(filter0(temp)) + outtem) ;
     inp.dn.dat.o2.setText(String.valueOf(filter3(press)) + outprs) ;
     inp.dn.dat.o3.setText(String.valueOf(filter5(rho)) + outden) ;
     inp.dn.dat.o4.setText(String.valueOf(filter0(a0)) + outvel) ;
     inp.dn.dat.o5.setText(String.valueOf(filter0(q0)) + outp2) ;
     inp.dn.dat.o6.setText(String.valueOf(filter0(lrat)) + outpc) ;
     inp.dn.dat.o7.setText(String.valueOf(filter0(tt0)) + outtem) ;
  }

  public void loadAlt() {
     if (lunits == 0) {
         inp.up.o1.setText(String.valueOf(filter0(alt))) ;
         inp.up.o3.setText(String.valueOf(filter0(vel * 60.0 / 88.0))) ;
     }
     if (lunits == 1) {
         inp.up.o1.setText(String.valueOf(filter0(alt*.3048))) ;
         inp.up.o3.setText(String.valueOf(filter0(vel*.3048 * 3.6))) ;
     }
  }

  public int filter0(double inumbr) {
     //  integer output
       float number ;
       int intermed ;

       intermed = (int) (inumbr) ;
       number = (float) (intermed);
       return intermed ;
  }
 
  public float filter3(double inumbr) {
     //  output only to .001
       float number ;
       int intermed ;
  
       intermed = (int) (inumbr * 1000.) ;
       number = (float) (intermed / 1000. );
       return number ;
  }

  public float filter5(double inumbr) {
     //  output only to .00001
       float number ;
       int intermed ;
  
       intermed = (int) (inumbr * 100000.) ;
       number = (float) (intermed / 100000. );
       return number ;
  }

  public void loadGage() {
     ther = 70 - (int) (.5 * temp) ;
     if(lunits == 0) 
         prs = 152 - (int) (10. * press) ;
     if(lunits == 1) 
         prs = 152 - (int) (press) ;

     inp.dn.grf.repaint() ;
  }

  class Inp extends Panel {
     Atmos outerparent ;
     Up up ;
     Dn dn ;

     Inp (Atmos target) {                           
        outerparent = target ;
        setLayout(new GridLayout(2,1,5,5)) ;

        up = new Up(outerparent) ;
        dn = new Dn(outerparent) ;

        add(up) ;
        add(dn) ;
     }

     class Up extends Panel {
        TextField o1,o2,o3,o4 ;
        Label la,lb,lc ;
        Label l1,l1u,l2u,l3,l3u ;
        Choice untch,plnitch,pltch,outch ;

        Up (Atmos target) {                           
            outerparent = target ;
            setLayout(new GridLayout(6,4,0,5)) ;
    
            l1 = new Label("Altitude", Label.CENTER) ;
            l1u = new Label(" feet ", Label.LEFT) ;
   
            o1 = new TextField() ;
            o1.setBackground(Color.white) ;
            o1.setForeground(Color.black) ;

            l2u = new Label(" F ", Label.LEFT) ;

            o2 = new TextField() ;
            o2.setBackground(Color.black) ;
            o2.setForeground(Color.yellow) ;
 
            o4 = new TextField() ;
            o4.setBackground(Color.black) ;
            o4.setForeground(Color.yellow) ;
 
            l3 = new Label("Velocity", Label.CENTER) ;
            l3u = new Label("mph", Label.LEFT) ;
   
            o3 = new TextField() ;
            o3.setBackground(Color.white) ;
            o3.setForeground(Color.black) ;

            untch = new Choice() ;
            untch.addItem("English Units") ;
            untch.addItem("Metric Units");
            untch.select(0) ;

            plnitch = new Choice() ;
            plnitch.addItem("Earth") ;
            plnitch.addItem("Mars");
            plnitch.select(0) ;

            pltch = new Choice() ;
            pltch.addItem("Air Temp.") ;
            pltch.addItem("Pressure");
            pltch.addItem("Density");
            pltch.addItem("Sound Speed");
            pltch.addItem("Dynamic Press");
            pltch.addItem("Force Ratio");
            pltch.addItem("Mach");
            pltch.addItem("Total Temp.");
            pltch.select(0) ;

            la = new Label(" ", Label.LEFT) ;
            lb = new Label("  ", Label.CENTER) ;
            lc = new Label("  ", Label.CENTER) ;

            untch = new Choice() ;
            untch.addItem("English") ;
            untch.addItem("Metric");
            untch.select(0) ;

            outch = new Choice() ;
            outch.addItem("Graph") ;
            outch.addItem("Data");
            outch.select(0) ;

            add(new Label("Planet :", Label.RIGHT)) ;
            add(plnitch) ;
            add(new Label("Units :", Label.RIGHT)) ;
            add(untch) ;

            add(new Label("Type In", Label.RIGHT)) ;
            add(l1) ;
            add(o1) ;
            add(l1u) ;

            add(new Label("Type In", Label.RIGHT)) ;
            add(l3) ;
            add(o3) ;
            add(l3u) ;

            add(new Label("Output :", Label.RIGHT)) ;
            add(outch) ;
            add(new Label("Mach ", Label.RIGHT)) ;
            add(o4) ;
 
            add(new Label("Select Output:", Label.LEFT)) ;
            add(pltch) ;
            add(o2) ;
            add(l2u) ;

            add(new Label(" ", Label.LEFT)) ;
            add(la) ;
            add(lb) ;
            add(lc) ;
        }

        public Insets insets() {
           return new Insets(5,5,5,5) ;
        }

        public boolean action(Event evt, Object arg) {
            if(evt.target instanceof Choice) {
               this.handleProb(arg) ;
               return true ;
            }

            if(evt.id == Event.ACTION_EVENT) {
               this.handleText(evt) ;
               return true ;
            }

            else return false ;
        }
 
        public void handleProb(Object obj) {

            lunits  = untch.getSelectedIndex() ;
            var     = pltch.getSelectedIndex() ;
            planet  = plnitch.getSelectedIndex() ;
            outmod  = outch.getSelectedIndex() ;
 
            if (lunits == 0) {
                l3u.setText(" mph ") ;
                l1u.setText("  feet ") ;
            }
            if (lunits == 1) {
                l3u.setText(" km/hr ") ;
                l1u.setText("  meters ") ;
            }

            if(outmod == 0) layout.show(dn, "first") ;
            if(outmod == 1) layout.show(dn, "second") ;

            mode = 0 ;
            computeAtmos() ;
        }

        public void handleText(Event evt) {
            Double V1,V3 ;
            double v1,v3 ;

            V1 = Double.valueOf(o1.getText()) ;
            v1 = V1.doubleValue() ;
            V3 = Double.valueOf(o3.getText()) ;
            v3 = V3.doubleValue() ;

            if (lunits == 0) {
                if (v1 < 0.0) {
                   v1 = 0.0 ;
                   o1.setText(String.valueOf(filter0(v1))) ;
                }
                if (v1 >100000.0) {
                   v1 = 100000.0 ;
                   o1.setText(String.valueOf(filter0(v1))) ;
                }
                if (v3 < 0.0) {
                   v3 = 0.0 ;
                   o3.setText(String.valueOf(filter0(v3))) ;
                }
                if (v3 >5000.0) {
                   v3 = 5000.0 ;
                   o3.setText(String.valueOf(filter0(v3))) ;
                }
                alt = v1 ;
                vel = v3 * 88.0 / 60.0 ;
            }
            if (lunits == 1) {
                if (v1 < 0.0) {
                   v1 = 0.0 ;
                   o1.setText(String.valueOf(filter0(v1))) ;
                }
                if (v1 >30000.0) {
                   v1 = 30000.0 ;
                   o1.setText(String.valueOf(filter0(v1))) ;
                }
                if (v3 < 0.0) {
                   v3 = 0.0 ;
                   o3.setText(String.valueOf(filter0(v3))) ;
                }
                if (v3 >(2235.0 * 3.6)) {
                   v3 = (2235.0 * 3.6) ;
                   o3.setText(String.valueOf(filter0(v3))) ;
                }
                alt = v1 / .3048 ;
                vel = v3 / .3048 / 3.6 ;
            }

            mode = 1 ;
            locate.y = 300 - (int) (alt / 400.) ;
            ygrph = locate.y ;
            sldloc = 300 - (int) (vel * 60.0 / 88.0 / 20.) ;
            computeAtmos() ;
        }
     }

     class Dn extends Panel {
        Atmos outerparent ;
        Grf grf ;
        Dat dat ;

        Dn (Atmos target) {

           outerparent = target ;
           layout = new CardLayout() ;
           setLayout(layout) ;

           grf = new Grf(outerparent) ;
           dat = new Dat(outerparent) ;

           add ("first", grf) ;
           add ("second", dat) ;

        }

        class Dat extends Panel {
           Atmos outerparent ;
           TextField o1, o2, o3, o4 ;
           TextField o5, o6, o7 ;
   
           Dat (Atmos target) {
              outerparent = target ;
              setLayout(new GridLayout(7,3,2,2)) ;

              o1 = new TextField() ;
              o1.setBackground(Color.black) ;
              o1.setForeground(Color.yellow) ;
              o2 = new TextField() ;
              o2.setBackground(Color.black) ;
              o2.setForeground(Color.yellow) ;
              o3 = new TextField() ;
              o3.setBackground(Color.black) ;
              o3.setForeground(Color.yellow) ;
              o4 = new TextField() ;
              o4.setBackground(Color.black) ;
              o4.setForeground(Color.yellow) ;
              o5 = new TextField() ;
              o5.setBackground(Color.black) ;
              o5.setForeground(Color.yellow) ;
              o6 = new TextField() ;
              o6.setBackground(Color.black) ;
              o6.setForeground(Color.yellow) ;
              o7 = new TextField() ;
              o7.setBackground(Color.black) ;
              o7.setForeground(Color.yellow) ;

              add(new Label("Static Temp ", Label.RIGHT)) ;
              add(o1) ;
              add(new Label(" ", Label.RIGHT)) ;

              add(new Label("Static Press ", Label.RIGHT)) ;
              add(o2) ;
              add(new Label(" ", Label.RIGHT)) ;

              add(new Label("Density", Label.RIGHT)) ;
              add(o3) ;
              add(new Label(" ", Label.RIGHT)) ;

              add(new Label("Sound Speed ", Label.RIGHT)) ;
              add(o4) ;
              add(new Label(" ", Label.RIGHT)) ;

              add(new Label("Dynamic Press ", Label.RIGHT)) ;
              add(o5) ;
              add(new Label(" ", Label.RIGHT)) ;

              add(new Label("Force Ratio ", Label.RIGHT)) ;
              add(o6) ;
              add(new Label(" ", Label.RIGHT)) ;

              add(new Label("Total Temp ", Label.RIGHT)) ;
              add(o7) ;
              add(new Label(" ", Label.RIGHT)) ;
           }
        }

        class Grf extends Canvas {
           Atmos outerparent ;

           Grf (Atmos target) {
              setBackground(Color.black) ;
           }

           public void update(Graphics g) {
              grf.paint(g) ;
           }
 
           public Insets insets() {
              return new Insets(5,5,5,5) ;
           }

           public void paint(Graphics g) {
              int ex[] = { 220,210,210 } ;
              int ey[] = { 50,30,50 } ;
              int index ;
   
              off2Gg.setColor(Color.white) ;
              off2Gg.fillRect(0,0,350,200) ;
               // Thermometer
   
              off2Gg.setColor(Color.black) ;
              off2Gg.drawRect(60,10,20,125) ;
              off2Gg.setColor(Color.white) ;
              off2Gg.fillRect(62,19,17,105) ;
              off2Gg.setColor(Color.red) ;
              off2Gg.fillOval(54,117,35,35);
              off2Gg.drawString("Atmospheric",85,15);
              off2Gg.drawString("Temperature",85,35);
              off2Gg.setColor(Color.black);
              off2Gg.drawString("100",25,25);
              off2Gg.drawString("50",25,50);
              off2Gg.drawString("0",25,75);
              off2Gg.drawString("-50",25,100);
              off2Gg.drawString("-100",25,125);
              for (index=0 ; index <= 4; index ++) {
                off2Gg.drawLine(50,20+index*25,60,20+index*25) ;
              }
              off2Gg.setColor(Color.red) ;
              off2Gg.fillRect(62,ther,17,135-ther) ;
              off2Gg.setColor(Color.black) ;
              if(lunits == 0) 
                 off2Gg.drawString("F",85,75);
              if(lunits == 1) 
                 off2Gg.drawString("C",85,75);
                  // Pressure Gage
              ey[0] = prs ;
              ey[1] = prs - 5;
              ey[2] = prs + 5;
              off2Gg.setColor(Color.black) ;
              off2Gg.fillRect(185,0,90,160) ;
              off2Gg.drawString("Pressure",130,150);
              off2Gg.setColor(Color.yellow) ;
              off2Gg.fillRect(194,prs-2,20,5) ;
              off2Gg.fillPolygon(ex,ey,3) ;
              for (index=0 ; index <= 15; index ++) {
                if (index/5 * 5 == index) {
                    off2Gg.drawLine(215,2+index*10,245,2+index*10) ;
                }
                else {
                    off2Gg.drawLine(220,2+index*10,240,2+index*10) ;
                }
              }
              if(lunits == 0) {
                 off2Gg.setColor(Color.yellow) ;
                 off2Gg.drawString("15",247,10);
                 off2Gg.drawString("10",247,57);
                 off2Gg.drawString("5",247,107);
                 off2Gg.drawString("0",247,152);
                 off2Gg.setColor(Color.black) ;
                 off2Gg.drawString("psi",277,75);
              }
              if(lunits == 1) {
                 off2Gg.setColor(Color.yellow) ;
                 off2Gg.drawString("150",247,10);
                 off2Gg.drawString("100",247,57);
                 off2Gg.drawString("50",247,107);
                 off2Gg.drawString("0",247,152);
                 off2Gg.setColor(Color.black) ;
                 off2Gg.drawString("K-Pa",277,75);
              }
   
              g.drawImage(offImg2,0,0,this) ;
           } // end paint

        }  // end grf
     } // end dn
  } // end  in

  class Viewer extends Panel {
     Atmos outerparent ;

     Viewer (Atmos target) {
         setBackground(Color.blue) ;
     }
 
     public boolean mouseUp(Event evt, int x, int y) {
        handle(x,y) ;
        return true;
     }

     public boolean mouseDrag(Event evt, int x, int y) {
        handle(x,y) ;
        return true;
     }

     public void handle(int x, int y) {
         // determine location and altitude
        locate = new Point(x,y) ;
        if (x <= 230) {
           if (locate.y > 300) locate.y = 300 ;
           if (locate.y < 50) locate.y = 50 ;
           ygrph = locate.y ;
           alt = 400. * (300 - ygrph) ;
        }
        if ( x >= 231) {
           if (locate.y > 300) locate.y = 300 ;
           if (locate.y < 50) locate.y = 50 ;
           sldloc = locate.y ;
           vel = (20. * (300 - sldloc)) * 88.0 / 60.0 ;
        }

        mode = 0; 
        computeAtmos() ;
     }

     public void update(Graphics g) {
         view.paint(g) ;
     }

     public void paint(Graphics g) {
      int iwidth = plnimg.getWidth(this) ;
      int iheight = plnimg.getHeight(this) ;
      int index;

      Font f = new Font("Helvetica",Font.PLAIN,18);
      offsGg.setFont(f) ;
        // Airplane on Background
      if (planet == 0) {
         offsGg.setColor(Color.blue) ;
         offsGg.fillRect(0,0,350,350) ;
         offsGg.drawImage(plnimg,110,ygrph-16,iwidth,iheight,this) ;
      }
      if (planet == 1) {
         offsGg.setColor(Color.orange) ;
         offsGg.fillRect(0,0,350,350) ;
         offsGg.drawImage(mplnimg,110,ygrph-16,iwidth,iheight,this) ;
      }
  // Altitude  Axis
      offsGg.setColor(Color.white) ;
      offsGg.drawLine(100,50,100,300) ;
      offsGg.drawLine(102,50,102,300) ;
      offsGg.drawString("Move Aircraft for",10,15);

      if (lunits == 0) {     // English units
        offsGg.drawString("Altitude (feet)",10,40);
        offsGg.drawString("100,000",10,60);
        offsGg.drawString("80,000",10,105);
        offsGg.drawString("60,000",10,155);
        offsGg.drawString("40,000",10,205);
        offsGg.drawString("20,000",10,255);
        offsGg.drawString("0",10,300);
        for (index=0 ; index <= 5; index ++) {
          offsGg.drawLine(90,50+index*50,110,50+index*50) ;
        }
      }

      if (lunits == 1) {   // metric units
        offsGg.drawString("Altitude (meters)",10,40);
        offsGg.drawString("30,000",10,59);
        offsGg.drawString("25,000",10,100);
        offsGg.drawString("20,000",10,141);
        offsGg.drawString("15,000",10,182);
        offsGg.drawString("10,000",10,219);
        offsGg.drawString("5,000",10,260);
        offsGg.drawString("0",10,300);
        for (index=0 ; index <= 7; index ++) {
          offsGg.drawLine(90,54+index*41,110,54+index*41) ;
        }
      }

  // Velocity  Axis
      offsGg.setColor(Color.white) ;
      offsGg.drawLine(250,50,250,300) ;
      offsGg.drawLine(252,50,252,300) ;
      offsGg.drawString("Slider for",230,15);

      if (lunits == 0) {     // English units
        offsGg.drawString("Velocity (mph)",230,40);
        offsGg.drawString("5,000",272,60);
        offsGg.drawString("4,000",272,105);
        offsGg.drawString("3,000",272,155);
        offsGg.drawString("2,000",272,205);
        offsGg.drawString("1,000",272,255);
        offsGg.drawString("0",272,300);
        for (index=0 ; index <= 5; index ++) {
          offsGg.drawLine(240,50+index*50,260,50+index*50) ;
        }
      }
      if (lunits == 1) {     // Metric units
        offsGg.drawString("Velocity (kmh)",230,40);
        offsGg.drawString("7,000",272,86);
        offsGg.drawString("6,000",272,117);
        offsGg.drawString("5,000",272,148);
        offsGg.drawString("4,000",272,179);
        offsGg.drawString("3,000",272,210);
        offsGg.drawString("2,000",272,241);
        offsGg.drawString("1,000",272,272);
        offsGg.drawString("0",272,300);
        for (index=0 ; index <= 7; index ++) {
          offsGg.drawLine(240,300-index*31,260,300-index*31) ;
        }
      }

      offsGg.setColor(Color.yellow) ;
      offsGg.fillRect(235,sldloc-2,30,5) ;

      g.drawImage(offscreenImg,0,0,this) ;
    }
  }
}
