/*  
                      Normal Shock Calculator
                     Interactive Program to solve 
                    Standard Atmosphere Equations
                     and Normal shock relations 
                  Including  Thermally Perfect Gas

                          A Java Applet

                     Version 1.2a   - 20 July 06

                            Written by 
 
                             Tom Benson
                            Chuck Trefny
                            Charles Lamp
                       NASA Glenn Research Center

>                              NOTICE
>This software is in the Public Domain.  It may be freely copied and used in
>non-commercial products, assuming proper credit to the author is given.  IT
>MAY NOT BE RESOLD.  If you want to use the software for commercial
>products, contact the author.
>No copyright is claimed in the United States under Title 17, U. S. Code.
>This software is provided "as is" without any warranty of any kind, either
>express, implied, or statutory, including, but not limited to, any warranty
>that the software will conform to specifications, any implied warranties of
>merchantability, fitness for a particular purpose, and freedom from
>infringement, and any warranty that the documentation will conform to the
>program, or any warranty that the software will be error free.
>In no event shall NASA be liable for any damages, including, but not
>limited to direct, indirect, special or consequential damages, arising out
>of, resulting from, or in any way connected with this software, whether or
>not based on warranty, contract, tort or otherwise, whether or not injury
>was sustained by persons or property or otherwise, and whether or not loss
>was sustained from, or arose out of the results of, or use of, the software
>or services provided hereunder.

      New Test :
                 * use input panel from Mach calculator
                 * add calculation routine from Shock simulator
                 * add special output panel 
                 * add calorically imperfect gas

                                                     TJB 20 July 06

*/

import java.awt.*;
import java.lang.Math ;

public class Normal extends java.applet.Applet {

   double gama,alt, rgas, a0, q0, vel ;
   double ps0, pt0, rho0, ts0, tt0, mach ;
   double prat, ptrat, rhorat, trat, ttrat, mrat ;
   double ps1, pt1, rho1, ts1, tt1, mach1;
   double G, M, Q, TT, T, TR, P, PT, PR, DR, M2, PTR;
   double TTR, MR, P2, PT2, T2, TT2, D2;
   double prati, ptrati, trati, ttrati, rhorati, machrati;
   int lunits, planet, itype ;
   double typeinpt, setit;
   int mode, Try ;

   In in ;

   public void init() {

     setLayout(new GridLayout(1,1,0,0)) ;

     setDefaults () ;

     in = new In(this) ;

     add(in) ;

     computeNormal() ;
  }
 
  public Insets insets() {
     return new Insets(10,10,10,10) ;
  }

  public void setDefaults() {
     lunits = 0;
     mode = 0 ;
     planet = 0 ;

     alt = 0.0 ;
     mach =  M = 2. ;
     gama = G = 1.4 ;
     Q = 5500.;
     itype = 0;
     typeinpt = 0;
  }

	double TQTT(double M, double G)
	{
		TTR = Math.pow((1 + (G - 1) / 2 * Math.pow(M, 2)), -1);
		return TTR;
	}

	double CAL_GAM (double T, double G, double Q)
	{
		return (1 + (G - 1) / (1 + (G - 1) * (Math.pow((Q / T), 2) * Math.exp(Q / T) / Math.pow((Math.exp(Q / T) - 1), 2))));
	}

	double CAL_TT (double T, double M, double G, double Q)
	{
		double TT = T / TQTT(M,G);
		double EPS = 0.00001;
		double Z = 1;
		double Z2 = 0;
		double TT2 = 0;
		double TT1;
		double Z1;
		double i = 1;
		while (Math.abs(Z) > EPS)
		{
			Z = Math.pow(M, 2) - 2 * TT / CAL_GAM(T, G, Q) / T * (G / (G - 1) * (1 - T / TT) + Q / TT * (1 / (Math.exp(Q / TT) - 1) - 1 / (Math.exp(Q / T) - 1)));
			if (i == 1)
			{
				Z2=Z;
				TT2 = TT;
				TT = TT2 + 100;
				i = 2;
			}
			else
			{
				TT1 = TT2;
				Z1 = Z2;
				TT2 = TT;
				Z2 = Z;
				TT = TT2 - Z2 * (TT2 - TT1) / (Z2 - Z1);
			}
		}
		return TT;
	}

	double CAL_T (double TT, double M, double G, double Q)
	{
		double T = TT * TQTT(M,G);
		double EPS = 0.00001;
		double Z=1;
		double Z2 = 0;
		double T2 = 0;
		double T1;
		double Z1;
		double i=1;
		while (Math.abs(Z) > EPS)
		{
			Z = Math.pow(M, 2) - 2 * TT / CAL_GAM(T, G, Q) / T * (G / (G - 1) * (1 - T / TT) + Q / TT * (1 / (Math.exp(Q / TT) - 1) - 1 / (Math.exp(Q / T) - 1)));
			if (i==1)
			{
				Z2 = Z;
				T2 = T;
				T = T2 + 100;
				i=2;
			}
			else 
			{
				T1 = T2;
				Z1 = Z2;
				T2 = T;
				Z2 = Z;
				T = T2 - Z2 * (T2 - T1) / (Z2 - Z1);
			}
		}
		return T;
	}

	double CAL_PQPT (double TT, double M, double G, double Q)
	{
		double T = CAL_T(TT, M, G, Q);
		return (((Math.exp(Q / TT) - 1) / (Math.exp(Q / T) - 1)) * Math.pow((T / TT), (G / (G - 1))) * Math.exp(Q / T * Math.exp(Q / T) / (Math.exp(Q / T) - 1) - Q / TT * Math.exp(Q / TT) / (Math.exp(Q / TT) - 1)));
	}

	double CAL_TNS (double TT, double M, double G, double Q)
	{
		double T = CAL_T(TT, M, G, Q);
		double TR = (2 * G * Math.pow(M, 2) - (G - 1)) * ((G - 1) * Math.pow(M, 2) + 2) / Math.pow((G + 1), 2) / Math.pow(M, 2);
		double EPS = 0.00001;
		double Z = 1;
		double Z2 = 0;
		double TR2 = 0;
		double TR1;
		double Z1;
		int i=0;
		while (Math.abs(Z) > EPS)
		{
			if (i==0)
			{
				Z2=Z;
				TR2=TR;
				TR=0.98*TR2;
				i=1;
			}
			else
			{
				TR1=TR2;
				Z1=Z2;
				TR2=TR;
				Z2=Z;
				TR=TR2 - Z2 * (TR2 - TR1) / (Z2 - Z1);
			}
			Z = Math.pow((Math.pow((1 + 1 / CAL_GAM(T, G, Q) / Math.pow(M, 2)), 2) - 2 / CAL_GAM(T, G, Q) / Math.pow(M, 2) * TR - 2 + G / (G - 1) * (TR - 1) * 4 / CAL_GAM(T, G, Q) / Math.pow(M, 2) + 4 / CAL_GAM(T, G, Q) / Math.pow(M, 2) * Q / T * (1 / (Math.exp(Q / T / TR) - 1) - 1 / (Math.exp(Q / T) - 1))), 2) - (Math.pow((1 + 1 / CAL_GAM(T, G, Q) / Math.pow(M, 2)), 2) * (Math.pow((1 + 1 / CAL_GAM(T, G, Q) / Math.pow(M, 2)), 2) - 4 / CAL_GAM(T, G, Q) / Math.pow(M, 2) * TR));
		}
		return TR;
	}

	double CAL_DNS (double TT, double M, double G, double Q)
	{
		double T = CAL_T(TT, M, G, Q);
		double VRAT2 = 1 - 2 * G / (G - 1) / CAL_GAM(T, G, Q) / Math.pow(M, 2) * (CAL_TNS(TT, M, G, Q) - 1) - 2 * (Q / T) / CAL_GAM(T, G, Q) / Math.pow(M, 2) * (1 / (Math.exp(Q / T / CAL_TNS(TT, M, G, Q)) - 1) - 1 / (Math.exp(Q / T) - 1));
		return (1 / Math.sqrt(VRAT2));
	}

	double CAL_PNS (double TT, double M, double G, double Q)
	{
		return (CAL_DNS(TT, M, G, Q) * CAL_TNS(TT, M, G, Q));
	}

	double CAL_MNS (double TT, double M, double G, double Q)
	{
		double T = CAL_T(TT, M, G, Q);
		double T2 = T * CAL_TNS(TT, M, G, Q);
		double VRAT2 = 1 - 2 * G / (G - 1) / CAL_GAM(T, G, Q) / Math.pow(M, 2) * (CAL_TNS(TT, M, G, Q) - 1) - 2 * (Q / T) / CAL_GAM(T, G, Q) / Math.pow(M, 2) * (1 / (Math.exp(Q / T / CAL_TNS(TT, M, G, Q)) - 1) - 1 / (Math.exp(Q / T) - 1));
		return (M * Math.sqrt(VRAT2) * Math.sqrt(T / T2 * CAL_GAM(T, G, Q) / CAL_GAM(T2, G, Q)));
	}

	double CAL_PTNS (double TT, double M, double G, double Q)
	{
		return (CAL_PQPT(TT, M, G, Q) / CAL_PQPT(TT, CAL_MNS(TT, M, G, Q), G, Q) * CAL_PNS(TT, M, G, Q));
	}


  public void computeNormal() 
  {
     double gm1,gp1,msq,m1sq ;

  // get free stream conditions
		Try = itype;
	  if (itype == 0)
	  {
		  alt = typeinpt;
		  if (planet == 0) 
		  {    // Earth  standard day
			  rgas = 1718. ;                /* ft2/sec2 R */

			  if (alt <= 36152.) 
			  {           // Troposphere
				  T = ts0 = 518.6 - 3.56 * alt/1000. ;
				  ps0 = 2116. * Math.pow(ts0/518.688,5.256) ;
			  }
			  if (alt >= 36152. && alt <= 82345.) 
			  {   // Stratosphere
				  T = ts0 = 389.98 ;
				  ps0 = 2116. * .2236 *
					  Math.exp((36152.-alt)/(53.35*389.98)) ;
			  }
			  if (alt >= 82345. && alt <= 155348.) 
			  {          
				  T = ts0 = 389.98 + 1.645 * (alt-82345.)/1000. ;
				  ps0 = 2116. *.02456 * Math.pow(ts0/389.98,-11.388) ;
			  }
			  if (alt >= 155348. && alt <= 175346.) 
			  {          
				  T = ts0 = 508.788 ;
				  ps0 = 2116. * .00118866 *
					  Math.exp((155348.-alt)/(53.35*508.788)) ;
			  }
			  if (alt >= 175346. && alt <= 262448.) 
			  {          
				  T = ts0 = 508.788 - 2.46888 * (alt-175346.)/1000. ;
				  ps0 = 2116. *.00057596 * Math.pow(ts0/508.788,7.59216) ;
			  }
		  }

		  if (planet == 1) 
		  {   // Mars - curve fit of orbiter data
			  rgas = 1149. ;                /* ft2/sec2 R */

			  if (alt <= 22960.) 
			  {
				  T = ts0 = 434.02 - .548 * alt/1000. ;
				  ps0 = 14.62 * Math.pow(2.71828,-.00003 * alt) ;
			  }
			  if (alt > 22960.) 
			  {
				  T = ts0 = 449.36 - 1.217 * alt/1000. ;
				  ps0 = 14.62 * Math.pow(2.71828,-.00003 * alt) ;
			  }
		  }
		  TT = CAL_TT (T, M, G, Q);
		  tt0 = 1 / TQTT(M, G) * ts0;
	  }
	  if (itype == 1)
	  {
		  TT = tt0 = typeinpt;
		  T = CAL_T (TT, M, G, Q);
		  ts0 = TQTT(M, G) * TT;
	  }

     gm1 = gama - 1.0 ;
     gp1 = gama + 1.0 ;

     rho0 = ps0 / rgas / ts0 ;
     a0 = Math.sqrt(gama*rgas*ts0) ;  // feet /sec
     a0 = a0 * 60.0 / 88. ;   // mph
                       // compute either mach or velocity 
     msq = mach * mach ;

     q0 = gama / 2.0 * msq * ps0 ;
     pt0 = ps0 * Math.pow((1.0 + .5 * gm1 * msq), (gama/gm1)) ;

     prat   = (2.0*gama*msq - gm1)/gp1 ;
     rhorat = (gp1*msq)/(gm1*msq + 2.0) ;
     trat   = prat / rhorat ;
     ttrat  = 1.0 ;
     ptrat  = (Math.pow(rhorat,(gama/gm1)))
               * (Math.pow((1.0/prat),(1.0/gm1))) ;
     m1sq   = msq / (prat * rhorat) ;
     mach1  = Math.sqrt(m1sq) ;

     ps1 = prat * ps0 ;
     pt1 = ptrat * pt0 ;
     ts1 = trat * ts0 ;
     tt1 = ttrat * tt0 ;
     rho1 = rhorat * rho0 ;
     mrat = mach1 / mach ;
     
// Calorically Imperfect Calculation
	  P = ps0;
	  PT = P/CAL_PQPT (TT, M, G, Q);
	  TR = CAL_TNS (TT, M, G, Q);
	  PR = CAL_PNS (TT, M, G, Q);
	  PTR = CAL_PTNS (TT, M, G, Q);
	  DR = CAL_DNS (TT, M, G, Q);
	  M2 = CAL_MNS (TT, M, G, Q);
      MR = M2 / M;  
	  TT2 = TT;
	  P2 = P * PR;
	  D2 = rho0 * DR;
	  T2 = T * TR;
	  PT2 = PT * PTR;

// Ratio of Calorically Imperfect to Perfect
      prati = P2 / ps1;
	  ptrati = PT2 / pt1;
	  trati = T2 / ts1;
	  ttrati = TT2 / tt1;
      machrati = M2 / mach1;
	  rhorati = D2 / rho1;



     if (lunits == 0) {
        in.dni.o1.setText(String.valueOf(filter3(ps0))) ;
        in.dni.o2.setText(String.valueOf(filter3(prat))) ;
        in.dni.o3.setText(String.valueOf(filter3(ps1))) ;
        in.dni.o4.setText(String.valueOf(filter3(pt0))) ;
        in.dni.o5.setText(String.valueOf(filter5(ptrat))) ;
        in.dni.o6.setText(String.valueOf(filter3(pt1))) ;
        in.dni.o7.setText(String.valueOf(filter0(ts0))) ;
        in.dni.o8.setText(String.valueOf(filter3(trat))) ;
        in.dni.o9.setText(String.valueOf(filter0(ts1))) ;
		in.dni.o16.setText(String.valueOf(filter0(tt0))) ;
		in.dni.o17.setText(String.valueOf(filter3(ttrat))) ;
    	in.dni.o18.setText(String.valueOf(filter0(tt1))) ;
        in.dni.o10.setText(String.valueOf(filter9(rho0))) ;
        in.dni.o11.setText(String.valueOf(filter3(rhorat))) ;
        in.dni.o12.setText(String.valueOf(filter9(rho1))) ;
        in.dni.o13.setText(String.valueOf(filter5(mach))) ;
        in.dni.o14.setText(String.valueOf(filter3(mrat))) ;
        in.dni.o15.setText(String.valueOf(filter5(mach1))) ;

	    in.dntp.o1.setText(String.valueOf(filter3(P))) ;
	    in.dntp.o2.setText(String.valueOf(filter3(PR))) ;
	    in.dntp.o3.setText(String.valueOf(filter3(P2))) ;
	    in.dntp.o4.setText(String.valueOf(filter3(PT))); 
		in.dntp.o5.setText(String.valueOf(filter5(PTR))) ;
	    in.dntp.o6.setText(String.valueOf(filter3(PT2))) ;
	    in.dntp.o7.setText(String.valueOf(filter0(T))) ;
    	in.dntp.o8.setText(String.valueOf(filter3(TR))) ;
	    in.dntp.o9.setText(String.valueOf(filter0(T2))) ;
	    in.dntp.o16.setText(String.valueOf(filter0(TT))) ;
	    in.dntp.o17.setText(String.valueOf(filter3(ttrat))) ;
	    in.dntp.o18.setText(String.valueOf(filter0(TT2))) ;
	    in.dntp.o10.setText(String.valueOf(filter9(rho0))) ;
	    in.dntp.o11.setText(String.valueOf(filter3(DR))) ;
	    in.dntp.o12.setText(String.valueOf(filter9(D2))) ;
	    in.dntp.o13.setText(String.valueOf(filter5(M))) ;
	    in.dntp.o14.setText(String.valueOf(filter3(MR))) ;
	    in.dntp.o15.setText(String.valueOf(filter5(M2))) ;
	    
     }
     if (lunits == 1) {
        in.dni.o1.setText(String.valueOf(filter5(ps0* 4.448/.3048/.3048/1000.))) ;
        in.dni.o2.setText(String.valueOf(filter3(prat))) ;
        in.dni.o3.setText(String.valueOf(filter5(ps1* 4.448/.3048/.3048/1000.))) ;
        in.dni.o4.setText(String.valueOf(filter5(pt0* 4.448/.3048/.3048/1000.))) ;
        in.dni.o5.setText(String.valueOf(filter5(ptrat))) ;
        in.dni.o6.setText(String.valueOf(filter5(pt1* 4.448/.3048/.3048/1000.))) ;
        in.dni.o7.setText(String.valueOf(filter0(ts0 * .55555))) ;
        in.dni.o8.setText(String.valueOf(filter3(trat))) ;
        in.dni.o9.setText(String.valueOf(filter0(ts1 * .55555))) ;
		in.dni.o16.setText(String.valueOf(filter0(tt0 * .55555))) ;
		in.dni.o17.setText(String.valueOf(filter3(ttrat))) ;
		in.dni.o18.setText(String.valueOf(filter0(tt1 * .55555))) ;
        in.dni.o10.setText(String.valueOf(filter9(rho0 * 515.4))) ;
        in.dni.o11.setText(String.valueOf(filter3(rhorat))) ;
        in.dni.o12.setText(String.valueOf(filter5(rho1 * 515.4))) ;
        in.dni.o13.setText(String.valueOf(filter5(mach))) ;
        in.dni.o14.setText(String.valueOf(filter3(mrat))) ;
        in.dni.o15.setText(String.valueOf(filter5(mach1))) ;

	    in.dntp.o1.setText(String.valueOf(filter5(P* 4.448/.3048/.3048/1000.))) ;
		in.dntp.o2.setText(String.valueOf(filter3(PR))) ;
	    in.dntp.o3.setText(String.valueOf(filter5(P2* 4.448/.3048/.3048/1000.))) ;
        in.dntp.o4.setText(String.valueOf(filter5(PT* 4.448/.3048/.3048/1000.))) ;
		in.dntp.o5.setText(String.valueOf(filter5(PTR))) ;
	    in.dntp.o6.setText(String.valueOf(filter5(PT2* 4.448/.3048/.3048/1000.))) ;
		in.dntp.o7.setText(String.valueOf(filter0(T * .555555))) ;
		in.dntp.o8.setText(String.valueOf(filter3(TR))) ;
	    in.dntp.o9.setText(String.valueOf(filter0(T2 * .55555))) ;
	    in.dntp.o16.setText(String.valueOf(filter0(TT * .555555))) ;
	    in.dntp.o17.setText(String.valueOf(filter3(ttrat))) ;
		in.dntp.o18.setText(String.valueOf(filter0(TT2 * .55555))) ;
		in.dntp.o10.setText(String.valueOf(filter9(rho0 * 515.4))) ;
		in.dntp.o11.setText(String.valueOf(filter3(DR))) ;
	    in.dntp.o12.setText(String.valueOf(filter5(D2 * 515.4))) ;
	    in.dntp.o13.setText(String.valueOf(filter5(M))) ;
		in.dntp.o14.setText(String.valueOf(filter3(MR))) ;
		in.dntp.o15.setText(String.valueOf(filter5(M2))) ;

     }
     in.dntp.or1.setText(String.valueOf(filter3(prati))) ;
	 in.dntp.or2.setText(String.valueOf(filter3(ptrati))) ;
	 in.dntp.or3.setText(String.valueOf(filter3(trati))) ;
	 in.dntp.or4.setText(String.valueOf(filter3(ttrati))) ;
	 in.dntp.or5.setText(String.valueOf(filter3(rhorati))) ;
	 in.dntp.or6.setText(String.valueOf(filter3(machrati))) ;
     if (mode == 0) loadInpt() ;
  }
 
  public void loadInpt() {
     if (lunits == 0) {
         in.up.o1.setText(String.valueOf(filter0(alt))) ;
         in.up.o2.setText(String.valueOf(filter3(mach))) ;
         in.up.o3.setText(String.valueOf(filter3(gama))) ;
     }
     if (lunits == 1) {
         in.up.o1.setText(String.valueOf(filter0(alt*.3048))) ;
         in.up.o2.setText(String.valueOf(filter3(mach))) ;
         in.up.o3.setText(String.valueOf(filter3(gama))) ;
     }
  }

  public int filter0(double inumbr) {
     //  integer output
       float number ;
       int intermed ;

       intermed = (int) (inumbr) ;
       number = (float) (intermed);
       return intermed ;
  }
 
  public float filter3(double inumbr) {
     //  output only to .001
       float number ;
       int intermed ;
  
       intermed = (int) (inumbr * 1000.) ;
       number = (float) (intermed / 1000. );
       return number ;
  }

  public float filter5(double inumbr) {
     //  output only to .00001
       float number ;
       int intermed ;
  
       intermed = (int) (inumbr * 100000.) ;
       number = (float) (intermed / 100000. );
       return number ;
  }

  public float filter9(double inumbr) {
     //  output only to .000000001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 1000000000.) ;
       number = (float) (intermed / 1000000000. );
       return number ;
  }

  class In extends Panel {
     Normal outerparent ;
     Titl titl ;
     Up up ;
     Dni dni ;
	 Dntp dntp ;

     In (Normal target) {                           
        outerparent = target ;
        setLayout(new GridLayout(4,1,5,5)) ;

        titl = new Titl(outerparent) ;
        up = new Up(outerparent) ;
        dni = new Dni(outerparent) ;
	    dntp = new Dntp(outerparent) ;

        add(titl) ;
        add(up) ;
        add(dni) ;
	    add(dntp) ;
     }

     public Insets insets() {
        return new Insets(5,5,0,0) ;
     }

     class Titl extends Panel {
        Label la ;
        In2 in2;

        Titl (Normal target) {                           
            outerparent = target ;
            setLayout(new GridLayout(2,1,0,0)) ;

            la = new Label("Normal Shock Calculator", Label.CENTER) ;
            la.setForeground(Color.red) ;

            in2 = new In2(outerparent) ;
 
            add(la) ;
            add(in2) ;
        }
 
        class In2 extends Panel {
           Label lc,lb ;
           Choice plntch,untch ;

           In2 (Normal target) {                           
               outerparent = target ;
               setLayout(new GridLayout(2,4,0,0)) ;

               lb = new Label("Units:", Label.RIGHT) ;
               lb.setForeground(Color.red) ;
 
               lc = new Label("Planet:", Label.RIGHT) ;
               lc.setForeground(Color.red) ;
 
               plntch = new Choice() ;
               plntch.addItem("Earth") ;
               plntch.addItem("Mars");
               plntch.setBackground(Color.white) ;
               plntch.setForeground(Color.red) ;
               plntch.select(0) ;

               untch = new Choice() ;
               untch.addItem("Imperial") ;
               untch.addItem("Metric");
               untch.setBackground(Color.white) ;
               untch.setForeground(Color.red) ;
               untch.select(0) ;

               add(new Label(" ", Label.RIGHT)) ;
               add(lb) ;
               add(untch) ;
               add(new Label(" ", Label.RIGHT)) ;

               add(new Label(" ", Label.RIGHT)) ;
             //  add(lc) ;
             //  add(plntch) ;
               add(new Label(" ", Label.RIGHT)) ;
			   add(new Label(" ", Label.RIGHT)) ;
			   add(new Label(" ", Label.RIGHT)) ;
           }

           public boolean handleEvent(Event evt) {
               if(evt.id == Event.ACTION_EVENT) {
                 // planet  = plntch.getSelectedIndex() ;
                  lunits  = untch.getSelectedIndex() ;
			      itype = up.type.getSelectedIndex();

                  if (planet == 0) {  // Earth
                     gama = 1.4 ;
                     up.o3.setText(String.valueOf(filter3(gama))) ;
                  }
				   if (planet == 1) 
				   {  // Mars
					   gama = 1.29 ;
					   up.o3.setText(String.valueOf(filter3(gama))) ;
				   }

				   double setit;
				   if (lunits == 0) {  // English units labels
					  if (itype == 0)
					  {
						  setit = 0;
						  typeinpt = setit;
						  up.o1.setText(String.valueOf(setit));
						  up.l1u.setText("feet") ;
					  }
					  if (itype == 1)
					  {
						  setit = 500;
						  typeinpt = setit;
						  up.o1.setText(String.valueOf(setit));
						  up.l1u.setText("Rankin") ;
					  }
                      dni.l1.setText("Static Pressure-psf") ;
                      dni.l2.setText("Total Pressure-psf") ;
                      dni.l3.setText("Static Temperature-R") ;
					  dni.l6.setText("Total Temperature-R") ;
                      dni.l4.setText("Density-slug/cu ft") ;
					  dntp.l1.setText("Static Pressure-psf") ;
					  dntp.l2.setText("Total Pressure-psf") ;
					  dntp.l3.setText("Static Temperature-R") ;
					  dntp.l6.setText("Total Temperature-R") ;
					  dntp.l4.setText("Density-slug/cu ft") ;
                  }
				   if (lunits == 1) 
				   {  // Metric units labels
					   if (itype == 0)
					   {
						   setit = 0;
						   typeinpt = setit;
						   up.o1.setText(String.valueOf(setit));
						   up.l1u.setText("meters") ;
					   }
					   if (itype == 1)
					   {
						   setit = 278;
						   typeinpt = setit / .55555;
						   up.o1.setText(String.valueOf(setit));
						   up.l1u.setText("Kelvin") ;
					   }
					   dni.l1.setText("Static Pressure-kPa") ;
					   dni.l2.setText("Total Pressure-kPa") ;
					   dni.l3.setText("Static Temperature-K") ;
					   dni.l6.setText("Total Temperature-K") ;
					   dni.l4.setText("Density-kg/cu m") ;
					   dntp.l1.setText("Static Pressure-kPa") ;
					   dntp.l2.setText("Total Pressure-kPa") ;
					   dntp.l3.setText("Static Temperature-K") ;
					   dntp.l6.setText("Total Temperature-K") ;
					   dntp.l4.setText("Density-kg/cu m") ;
				   }
 
                  mode = 1 ;
                  computeNormal() ;
                  return true ;
               }
               else return false ;
           }
        }
     }

     class Up extends Panel {
        TextField o1,o2,o3 ;
        Label l1,l1u,l2u, la ;
        Button bt1;
	    Choice type;

        Up (Normal target) {                           
            outerparent = target ;
            setLayout(new GridLayout(4,4,5,5)) ;
    
            la = new Label("Input", Label.LEFT) ;
            la.setForeground(Color.red) ;

            l1 = new Label("Altitude", Label.RIGHT) ;
            l1u = new Label(" feet ", Label.LEFT) ;
   
			o1 = new TextField() ;
            o1.setBackground(Color.white) ;
            o1.setForeground(Color.black) ;

            o2 = new TextField() ;
            o2.setBackground(Color.white) ;
            o2.setForeground(Color.black) ;
 
            o3 = new TextField() ;
            o3.setBackground(Color.white) ;
            o3.setForeground(Color.black) ;
 
            bt1 = new Button("COMPUTE") ;
            bt1.setBackground(Color.red) ;
            bt1.setForeground(Color.white) ;

			type = new Choice();
			type.addItem("Altitude");
			type.addItem("Total Temperature");
			type.select(0);

            add(la) ;
            add(new Label("Mach ", Label.RIGHT)) ;
            add(o2) ;
            add(new Label(" ", Label.RIGHT)) ;

            add(new Label(" ", Label.RIGHT)) ;
            add(new Label("Gamma ", Label.RIGHT)) ;
            add(o3) ;
            add(new Label(" ", Label.RIGHT)) ;

            add(new Label(" ", Label.RIGHT)) ;
            add(type) ;
            add(o1) ;
            add(l1u) ;

            add(new Label(" ", Label.RIGHT)) ;
            add(new Label("Press -> ", Label.RIGHT)) ;
            add(bt1) ;
            add(new Label(" ", Label.RIGHT)) ;
        }

        public Insets insets() {
           return new Insets(5,5,5,5) ;
        }

        public boolean action(Event evt, Object arg) {
            if(evt.target instanceof Button) {
               this.handleBut(evt) ;
               return true ;
            }
			if (evt.id == Event.ACTION_EVENT)
			{
				this.handleBut(evt);
				return true;
			}
			else return false ;
        }
 
        public void handleBut(Event evt) {
            Double V1,V2,V3 ;
            double v1,v2,v3 ;

            V1 = Double.valueOf(o1.getText()) ;
            v1 = V1.doubleValue() ;
            V2 = Double.valueOf(o2.getText()) ;
            v2 = V2.doubleValue() ;
            V3 = Double.valueOf(o3.getText()) ;
            v3 = V3.doubleValue() ;
			itype = type.getSelectedIndex();

			if (lunits == 0) 
			{
				if (itype == 0)
				{
					if (v1 < 0.0) 
					{
						v1 = 0.0 ;
						o1.setText(String.valueOf(filter0(v1))) ;
					}
					if (v1 >250000.0) 
					{
						v1 = 250000.0 ;
						o1.setText(String.valueOf(filter0(v1))) ;
					}
					typeinpt = v1 ;
					l1u.setText("feet") ;
				}
				if (itype == 1)
				{
					if (v1 < 500.0)
					{
						v1 = 500.0;
						o1.setText(String.valueOf(filter0(v1))) ;
					}
					if (v1 > 5000.0)
					{
						v1 = 5000.0;
						o1.setText(String.valueOf(filter0(v1))) ;
					}
					typeinpt = v1;
					l1u.setText("Rankin") ;
				}
			}
			if (lunits == 1) 
			{
				if (itype == 0)
				{
					if (v1 < 0.0) 
					{
						v1 = 0.0 ;
						o1.setText(String.valueOf(filter0(v1))) ;
					}
					if (v1 >76200.0) 
					{
						v1 = 76200.0 ;
						o1.setText(String.valueOf(filter0(v1))) ;
					}
					typeinpt = v1 / .3048 ;
					l1u.setText("meters") ;
				}
				if (itype == 1)
				{
					if (v1 < 278.0)
					{
						v1 = 278.0;
						o1.setText(String.valueOf(filter0(v1))) ;
					}
					if (v1 > 2778.0)
					{
						v1 = 2778.0;
						o1.setText(String.valueOf(filter0(v1))) ;
					}
					typeinpt = v1 / .55555;
					l1u.setText("Kelvin") ;
				}

			}

            if (v2 < 1.0) {
                v2 = 1.0 ;
                o2.setText(String.valueOf(filter0(v2))) ;
            }
            if (v2 >25.0) {
                v2 = 25.0 ;
                o2.setText(String.valueOf(filter0(v2))) ;
            }
            mach = M = v2 ;

            if (v3 < 1.0) {
                v3 = 1.0 ;
                o3.setText(String.valueOf(filter0(v3))) ;
            }
            if (v3 > 2.0) {
                v3 = 2.0 ;
                o3.setText(String.valueOf(filter0(v3))) ;
            }
            gama = G = v3 ;

            mode = 1;
            computeNormal() ;
        }
     }

     class Dni extends Panel {
        Normal outerparent ;
        TextField o1,o2,o3,o4,o5,o6,o7,o8,o9,o10 ;
        TextField o11,o12,o13,o14,o15,o16,o17,o18 ;
        Label l1,l2,l3,l4,l5,l6 ;
        Label lb ;

        Dni (Normal target) {
            outerparent = target ;
            setLayout(new GridLayout(7,5,0,0)) ;
    
            lb = new Label("Output- Perfect Gas", Label.LEFT) ;
            lb.setForeground(Color.blue) ;

            l1 = new Label("Static Pressure-psf", Label.CENTER) ;
            l2 = new Label("Total Pressure-psf", Label.CENTER) ;
            l3 = new Label("Static Temperature-R", Label.CENTER) ;
            l4 = new Label("Density-slug/cu ft", Label.CENTER) ;
            l5 = new Label("Mach", Label.CENTER) ;
            l6 = new Label("Total Temperature-R", Label.CENTER) ;
   
            o1 = new TextField() ;
            o1.setBackground(Color.black) ;
            o1.setForeground(Color.yellow) ;
            o2 = new TextField() ;
            o2.setBackground(Color.black) ;
            o2.setForeground(Color.yellow) ;
            o3 = new TextField() ;
            o3.setBackground(Color.black) ;
            o3.setForeground(Color.yellow) ;
            o4 = new TextField() ;
            o4.setBackground(Color.black) ;
            o4.setForeground(Color.yellow) ;
            o5 = new TextField() ;
            o5.setBackground(Color.black) ;
            o5.setForeground(Color.yellow) ;
            o6 = new TextField() ;
            o6.setBackground(Color.black) ;
            o6.setForeground(Color.yellow) ;
            o7 = new TextField() ;
            o7.setBackground(Color.black) ;
            o7.setForeground(Color.yellow) ;
            o8 = new TextField() ;
            o8.setBackground(Color.black) ;
            o8.setForeground(Color.yellow) ;
            o9 = new TextField() ;
            o9.setBackground(Color.black) ;
            o9.setForeground(Color.yellow) ;
            o10 = new TextField() ;
            o10.setBackground(Color.black) ;
            o10.setForeground(Color.yellow) ;
            o11 = new TextField() ;
            o11.setBackground(Color.black) ;
            o11.setForeground(Color.yellow) ;
            o12 = new TextField() ;
            o12.setBackground(Color.black) ;
            o12.setForeground(Color.yellow) ;
            o13 = new TextField() ;
            o13.setBackground(Color.black) ;
            o13.setForeground(Color.yellow) ;
            o14 = new TextField() ;
            o14.setBackground(Color.black) ;
            o14.setForeground(Color.yellow) ;
            o15 = new TextField() ;
            o15.setBackground(Color.black) ;
            o15.setForeground(Color.yellow) ;
            o16 = new TextField() ;
            o16.setBackground(Color.black) ;
            o16.setForeground(Color.yellow) ;
            o17 = new TextField() ;
            o17.setBackground(Color.black) ;
            o17.setForeground(Color.yellow) ;
            o18 = new TextField() ;
            o18.setBackground(Color.black) ;
            o18.setForeground(Color.yellow) ;

            add(lb) ;
            add(new Label("Upstream ", Label.CENTER)) ;
            add(new Label("Downstream ", Label.CENTER)) ;
            add(new Label("Ratio ", Label.CENTER)) ;
            add(new Label(" ", Label.CENTER)) ;

            add(l1) ;
            add(o1) ;
            add(o3) ;
            add(o2) ;
            add(new Label(" ", Label.CENTER)) ;

            add(l2) ;
            add(o4) ;
            add(o6) ;
            add(o5) ;
            add(new Label(" ", Label.CENTER)) ;

            add(l3) ;
            add(o7) ;
            add(o9) ;
            add(o8) ;
            add(new Label(" ", Label.CENTER)) ;

            add(l6) ;
            add(o16) ;
            add(o18) ;
            add(o17) ;
            add(new Label(" ", Label.CENTER)) ;

            add(l4) ;
            add(o10) ;
            add(o12) ;
            add(o11) ;
            add(new Label(" ", Label.CENTER)) ;

            add(l5) ;
            add(o13) ;
            add(o15) ;
            add(o14) ;
            add(new Label(" ", Label.CENTER)) ;

        }
     }

     class Dntp extends Panel 
     {
         Normal outerparent ;
         TextField o1,o2,o3,o4,o5,o6,o7,o8,o9,o10 ;
         TextField o11,o12,o13,o14,o15,o16,o17,o18 ;
         TextField or1,or2,or3,or4,or5,or6;
         Label l1,l2,l3,l4,l5,l6 ;
         Label lb ;

         Dntp (Normal target) 
         {
             outerparent = target ;
             setLayout(new GridLayout(7,5,0,0)) ;
  
             lb = new Label("Calorically Imperfect", Label.LEFT) ;
             lb.setForeground(Color.blue) ;

             l1 = new Label("Static Pressure-psf", Label.CENTER) ;
             l2 = new Label("Total Pressure-psf", Label.CENTER) ;
             l3 = new Label("Static Temperature-R", Label.CENTER) ;
             l4 = new Label("Density-slug/cu ft", Label.CENTER) ;
             l5 = new Label("Mach", Label.CENTER) ;
             l6 = new Label("Total Temperature-R", Label.CENTER) ;
   
             o1 = new TextField() ;
             o1.setBackground(Color.black) ;
             o1.setForeground(Color.yellow) ;
             o2 = new TextField() ;
             o2.setBackground(Color.black) ;
             o2.setForeground(Color.yellow) ;
             o3 = new TextField() ;
             o3.setBackground(Color.black) ;
             o3.setForeground(Color.yellow) ;
             o4 = new TextField() ;
             o4.setBackground(Color.black) ;
             o4.setForeground(Color.yellow) ;
             o5 = new TextField() ;
             o5.setBackground(Color.black) ;
             o5.setForeground(Color.yellow) ;
             o6 = new TextField() ;
             o6.setBackground(Color.black) ;
             o6.setForeground(Color.yellow) ;
             o7 = new TextField() ;
             o7.setBackground(Color.black) ;
             o7.setForeground(Color.yellow) ;
             o8 = new TextField() ;
             o8.setBackground(Color.black) ;
             o8.setForeground(Color.yellow) ;
             o9 = new TextField() ;
             o9.setBackground(Color.black) ;
             o9.setForeground(Color.yellow) ;
             o10 = new TextField() ;
             o10.setBackground(Color.black) ;
             o10.setForeground(Color.yellow) ;
             o11 = new TextField() ;
             o11.setBackground(Color.black) ;
             o11.setForeground(Color.yellow) ;
             o12 = new TextField() ;
             o12.setBackground(Color.black) ;
             o12.setForeground(Color.yellow) ;
             o13 = new TextField() ;
             o13.setBackground(Color.black) ;
             o13.setForeground(Color.yellow) ;
             o14 = new TextField() ;
             o14.setBackground(Color.black) ;
             o14.setForeground(Color.yellow) ;
             o15 = new TextField() ;
             o15.setBackground(Color.black) ;
             o15.setForeground(Color.yellow) ;
             o16 = new TextField() ;
             o16.setBackground(Color.black) ;
             o16.setForeground(Color.yellow) ;
             o17 = new TextField() ;
             o17.setBackground(Color.black) ;
             o17.setForeground(Color.yellow) ;
             o18 = new TextField() ;
             o18.setBackground(Color.black) ;
             o18.setForeground(Color.yellow) ;
           
             or1 = new TextField() ;
             or1.setBackground(Color.black) ;
             or1.setForeground(Color.cyan) ;
             or2 = new TextField() ;
             or2.setBackground(Color.black) ;
             or2.setForeground(Color.cyan) ;
             or3 = new TextField() ;
             or3.setBackground(Color.black) ;
             or3.setForeground(Color.cyan) ;
             or4 = new TextField() ;
             or4.setBackground(Color.black) ;
             or4.setForeground(Color.cyan) ;
             or5 = new TextField() ;
             or5.setBackground(Color.black) ;
             or5.setForeground(Color.cyan) ;
             or6 = new TextField() ;
             or6.setBackground(Color.black) ;
             or6.setForeground(Color.cyan) ;
           
             add(lb) ;
             add(new Label("Upstream ", Label.CENTER)) ;
             add(new Label("Downstream ", Label.CENTER)) ;
             add(new Label("Ratio ", Label.CENTER)) ;
             add(new Label("Ratio to Perfect", Label.CENTER)) ;

             add(l1) ;
             add(o1) ;
             add(o3) ;
             add(o2) ;
             add(or1);

             add(l2) ;
             add(o4) ;
             add(o6) ;
             add(o5) ;
             add(or2);

             add(l3) ;
             add(o7) ;
             add(o9) ;
             add(o8) ;
             add(or3);

             add(l6) ;
             add(o16) ;
             add(o18) ;
             add(o17) ;
             add(or4);

             add(l4) ;
             add(o10) ;
             add(o12) ;
             add(o11) ;
             add(or5);

             add(l5) ;
             add(o13) ;
             add(o15) ;
             add(o14) ;
             add(or6);

       }
     }
  }
}
